/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.module.soapkit.metadata;

import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Named;
import org.junit.Assert;
import org.junit.Rule;
import org.junit.runners.Parameterized;
import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.metadata.MetadataKey;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.MetadataService;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataDescriptor;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.test.runner.ArtifactClassLoaderRunnerConfig;
import org.mule.test.runner.RunnerDelegateTo;

import static java.util.stream.Collectors.joining;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.core.Is.is;
import static org.mule.module.soapkit.metadata.AbstractMetadataTestCase.ResolutionType.DSL_RESOLUTION;
import static org.mule.module.soapkit.metadata.AbstractMetadataTestCase.ResolutionType.EXPLICIT_RESOLUTION;
import static org.mule.runtime.api.metadata.MetadataService.METADATA_SERVICE_KEY;
import static org.mule.runtime.api.util.Preconditions.checkArgument;

// Class inspired in Mule class AbstractMetadataOperationTestCase and MetadataExtensionFunctionalTestCase

@RunnerDelegateTo(Parameterized.class)
@ArtifactClassLoaderRunnerConfig(testInclusions = {"org.mule.modules:mule-soapkit-common-test"})
public abstract class AbstractMetadataTestCase extends MuleArtifactFunctionalTestCase {

  private final ResolutionType resolutionType;
  private MetadataComponentDescriptorProvider<OperationModel> provider;

  @Inject
  @Named(METADATA_SERVICE_KEY)
  protected MetadataService metadataService;

  @Rule
  public DynamicPort httpPort = new DynamicPort("httpPort");

  AbstractMetadataTestCase(final ResolutionType resolutionType) {

    this.resolutionType = resolutionType;
    this.provider = resolutionType == ResolutionType.EXPLICIT_RESOLUTION ? MetadataService::getOperationMetadata
        : (metadataService, componentId, key) -> metadataService.getOperationMetadata(componentId);
  }

  protected Optional<ExtensionModel> getExtensionModel(String name) {
    return muleContext.getExtensionManager().getExtension(name);
  }

  @Override
  protected boolean isDisposeContextPerClass() {
    return true;
  }

  public enum ResolutionType {
    EXPLICIT_RESOLUTION, DSL_RESOLUTION
  }

  @Parameterized.Parameters(name = "{0}")
  public static Collection<Object[]> data() {
    return Arrays.asList(new Object[][] {
        {EXPLICIT_RESOLUTION},
        {DSL_RESOLUTION}
    });
  }

  protected void assertSuccessMetadataResult(final MetadataResult<?> result) {
    assertThat(result.getFailures(), is(empty()));
    final String failures = result.getFailures().stream().map(Object::toString).collect(joining(", "));
    assertThat("Expecting success but this failure/s result/s found:\n " + failures, result.isSuccess(), is(true));
  }

  protected Set<MetadataKey> getKeySetFromContainer(final MetadataKeysContainer metadataKeysContainer) {
    return metadataKeysContainer.getKeys(metadataKeysContainer.getCategories().iterator().next()).get();
  }

  protected Map<String, Set<MetadataKey>> getKeyMapFromContainer(final MetadataKeysContainer metadataKeysContainer) {
    return metadataKeysContainer
        .getCategories()
        .stream()
        .collect(Collectors.toMap(resolver -> resolver, resolver -> metadataKeysContainer.getKeys(resolver).get()));
  }

  MetadataResult<ComponentMetadataDescriptor<OperationModel>> getComponentDynamicMetadata(final Location location,
                                                                                          final MetadataKey key) {
    checkArgument(location != null, "Unable to resolve Metadata. The location has not been configured.");
    return provider.resolveDynamicMetadata(metadataService, location, key);
  }

  ComponentMetadataDescriptor<OperationModel> getSuccessComponentDynamicMetadataWithKey(final Location location,
                                                                                        MetadataKey key) {
    return getSuccessComponentDynamicMetadata(location, key, this::assertResolvedKey);
  }

  ComponentMetadataDescriptor<OperationModel> getSuccessComponentDynamicMetadata(final Location location, MetadataKey key) {
    return getSuccessComponentDynamicMetadata(location, key, (a, b) -> {
    });
  }

  private ComponentMetadataDescriptor<OperationModel> getSuccessComponentDynamicMetadata(Location location, MetadataKey key,
                                                                                         BiConsumer<MetadataResult<ComponentMetadataDescriptor<OperationModel>>, MetadataKey> assertKeys) {
    MetadataResult<ComponentMetadataDescriptor<OperationModel>> componentMetadata = getComponentDynamicMetadata(location, key);
    String msg = componentMetadata.getFailures().stream().map(f -> "Failure: " + f.getMessage()).collect(joining(", "));
    Assert.assertThat(msg, componentMetadata.isSuccess(), is(true));
    assertKeys.accept(componentMetadata, key);
    return componentMetadata.get();
  }

  private <T extends ComponentModel> void assertResolvedKey(MetadataResult<ComponentMetadataDescriptor<OperationModel>> result,
                                                            MetadataKey metadataKey) {
    Assert.assertThat(result.get().getMetadataAttributes().getKey().isPresent(), is(true));
    MetadataKey resultKey = result.get().getMetadataAttributes().getKey().get();
    assertSameKey(metadataKey, resultKey);

    MetadataKey child = metadataKey.getChilds().stream().findFirst().orElseGet(() -> null);
    MetadataKey otherChild = resultKey.getChilds().stream().findFirst().orElseGet(() -> null);
    while (child != null && otherChild != null) {
      assertSameKey(child, otherChild);
      child = child.getChilds().stream().findFirst().orElseGet(() -> null);
      otherChild = otherChild.getChilds().stream().findFirst().orElseGet(() -> null);
    }
    Assert.assertThat(child == null && otherChild == null, is(true));
  }

  private void assertSameKey(MetadataKey metadataKey, MetadataKey resultKey) {
    Assert.assertThat(resultKey.getId(), is(metadataKey.getId()));
    Assert.assertThat(resultKey.getChilds(), hasSize(metadataKey.getChilds().size()));
  }

  interface MetadataComponentDescriptorProvider<T extends ComponentModel> {

    MetadataResult<ComponentMetadataDescriptor<T>> resolveDynamicMetadata(MetadataService metadataService,
                                                                          Location location, MetadataKey key);
  }
}
