/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.module.soapkit;

import java.util.List;
import javax.xml.namespace.QName;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.xml.api.XmlTypeLoader;
import org.mule.wsdl.parser.WsdlParser;
import org.mule.wsdl.parser.exception.WsdlParsingException;
import org.mule.wsdl.parser.model.FaultModel;
import org.mule.wsdl.parser.model.PortModel;
import org.mule.wsdl.parser.model.ServiceModel;
import org.mule.wsdl.parser.model.WsdlModel;
import org.mule.wsdl.parser.model.operation.OperationModel;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.isIn;
import static org.hamcrest.Matchers.notNullValue;

public class WsdlParserTestCase {

  private static final String TSHIRT_FILE_LOCATION = "src/test/resources/api/tshirt.wsdl";
  private static final String TSHIRT_SERVICE_NAME = "TshirtService";
  private static final String TSHIRT_PORT_NAME = "TshirtServicePort";
  private static final String[] TSHIRT_OPERATIONS = {"OrderTshirt", "ListInventory", "TrackOrder"};
  private static final String TSHIRT_FAULT_NAME = "TshirtFault";
  private static final String TSHIRT_NAMESPACEURI = "http://mulesoft.org/tshirt-service";

  @Rule
  public ExpectedException expectedException = ExpectedException.none();

  @Test
  public void whenWsdlExistsDefinitionIsObtained() {

    assertThat(getWsdlModel(), notNullValue());
  }

  @Test
  public void whenWsdlIsProvidedServicesAreObtained() {
    final List<ServiceModel> services = getWsdlModel().getServices();
    assertThat(services.size(), is(1));
    assertThat(services.get(0).getName(), is(TSHIRT_SERVICE_NAME));
  }

  @Test
  public void whenWsdlIsProvidedPortsAreObtained() {
    final List<PortModel> ports = getServiceModel().getPorts();
    assertThat(ports.size(), is(1));
    assertThat(ports.get(0).getName(), is(TSHIRT_PORT_NAME));
  }

  @Test
  public void whenWsdlIsProvidedOperationsAreObtained() {
    final List<OperationModel> operations = getPortModel().getOperations();
    for (OperationModel operation : operations) {
      assertThat(operation.getName(), isIn(TSHIRT_OPERATIONS));
    }
  }

  @Test
  public void whenServiceNameIsProvidedServiceIsObtained() {
    final ServiceModel service = getServiceModel();
    assertThat(service, notNullValue());
    assertThat(service.getQName(), is(new QName(TSHIRT_NAMESPACEURI, TSHIRT_SERVICE_NAME)));
  }

  @Test
  public void whenPortNameIsProvidedPortIsObtained() {
    final PortModel port = getPortModel();
    assertThat(port, notNullValue());
    assertThat(port.getName(), is(TSHIRT_PORT_NAME));
  }

  @Test
  public void whenOperationNameIsProvidedOperationIsObtained() {
    final PortModel port = getPortModel();
    for (final String operationName : TSHIRT_OPERATIONS) {
      OperationModel operation = port.getOperation(operationName);
      assertThat(operation, notNullValue());
      assertThat(operation.getName(), is(operationName));
    }
  }

  @Test
  public void whenFaultNameIsProvidedFaultIsObtained() {
    final PortModel port = getPortModel();
    for (final String operationName : TSHIRT_OPERATIONS) {
      final FaultModel fault = port.getOperation(operationName).getFault(TSHIRT_FAULT_NAME);
      assertThat(fault, notNullValue());
      assertThat(fault.getName(), is(TSHIRT_FAULT_NAME));
    }
  }

  @Test
  public void whenFaultIsProvidedMetadataTypeIsObtained() {
    final XmlTypeLoader typeLoader = getWsdlModel().getLoader().getValue();
    final PortModel port = getPortModel();
    for (final String operationName : TSHIRT_OPERATIONS) {
      final FaultModel fault = port.getOperation(operationName).getFault(TSHIRT_FAULT_NAME);
      assertThat(fault, notNullValue());
      assertThat(fault.getName(), is(TSHIRT_FAULT_NAME));

      final MetadataType metadataType = typeLoader.load(fault.getMessage().getQName().toString()).get();
      assertThat(metadataType, notNullValue());
      assertThat(metadataType.toString(), is("{http://mulesoft.org/tshirt-service}TshirtFault"));
    }
  }

  /*
   * FAILING SCENARIOS BELOW
   */

  @Test
  public void whenWsdlDoesntExistsExceptionIsThrown() {
    expectedException.expect(WsdlParsingException.class);
    expectedException
        .expectMessage("Error processing WSDL file [src/test/resources/non-existing-file.wsdl]: Unable to locate document at 'src/test/resources/non-existing-file.wsdl'.");
    WsdlParser.Companion.parse("src/test/resources/non-existing-file.wsdl");
  }

  @Test
  public void whenWsdlIsCorruptedExceptionIsThrown() {
    expectedException.expect(WsdlParsingException.class);
    expectedException
        .expectMessage("Error processing WSDL file [src/test/resources/api/corrupted.wsdl]: faultCode=PARSER_ERROR: Problem parsing 'src/test/resources/api/corrupted.wsdl'.: org.xml.sax.SAXParseException: Content is not allowed in prolog.");
    WsdlParser.Companion.parse("src/test/resources/api/corrupted.wsdl");
  }

  /*
   * HELPER METHODS
   */

  private WsdlModel getWsdlModel() {
    final WsdlModel wsdlModel = WsdlParser.Companion.parse(TSHIRT_FILE_LOCATION);
    assertThat(wsdlModel, notNullValue());
    return wsdlModel;
  }

  private ServiceModel getServiceModel() {
    final ServiceModel service = getWsdlModel().getService(TSHIRT_SERVICE_NAME);
    assertThat(service, notNullValue());
    return service;
  }

  private PortModel getPortModel() {
    final PortModel port = getServiceModel().getPort(TSHIRT_PORT_NAME);
    assertThat(port, notNullValue());
    return port;
  }
}


