/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.module.soapkit;

import java.util.List;
import javax.xml.namespace.QName;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.mule.wsdl.parser.WsdlParser;
import org.mule.wsdl.parser.exception.WsdlParsingException;
import org.mule.wsdl.parser.model.FaultModel;
import org.mule.wsdl.parser.model.PortModel;
import org.mule.wsdl.parser.model.ServiceModel;
import org.mule.wsdl.parser.model.WsdlModel;
import org.mule.wsdl.parser.model.operation.OperationModel;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.isIn;
import static org.hamcrest.Matchers.notNullValue;

public class WsdlGlobalWeatherParserTestCase {

  private static final String WSDL_LOCATION = "src/test/resources/api/globalWeather.wsdl";
  private static final String SERVICE_NAME = "GlobalWeather";
  private static final String PORT_NAME = "GlobalWeatherHttpPost";
  private static final String[] OPERATIONS = {"GetWeather", "GetCitiesByCountry"};
  private static final String NAMESPACE_URI = "http://www.webserviceX.NET";

  @Rule
  public ExpectedException expectedException = ExpectedException.none();

  @Test
  public void whenWsdlExistsDefinitionIsObtained() {

    assertThat(getWsdlModel(), notNullValue());
  }

  @Test
  public void whenWsdlIsProvidedServicesAreObtained() {
    final List<ServiceModel> services = getWsdlModel().getServices();
    assertThat(services.size(), is(1));
    assertThat(services.get(0).getName(), is(SERVICE_NAME));
  }

  @Test
  public void whenWsdlIsProvidedPortsAreObtained() {
    final List<PortModel> ports = getServiceModel().getPorts();
    assertThat(ports.size(), is(4));
  }

  @Test
  public void whenWsdlIsProvidedOperationsAreObtained() {
    final List<OperationModel> operations = getPortModel().getOperations();
    for (OperationModel operation : operations) {
      assertThat(operation.getName(), isIn(OPERATIONS));
    }
  }

  @Test
  public void whenServiceNameIsProvidedServiceIsObtained() {
    final ServiceModel service = getServiceModel();
    assertThat(service, notNullValue());
    assertThat(service.getQName(), is(new QName(NAMESPACE_URI, SERVICE_NAME)));
  }

  @Test
  public void whenPortNameIsProvidedPortIsObtained() {
    final PortModel port = getPortModel();
    assertThat(port, notNullValue());
    assertThat(port.getName(), is(PORT_NAME));
  }

  @Test
  public void whenOperationNameIsProvidedOperationIsObtained() {
    final PortModel port = getPortModel();
    for (final String operationName : OPERATIONS) {
      OperationModel operation = port.getOperation(operationName);
      assertThat(operation, notNullValue());
      assertThat(operation.getName(), is(operationName));
    }
  }

  @Test
  public void whenFaultNameIsProvidedFaultIsObtained() {
    final PortModel port = getPortModel();
    for (final String operationName : OPERATIONS) {
      final List<FaultModel> fault = port.getOperation(operationName).getFaults();
      assertThat(fault, notNullValue());
      assertThat(fault.size(), is(0));
    }
  }

  /*
   * FAILING SCENARIOS BELOW
   */

  @Test
  public void whenWsdlDoesntExistsExceptionIsThrown() {
    expectedException.expect(WsdlParsingException.class);
    expectedException
        .expectMessage("Error processing WSDL file [src/test/resources/non-existing-file.wsdl]: Unable to locate document at 'src/test/resources/non-existing-file.wsdl'.");
    WsdlParser.Companion.parse("src/test/resources/non-existing-file.wsdl");
  }

  @Test
  public void whenWsdlIsCorruptedExceptionIsThrown() {
    expectedException.expect(WsdlParsingException.class);
    expectedException
        .expectMessage("Error processing WSDL file [src/test/resources/api/corrupted.wsdl]: faultCode=PARSER_ERROR: Problem parsing 'src/test/resources/api/corrupted.wsdl'.: org.xml.sax.SAXParseException: Content is not allowed in prolog.");
    WsdlParser.Companion.parse("src/test/resources/api/corrupted.wsdl");
  }

  /*
   * HELPER METHODS
   */

  private WsdlModel getWsdlModel() {
    final WsdlModel wsdlModel = WsdlParser.Companion.parse(WSDL_LOCATION);
    assertThat(wsdlModel, notNullValue());
    return wsdlModel;
  }

  private ServiceModel getServiceModel() {
    final ServiceModel service = getWsdlModel().getService(SERVICE_NAME);
    assertThat(service, notNullValue());
    return service;
  }

  private PortModel getPortModel() {
    final PortModel port = getServiceModel().getPort(PORT_NAME);
    assertThat(port, notNullValue());
    return port;
  }
}

