/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.module.soapkit.internal.util;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;
import org.apache.log4j.Logger;
import org.mule.runtime.api.component.Component;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.extension.api.runtime.config.ConfigurationState;
import org.mule.runtime.extension.api.runtime.config.ConfiguredComponent;
import org.mule.runtime.extension.api.runtime.source.ParameterizedSource;

import static org.apache.commons.lang3.StringUtils.prependIfMissing;
import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;
import static org.mule.soapkit.soap.util.Cast.cast;

public class HttpListenerUtils {

  private static final Logger LOGGER = Logger.getLogger(HttpListenerUtils.class);
  private static final String PATH = "path";
  private static final String BASE_PATH = "basePath";
  private static final String HOST = "host";
  private static final String PORT = "port";
  private static final String PROTOCOL = "protocol";
  private static final String HTTP_LISTENER = "http:listener";

  private HttpListenerUtils() {}

  /**
   * Extracts the configured HTTP URI from a flow. It only works for flows that uses the HTTP extension.
   *
   * @param source where to extract the URI
   * @return the URI
   */

  public static URI getUriFromFlow(final Component source) {
    if (source != null && isHttpExtensionSource(source)) {
      try {
        String resolvedPath = getListenerPath(source);
        return buildListenerUri(getConfigState(source).getConnectionParameters(), resolvedPath);
      } catch (Exception e) {
        LOGGER.warn("Error getting uri from flow " + source.getLocation().getRootContainerName(), e);
      }
    }

    return null;
  }

  private static String getListenerPath(final Component source) {
    final ParameterizedSource parameterizedSource = cast(source);
    String listenerPath = cast(parameterizedSource.getInitialisationParameters().get(PATH));
    final String basePath = cast(getConfigState(source).getConfigParameters().get(BASE_PATH));
    listenerPath = prependIfMissing(listenerPath, "/");
    return basePath == null ? listenerPath : prependIfMissing(basePath, "/") + listenerPath;
  }

  private static ConfigurationState getConfigState(final Component source) {
    final ConfiguredComponent configuredComponent = cast(source);
    return configuredComponent.getConfigurationInstance()
        .orElseThrow(() -> new RuntimeException("Source does not contain a configuration instance"))
        .getState();
  }

  private static boolean isHttpExtensionSource(final Component source) {
    final ComponentIdentifier identifier = source.getLocation().getComponentIdentifier().getIdentifier();
    return identifier.equals(buildFromStringRepresentation(HTTP_LISTENER));
  }

  private static URI buildListenerUri(final Map<String, Object> connectionParams, final String path)
      throws URISyntaxException {
    final String host = cast(connectionParams.get(HOST));
    final Integer port = cast(connectionParams.get(PORT));
    final String scheme = connectionParams.get(PROTOCOL).toString().toLowerCase();
    return new URI(scheme, null, host, port, path, null, null);
  }

}
