/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.module.soapkit.internal.metadata;

import java.net.URL;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import javax.xml.namespace.QName;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.xml.api.SchemaCollector;
import org.mule.metadata.xml.api.XmlTypeLoader;
import org.mule.module.soapkit.internal.OperationFaultPair;
import org.mule.module.soapkit.internal.SoapkitConfiguration;
import org.mule.module.soapkit.internal.WsdlConnectionInfo;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.metadata.MetadataContext;
import org.mule.runtime.api.metadata.MetadataKey;
import org.mule.runtime.api.metadata.MetadataKeyBuilder;
import org.mule.runtime.api.metadata.MetadataResolvingException;
import org.mule.runtime.api.metadata.resolving.InputTypeResolver;
import org.mule.runtime.api.metadata.resolving.OutputTypeResolver;
import org.mule.runtime.api.metadata.resolving.TypeKeysResolver;
import org.mule.runtime.soap.api.SoapVersion;
import org.mule.wsdl.parser.model.FaultModel;
import org.mule.wsdl.parser.model.PortModel;
import org.mule.wsdl.parser.model.operation.OperationModel;

import static java.lang.String.format;
import static org.mule.runtime.api.metadata.resolving.FailureCode.CONNECTION_FAILURE;
import static org.mule.runtime.api.metadata.resolving.FailureCode.INVALID_METADATA_KEY;
import static org.mule.runtime.api.metadata.resolving.FailureCode.RESOURCE_UNAVAILABLE;

public class OperationFaultPairResolver extends AbstractBaseResolver
    implements TypeKeysResolver, InputTypeResolver<OperationFaultPair>, OutputTypeResolver<OperationFaultPair> {

  public static final String GENERIC_SF = "GenericSF";
  private static final String GENERIC_SF_LABEL = "Generic Soap Fault";
  private static final String FAULT = "fault";

  @Override
  public String getResolverName() {
    return "OperationFaultPairResolver";
  }

  @Override
  public Set<MetadataKey> getKeys(MetadataContext metadataContext) throws MetadataResolvingException, ConnectionException {

    final SoapkitConfiguration config = metadataContext.<SoapkitConfiguration>getConfig()
        .orElseThrow(() -> new MetadataResolvingException("Could not obtain config to retrieve metadata",
                                                          CONNECTION_FAILURE));

    final WsdlConnectionInfo info = config.getInfo();
    final PortModel portModel = info.getPortModel();
    final SoapVersion soapVersion = config.getSoapVersion(portModel);

    final HashSet<MetadataKey> keys = new HashSet<>();

    portModel.getOperations().forEach(operation -> {
      final MetadataKeyBuilder builder = MetadataKeyBuilder.newKey(operation.getName());

      operation.getFaults().forEach(fault -> builder.withChild(MetadataKeyBuilder.newKey(fault.getName())));

      // Add generic Fault
      builder.withChild(MetadataKeyBuilder.newKey(GENERIC_SF).withDisplayName(GENERIC_SF_LABEL + " " + soapVersion));

      keys.add(builder.build());
    });

    return keys;
  }

  @Override
  public MetadataType getOutputType(final MetadataContext ctx, OperationFaultPair key)
      throws MetadataResolvingException, ConnectionException {
    return getMetadataType(ctx, key);
  }

  @Override
  public MetadataType getInputMetadata(final MetadataContext ctx, OperationFaultPair key)
      throws MetadataResolvingException, ConnectionException {
    return getMetadataType(ctx, key);
  }

  private MetadataType getMetadataType(final MetadataContext ctx, OperationFaultPair key)
      throws MetadataResolvingException, ConnectionException {

    final SoapkitConfiguration config = ctx.<SoapkitConfiguration>getConfig()
        .orElseThrow(() -> new MetadataResolvingException("Could not obtain config to retrieve metadata",
                                                          CONNECTION_FAILURE));

    final WsdlConnectionInfo info = config.getInfo();
    final PortModel portModel = info.getPortModel();
    final SoapVersion soapVersion = config.getSoapVersion(portModel);


    final String fault = key.getFault();

    final Optional<MetadataType> metadataType;

    if (OperationFaultPairResolver.GENERIC_SF.equals(fault)) {

      final String schema = SoapVersion.SOAP11.equals(soapVersion) ? SF_V11_XSD : SF_V12_XSD;
      final URL schemaUrl = getSchemaUrl(schema);
      if (schemaUrl == null)
        throw new MetadataResolvingException(format("Could not obtain schema '%s' for operation='%s', fault='%s'", schema,
                                                    key.getOperation(), key.getFault()),
                                             RESOURCE_UNAVAILABLE);

      final SchemaCollector collector = SchemaCollector.getInstance().addSchema(schemaUrl);
      final XmlTypeLoader typeLoader = new XmlTypeLoader(collector);
      metadataType = typeLoader.load(FAULT);
    } else {
      try {

        final OperationModel operationModel = portModel.getOperation(key.getOperation());
        final FaultModel faultModel = operationModel.getFault(fault);
        if (faultModel.getMessage() == null || faultModel.getMessage().getQName() == null)
          throw new MetadataResolvingException(format("Could not obtain fault Metadata, missing QName for operation='%s', fault='%s'",
                                                      key.getOperation(),
                                                      key.getFault()),
                                               INVALID_METADATA_KEY);

        final QName qName = faultModel.getMessage().getQName();
        metadataType = info.getXmlTypeLoader().load(qName.toString());
      } catch (final Exception e) {
        throw new MetadataResolvingException(format("Could not obtain fault Metadata for operation='%s', fault='%s'",
                                                    key.getOperation(),
                                                    key.getFault()),
                                             INVALID_METADATA_KEY, e);
      }
    }

    return metadataType
        .orElseThrow(() -> new MetadataResolvingException(format("Could not obtain fault Metadata for operation='%s', fault='%s'",
                                                                 key.getOperation(),
                                                                 key.getFault()),
                                                          INVALID_METADATA_KEY));
  }

}
