/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.module.soapkit.internal.exception;

import org.mule.runtime.api.component.execution.ComponentExecutionException;
import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.exception.ErrorMessageAwareException;
import org.mule.runtime.api.exception.TypedException;
import org.mule.runtime.api.message.Error;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.api.message.Message;

import java.util.Optional;

/**
 * This exception carries the exception raised inside an executing sub flow.
 *
 * There are 2 important things in this exception:
 *  -> It's a {@link TypedException} that propagates the ErrorType that comes in the Event that was being processed
 *  in the sub flow.
 *  -> The cause of this exception is a {@link SubFlowFailureErrorMessageAwareException} that is defined in this same class and
 *  implements the {@link ErrorMessageAwareException} interface, by creating this structure of {@link TypedException} that
 *  contains an {@link ErrorMessageAwareException} we are able being able to propagate the errorType and errorMessage to the
 *  the runtime so the main flow can handle this error with the same errorType raised by the subflow and access the payload and
 *  attributes of the sub flow when failed by using this expression: #[error.errorMessage]
 *
 *  Also note that all information provided by this exception is extracted from the {@link Event} of a
 *  {@link ComponentExecutionException} that is the exception raised by a flow when fails.
 */
public class SubFlowFailureException extends TypedException {

  private SubFlowFailureException(Exception cause, ErrorType errorType, String description) {
    super(cause, errorType, description);
  }

  public static SubFlowFailureException newInstance(ComponentExecutionException e) {
    Event event = e.getEvent();
    Optional<Error> error = event.getError();
    if (error.isPresent()) {
      String description = error.get().getDescription();
      Exception errorMessageAwareException = new SubFlowFailureErrorMessageAwareException(event.getMessage(), description);
      return new SubFlowFailureException(errorMessageAwareException, error.get().getErrorType(), description);
    }
    throw e;
  }

  private static class SubFlowFailureErrorMessageAwareException extends RuntimeException implements ErrorMessageAwareException {

    private final Message message;

    SubFlowFailureErrorMessageAwareException(Message message, String description) {
      super(description);
      this.message = message;
    }

    @Override
    public Message getErrorMessage() {
      return message;
    }
  }
}
