/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.module.soapkit.internal;

import java.net.MalformedURLException;
import java.net.URL;
import org.mule.metadata.xml.api.XmlTypeLoader;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.Path;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.mule.runtime.extension.api.annotation.values.ValuePart;
import org.mule.wsdl.parser.WsdlParser;
import org.mule.wsdl.parser.model.PortModel;
import org.mule.wsdl.parser.model.ServiceModel;
import org.mule.wsdl.parser.model.WsdlModel;

import static java.lang.String.format;
import static org.mule.runtime.api.meta.model.display.PathModel.Type.FILE;
import static org.mule.runtime.core.api.util.StringUtils.isBlank;

/**
 * Groups together the parameters retrieved from a WSDL that are going to be used for establishing a connection to a SOAP endpoint.
 *
 * @since 1.1
 */
public class WsdlConnectionInfo {


  private static final String ROOT_API = "api/";

  /**
  * The WSDL file URL remote or local.
  */
  @Placement(order = 1)
  @Parameter
  @Path(type = FILE, acceptedFileExtensions = "wsdl", acceptsUrls = true)
  private String wsdlLocation;

  /**
   * The service name.
   */
  @Placement(order = 2)
  @Parameter
  @ValuePart(order = 1)
  private String service;

  /**
   * The port name.
   */
  @Placement(order = 3)
  @Parameter
  @ValuePart(order = 2)
  private String port;

  private WsdlModel wsdlModel = null;
  private XmlTypeLoader typeLoader = null;

  public String getService() {
    return service;
  }

  public String getPort() {
    return port;
  }

  public String getWsdlLocation() {
    return wsdlLocation;
  }

  public void setWsdlLocation(String wsdlLocation) {
    this.wsdlLocation = wsdlLocation;
  }

  public void setService(String service) {
    this.service = service;
  }

  public void setPort(String port) {
    this.port = port;
  }

  public WsdlModel getWsdlModel() throws ConnectionException {

    // Invalidate if Location change
    if (wsdlModel != null && !wsdlModel.getLocation().equals(wsdlLocation))
      wsdlModel = null;

    if (wsdlModel == null) {
      wsdlModel = getWsdlModel(wsdlLocation);
      //typeLoader = wsdlModel.getLoader().getValue();
    }

    return wsdlModel;
  }

  public XmlTypeLoader getXmlTypeLoader() throws ConnectionException {

    if (typeLoader == null) {
      typeLoader = getWsdlModel().getLoader().getValue();
    }
    return typeLoader;
  }

  public URL getWsdlLocationUrl() throws ConnectionException {
    return getWsdlLocationUrl(wsdlLocation);
  }

  public ServiceModel getServiceModel() throws ConnectionException {

    final WsdlModel wsdlModel = getWsdlModel();
    if (isBlank(service))
      throw new ConnectionException("Parameter service is missing in config");

    final ServiceModel serviceModel = wsdlModel.getService(service);
    if (serviceModel == null)
      throw new ConnectionException(format("The service name '%s' was not found in the current wsdl file.",
                                           service));

    return serviceModel;
  }

  public PortModel getPortModel() throws ConnectionException {

    if (isBlank(port))
      throw new ConnectionException("Parameter port is missing in config");

    final PortModel portModel = getServiceModel().getPort(port);
    if (portModel == null)
      throw new ConnectionException(format("The port name '%s' was not found in the current wsdl file.",
                                           port));

    return portModel;
  }

  public static WsdlModel getWsdlModel(final String wsdlLocation) throws ConnectionException {
    if (isBlank(wsdlLocation))
      throw new ConnectionException("Parameter wsdlLocation is missing in config");

    final URL wsdlUrl = getWsdlLocationUrl(wsdlLocation);

    try {
      final String location = wsdlUrl.toString();
      return WsdlParser.Companion.parse(location);
    } catch (Exception e) {
      throw new ConnectionException(format("Error parsing WSDL file '%s'", wsdlUrl.toString()), e);
    }
  }

  public static URL getWsdlLocationUrl(final String wsdlLocation) throws ConnectionException {

    final java.util.Optional<URL> url = getUrl(wsdlLocation);
    if (url.isPresent())
      return url.get();

    // Back to normal, https://www.mulesoft.org/jira/browse/MULE-14694 was fixed
    final ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
    URL resource = classLoader.getResource(wsdlLocation);
    if (resource == null)
      resource = classLoader.getResource(ROOT_API + wsdlLocation);

    if (resource == null)
      throw new ConnectionException(format("Cannot create URL for wsdlLocation '%s' in config [%s]", wsdlLocation,
                                           classLoader.getClass().getName()));

    return resource;
  }

  private static java.util.Optional<URL> getUrl(final String wsdlLocation) {
    try {
      final URL url = new URL(wsdlLocation);
      return java.util.Optional.of(url);
    } catch (MalformedURLException e) {
      return java.util.Optional.empty();
    }
  }

  public PortModel validate() throws ConnectionException {
    getWsdlModel();
    getServiceModel();

    return getPortModel();
  }

}
