/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.module.soapkit.internal;

import com.google.common.collect.ImmutableMap;
import org.mule.module.soapkit.internal.exception.SoapkitRouterException;
import org.mule.module.soapkit.internal.exception.SubFlowFailureException;
import org.mule.module.soapkit.internal.exception.error.SoapkitExceptionEnricher;
import org.mule.runtime.api.component.execution.ComponentExecutionException;
import org.mule.runtime.api.component.location.ConfigurationComponentLocator;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.extension.api.runtime.config.ConfigurationInstance;
import org.mule.runtime.extension.api.runtime.operation.ComponentExecutor;
import org.mule.runtime.extension.api.runtime.operation.ExecutionContext;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.module.extension.api.runtime.privileged.ExecutionContextAdapter;
import org.mule.runtime.module.extension.api.runtime.privileged.StreamingHelperFactory;
import org.mule.runtime.soap.api.message.SoapRequest;
import org.mule.soapkit.soap.api.server.SoapServerHandler;
import org.mule.soapkit.soap.message.SoapAttributes;
import org.mule.soapkit.soap.message.SoapResponse;
import org.reactivestreams.Publisher;

import javax.inject.Inject;
import java.io.InputStream;
import java.util.Map;
import java.util.Optional;

import static org.apache.http.HttpHeaders.CONTENT_TYPE;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.core.api.rx.Exceptions.wrapFatal;
import static reactor.core.publisher.Mono.error;
import static reactor.core.publisher.Mono.justOrEmpty;

public class SoapkitRouterExecutor implements ComponentExecutor<OperationModel> {

  private static final String MESSAGE_PARAM = "message";
  private static final String OPERATION = "soapkit-operation";

  @Inject
  private ConfigurationComponentLocator componentLocator;

  private final SoapkitExceptionEnricher exceptionEnricher = new SoapkitExceptionEnricher();
  private final StreamingHelperFactory streamingHelperFactory = new StreamingHelperFactory();

  @Override
  public Publisher<Object> execute(final ExecutionContext<OperationModel> executionContext) {
    try {
      ExecutionContextAdapter<OperationModel> context = (ExecutionContextAdapter<OperationModel>) executionContext;
      CoreEvent event = context.getEvent();
      ConfigurationInstance configuration = getConfiguration(context);
      SoapkitConfiguration config = (SoapkitConfiguration) configuration.getValue();
      StreamingHelper streamingHelper = streamingHelperFactory.resolve(context);
      InputStream message = context.getParameter(MESSAGE_PARAM);
      SoapkitRouterAttributes attributes = SoapkitRouterAttributes.create(context, componentLocator, config.getAddress());

      SoapRequest request = SoapRequest.builder()
          .operation(OPERATION)
          .content(message)
          .transportHeaders(attributes.getTransportHeaders()).build();

      SoapServerHandler serverHandler = new DefaultSoapServerHandler(event, config, streamingHelper, componentLocator);
      SoapResponse response = config.getSoapServer().serve(request, serverHandler);

      Object messageValue = streamingHelper.resolveCursorProvider(response.getContent());
      return justOrEmpty(eventResult(event, response, messageValue));
    } catch (ComponentExecutionException e) {
      return error(SubFlowFailureException.newInstance(e));
    } catch (Exception e) {
      return error(exceptionEnricher.enrich(e));
    } catch (Throwable t) {
      return error(wrapFatal(t));
    }
  }

  private ConfigurationInstance getConfiguration(ExecutionContextAdapter<OperationModel> context) {
    Optional<ConfigurationInstance> configuration = context.getConfiguration();
    if (configuration.isPresent()) {
      return configuration.get();
    }
    throw new SoapkitRouterException(createStaticMessage("Could not found apikit-soap configuration to execute the request"));
  }

  private CoreEvent eventResult(CoreEvent parent, SoapResponse response, Object value) {
    Map<String, String> headers = ImmutableMap.<String, String>builder().putAll(response.getTransportHeaders())
        .put(CONTENT_TYPE, response.getContentType().toString()).build();

    Message message = Message.builder(parent.getMessage()).value(value)
        .attributesValue(new SoapAttributes(headers, response.getTransportAdditionalData())).build();

    return CoreEvent.builder(parent).variables(response.getVariables()).message(message).build();
  }
}
