/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.module.soapkit.internal;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import org.apache.commons.lang3.StringUtils;
import org.mule.module.soapkit.internal.exception.SoapkitRouterException;
import org.mule.module.soapkit.internal.util.HttpListenerUtils;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.component.location.ConfigurationComponentLocator;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.module.extension.api.runtime.privileged.ExecutionContextAdapter;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.soapkit.soap.SoapConstants.MULE_HTTP_ATTRIBUTES_LOCATION;
import static org.mule.soapkit.soap.SoapConstants.MULE_HTTP_ATTRIBUTES_METHOD;
import static org.mule.soapkit.soap.SoapConstants.MULE_HTTP_ATTRIBUTES_QUERY_STRING;
import static org.mule.soapkit.soap.SoapConstants.MULE_TRANSPORT_HEADERS_PREFIX;
import static org.mule.soapkit.soap.util.Cast.cast;

public class SoapkitRouterAttributes {

  private static final String HEADERS = "headers";
  private static final String METHOD = "method";
  private static final String QUERY_STRING = "queryString";

  private static final String ATTRIBUTES_PARAM = "attributes";

  private static final String BIND_TO_ALL_INTERFACES = "0.0.0.0";
  private static final String FULL_DOMAIN = "fullDomain";

  private Map<String, Object> attributes;
  private URI location;

  private static final String HTTP = "http://";

  private static final String HTTPS = "https://";

  private SoapkitRouterAttributes(final Map<String, Object> attributes, final URI location) {
    this.attributes = attributes;
    this.location = location;
    validate();
  }

  static SoapkitRouterAttributes create(final ExecutionContextAdapter<OperationModel> context,
                                        final ConfigurationComponentLocator componentLocator, String address) {

    final Map<String, Object> attributes = context.getParameter(ATTRIBUTES_PARAM);
    final ComponentLocation componentLocation = context.getComponent().getLocation();

    final String name = componentLocation.getRootContainerName();
    final URI location = componentLocator.find(Location.builder().globalName(name).addSourcePart().build())
        .map(HttpListenerUtils::getUriFromFlow)
        .orElseThrow(() -> new RuntimeException("Missing location form Http component name '" + name + "'"));

    return new SoapkitRouterAttributes(attributes, fixLocation(location, address));
  }

  private void validate() {
    final Object object = attributes.get(HEADERS);
    if (object == null)
      throw new SoapkitRouterException(createStaticMessage("Header attribute is missing"));
    if (!(object instanceof Map<?, ?>))
      throw new SoapkitRouterException(createStaticMessage("Header attribute is not a key -> value map"));
  }

  private Map<String, String> getHeaders() {
    return (Map<String, String>) cast(attributes.get(HEADERS));
  }

  public Map<String, String> getTransportHeaders() {

    final Map<String, String> transportHeaders = new HashMap<>();
    getHeaders().forEach((key, value) -> transportHeaders.put(MULE_TRANSPORT_HEADERS_PREFIX + key, value));

    getMethod().ifPresent(value -> transportHeaders.put(MULE_HTTP_ATTRIBUTES_METHOD, value));
    getQueryString().ifPresent(value -> transportHeaders.put(MULE_HTTP_ATTRIBUTES_QUERY_STRING, value));
    transportHeaders.put(MULE_HTTP_ATTRIBUTES_LOCATION, location.toString());

    return transportHeaders;
  }

  private Optional<String> getMethod() {
    return optionalAttribute(METHOD);
  }

  private Optional<String> getQueryString() {
    return optionalAttribute(QUERY_STRING);
  }

  private Optional<String> optionalAttribute(final String key) {
    final Object value = attributes.get(key);
    return (value instanceof String && StringUtils.isNotEmpty((String) value)) ? of((String) value) : empty();
  }

  private static URI fixLocation(final URI location, String address) {

    String baseUriReplacement = location.toString();
    if (!baseUriReplacement.contains(BIND_TO_ALL_INTERFACES))
      return location;

    String fullDomain = StringUtils.isNotEmpty(address) ? address : System.getProperty(FULL_DOMAIN);
    if (fullDomain != null) {

      String path = location.getPath();
      if (fullDomain.endsWith("/") && path.length() > 0 && path.startsWith("/")) {
        path = path.length() > 1 ? path.substring(1) : "";
      } else if (!fullDomain.endsWith("/") && path.length() > 0 && !path.startsWith("/")) {
        fullDomain += "/";
      }
      if (fullDomain.contains("://")) {
        baseUriReplacement = fullDomain + path;
      } else {
        String protocol = location.toString().contains(HTTPS) ? HTTPS : HTTP;
        baseUriReplacement = protocol + fullDomain + path;
      }
    } else { // 0.0.0.0
      baseUriReplacement = baseUriReplacement.replace(BIND_TO_ALL_INTERFACES, "localhost");
    }

    try {
      return new URI(baseUriReplacement);
    } catch (final Exception e) {
      return location;
    }
  }
}
