/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.module.soapkit.internal;

import java.net.URL;
import java.util.Map;
import org.mule.module.soapkit.internal.values.WsdlValueProvider;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.Startable;
import org.mule.runtime.api.lifecycle.Stoppable;
import org.mule.runtime.extension.api.annotation.Configuration;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.Operations;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.ParameterGroup;
import org.mule.runtime.extension.api.annotation.param.RefName;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.mule.runtime.extension.api.annotation.values.OfValues;
import org.mule.runtime.soap.api.SoapVersion;
import org.mule.soapkit.soap.SoapServiceImplementation;
import org.mule.soapkit.soap.api.server.SoapServer;
import org.mule.soapkit.soap.api.server.SoapServerConfiguration;
import org.mule.wsdl.parser.model.PortModel;
import org.mule.wsdl.parser.model.SoapBinding;
import org.mule.wsdl.parser.model.Version;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static java.lang.String.format;
import static org.mule.module.soapkit.internal.SoapkitConfiguration.CONFIG_NAME;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;

/**
 * Configuration element for a SOAP Router.
 *
 * @since 1.1
 */
@Configuration(name = CONFIG_NAME)
@Operations(SoapkitOperations.class)
@DisplayName("Configuration")
public class SoapkitConfiguration implements Startable, Stoppable {

  public static final String CONFIG_NAME = "config";

  @RefName
  private String name;

  @ParameterGroup(name = "Connection")
  @OfValues(WsdlValueProvider.class)
  private WsdlConnectionInfo info;

  /**
   * The soap version of the WSDL.
   */
  @Parameter
  @Placement(order = 5)
  @Optional
  @Expression(NOT_SUPPORTED)
  private SoapVersion soapVersion;

  /**
   * The address of the web service.
   */
  @Parameter
  @Optional
  @Placement(order = 6)
  private String address;

  /**
   * If should use the MTOM protocol to manage the attachments or not.
   */
  @Parameter
  @Placement(order = 7)
  @Optional(defaultValue = "false")
  @Expression(NOT_SUPPORTED)
  private boolean mtomEnabled;

  /**
   * If should use the MTOM protocol to manage the attachments or not.
   */
  @Parameter
  @Placement(order = 8)
  @Optional(defaultValue = "false")
  @Expression(NOT_SUPPORTED)
  private boolean inboundValidationEnabled;

  /**
   * The variable name for the http status.
   */
  @Parameter
  @Placement(order = 9)
  @Optional(defaultValue = "httpStatus")
  @Expression(NOT_SUPPORTED)
  private String httpStatusVarName;

  @Parameter
  @Placement(order = 10)
  @Optional
  @Expression(NOT_SUPPORTED)
  private Map<String, String> namespacePrefixes;

  private SoapServer soapServer = null;

  private final static Logger LOGGER = LoggerFactory.getLogger(SoapkitConfiguration.class);

  public String getName() {
    return name;
  }

  public WsdlConnectionInfo getInfo() {
    return info;
  }

  public SoapVersion getSoapVersion() {
    return soapVersion;
  }

  public boolean isMtomEnabled() {
    return mtomEnabled;
  }

  public boolean isInboundValidationEnabled() {
    return inboundValidationEnabled;
  }

  public void setInfo(WsdlConnectionInfo info) {
    this.info = info;
  }

  SoapServer getSoapServer() {
    return soapServer;
  }

  @Override
  public void start() throws MuleException {
    try {
      final PortModel portModel = validate();
      soapServer = new SoapServiceImplementation().getServerFactory().create(getConfiguration());
      soapServer.start();
      // Update SoapVersion from Port Model
      soapVersion = getSoapVersion(portModel);
    } catch (final ConnectionException e) {
      throw e;
    } catch (Exception e) {
      throw new ConnectionException(format("Error connecting Soap Server for config '%s'", name), e);
    }
  }

  @Override
  public void stop() throws MuleException {
    try {
      if (soapServer != null)
        soapServer.stop();
    } catch (Exception e) {
      LOGGER.error("Error disconnecting soap soapServer [" + soapServer.toString() + "]: " + e.getMessage(), e);
    }
  }

  private SoapServerConfiguration getConfiguration() throws ConnectionException {

    final URL wsdlUrl = info.getWsdlLocationUrl();

    return SoapServerConfiguration.builder()
        .withWsdlLocation(wsdlUrl)
        .withService(info.getService())
        .withPort(info.getPort())
        .withNamespaces(namespacePrefixes)
        //.withVersion(soapVersion) // Since default version can be wrong I prefer not set it and CXF do the job
        .enableMtom(mtomEnabled)
        .enableValidation(inboundValidationEnabled)
        .build();
  }

  private PortModel validate() throws ConnectionException {
    return getInfo().validate();
  }

  public SoapVersion getSoapVersion(final PortModel portModel) {
    final SoapBinding binding = portModel.getBinding();
    if (binding == null)
      return SoapVersion.SOAP11;

    final Version version = binding.getVersion();
    return version.equals(Version.V1_2) ? SoapVersion.SOAP12 : SoapVersion.SOAP11;
  }

  public String getAddress() {
    return address;
  }

  public String getHttpStatusVarName() {
    return httpStatusVarName;
  }
}
