/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.module.soapkit.internal;

import com.google.common.collect.ImmutableMap;

import java.util.concurrent.ExecutionException;

import org.mule.module.soapkit.internal.exception.SoapkitRouterException;
import org.mule.runtime.api.component.Component;
import org.mule.runtime.api.component.location.ConfigurationComponentLocator;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.api.construct.Flow;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.extension.api.soap.SoapAttachment;
import org.mule.runtime.soap.api.message.SoapRequest;
import org.mule.soapkit.soap.api.server.SoapServerHandler;
import org.mule.soapkit.soap.message.EmptySoapResponse;
import org.mule.soapkit.soap.message.ImmutableSoapResponse;
import org.mule.soapkit.soap.message.SoapResponse;
import org.mule.wsdl.parser.model.operation.OperationType;

import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import static java.lang.String.format;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JAVA;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_XML;
import static org.mule.soapkit.soap.server.SoapCxfServer.HTTP_STATUS_PROTOCOL_HEADER;
import static org.mule.soapkit.soap.server.SoapCxfServer.STATUS_CODE;
import static org.mule.soapkit.soap.util.Cast.cast;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.ACCEPTED;

public class DefaultSoapServerHandler implements SoapServerHandler {

  private final CoreEvent event;
  private final SoapkitConfiguration config;
  private final StreamingHelper streamingHelper;
  private final ConfigurationComponentLocator componentLocator;

  private static final String ERROR_MISSING_FLOW =
      "Despite the given WSDL file configurated in configuration '%s' does contains the name '%s', there is no current flow that maps to it (the expected flow should be called '%s')";

  DefaultSoapServerHandler(CoreEvent event,
                           SoapkitConfiguration config,
                           StreamingHelper streamingHelper,
                           ConfigurationComponentLocator componentLocator) {
    this.event = event;
    this.config = config;
    this.streamingHelper = streamingHelper;
    this.componentLocator = componentLocator;
  }

  @Override
  public SoapResponse handle(final SoapRequest request) throws ExecutionException {
    OperationType type = getType(request);
    try {
      Flow flow = findFlow(config, request.getOperation());
      Message message = buildMessage(request);
      CoreEvent requestEvent = CoreEvent.builder(event).message(message).build();
      Event responseEvent = flow.execute(requestEvent).get();
      return buildResponse(responseEvent, type);
    } catch (ExecutionException | SoapkitRouterException e) {
      throw e;
    } catch (Exception e) {
      throw new SoapkitRouterException(createStaticMessage("Error Routing Event with Soap Server"), e);
    }
  }

  private OperationType getType(SoapRequest request) {
    try {
      return config.getInfo().getPortModel().getOperation(request.getOperation()).getType();
    } catch (ConnectionException e) {
      throw new SoapkitRouterException(createStaticMessage("Error Getting Operation Type from Request"), e);
    }
  }

  private Message buildMessage(SoapRequest request) {
    Message.PayloadBuilder builder = Message.builder();
    TypedValue<SoapSubFlowPayload> payload = buildPayload(request);
    TypedValue<SoapSubFlowAttributes> attributes = buildAttributes(request);

    return builder.payload(payload).attributes(attributes).build();
  }

  private TypedValue<SoapSubFlowAttributes> buildAttributes(SoapRequest request) {
    final SoapSubFlowAttributes attributes = new SoapSubFlowAttributes(request.getTransportHeaders());
    return wrap(attributes, APPLICATION_JAVA);
  }

  private Flow findFlow(final SoapkitConfiguration config, final String operation) {
    // Flow name in Sync with Scaffolding, e.g. OrderTshirt:\api-config
    final String flowName = format("%s:\\%s", operation, config.getName());

    final Optional<Component> component = componentLocator.find(Location.builder().globalName(flowName).build());

    return (Flow) component
        .orElseThrow(() -> new SoapkitRouterException(createStaticMessage(format(ERROR_MISSING_FLOW,
                                                                                 config.getName(),
                                                                                 operation,
                                                                                 flowName))));
  }

  private SoapResponse buildResponse(Event event, OperationType type) {
    SoapSubFlowAttributes attributes = handleAttributes(event.getMessage().getAttributes());
    Map<String, String> transportAdditionalData = new HashMap<>();
    Map<String, String> protocolHeaders = new HashMap<>(attributes.getProtocolHeaders());
    String statusCode = getStatusCode(event, config.getHttpStatusVarName());

    if (statusCode != null) {
      transportAdditionalData.put(STATUS_CODE, statusCode);
      protocolHeaders.put(HTTP_STATUS_PROTOCOL_HEADER, statusCode);
    }

    if (OperationType.ONE_WAY.equals(type)) {
      if (statusCode == null) {
        transportAdditionalData.put(STATUS_CODE, String.valueOf(ACCEPTED.getStatusCode()));
        protocolHeaders.put(HTTP_STATUS_PROTOCOL_HEADER, String.valueOf(ACCEPTED.getStatusCode()));
      }

      return new EmptySoapResponse(protocolHeaders, transportAdditionalData, event.getVariables());
    }

    return new ImmutableSoapResponse(event,
                                     protocolHeaders,
                                     transportAdditionalData,
                                     APPLICATION_XML);
  }

  private static String getStatusCode(Event event, String httpStatusVarName) {
    TypedValue statusCode = httpStatusVarName != null ? event.getVariables().get(httpStatusVarName) : null;
    return statusCode == null ? null : String.valueOf(statusCode.getValue());
  }

  private SoapSubFlowAttributes handleAttributes(final TypedValue<Object> attributes) {

    final Object value = attributes.getValue();

    if (value instanceof Map) {
      final Map map = cast(value);
      final Object protocolHeaders = map.get("responseProtocolHeaders");

      if (protocolHeaders instanceof Map) {
        return new SoapSubFlowAttributes(cast(protocolHeaders));
      }
    }
    return new SoapSubFlowAttributes();
  }

  private TypedValue<SoapSubFlowPayload> buildPayload(SoapRequest request) {
    final SoapSubFlowPayload payload =
        new SoapSubFlowPayload(wrapBody(request.getContent()), wrapHeaders(request.getSoapHeaders()),
                               wrapAttachments(request.getAttachments()));
    return wrap(payload, APPLICATION_JAVA);
  }

  private TypedValue<InputStream> wrapBody(final InputStream body) {
    final DataType dataType = DataType.builder().type(InputStream.class).mediaType(APPLICATION_XML).build();
    return new TypedValue(streamingHelper.resolveCursorProvider(body), dataType);
  }

  private Map<String, TypedValue<InputStream>> wrapAttachments(final Map<String, SoapAttachment> attachments) {
    final ImmutableMap.Builder<String, TypedValue<InputStream>> wrapped = ImmutableMap.builder();
    attachments.forEach((k, v) -> {
      DataType dataType = DataType.builder().type(InputStream.class).mediaType(v.getContentType()).build();
      wrapped.put(k, new TypedValue(streamingHelper.resolveCursorProvider(v.getContent()), dataType));
    });
    return wrapped.build();
  }

  private <T> TypedValue<T> wrap(T object, org.mule.runtime.api.metadata.MediaType mediaType) {
    return new TypedValue<>(object, DataType.builder().mediaType(mediaType).build());
  }

  private Map<String, TypedValue<String>> wrapHeaders(Map<String, String> headers) {
    final ImmutableMap.Builder<String, TypedValue<String>> wrapped = ImmutableMap.builder();
    headers.forEach((k, v) -> wrapped.put(k, new TypedValue(v, DataType.builder().mediaType(APPLICATION_XML).build())));
    return wrapped.build();
  }
}
