/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.soapkit.soap.server;

import org.apache.cxf.Bus;
import org.apache.cxf.BusException;
import org.apache.cxf.binding.Binding;
import org.apache.cxf.binding.soap.interceptor.SoapHeaderInterceptor;
import org.apache.cxf.bus.spring.SpringBusFactory;
import org.apache.cxf.databinding.stax.StaxDataBinding;
import org.apache.cxf.databinding.stax.StaxDataBindingFeature;
import org.apache.cxf.databinding.stax.StaxDataBindingInterceptor;
import org.apache.cxf.endpoint.Server;
import org.apache.cxf.frontend.ServerFactoryBean;
import org.apache.cxf.interceptor.AttachmentOutInterceptor;
import org.apache.cxf.interceptor.Interceptor;
import org.apache.cxf.message.Message;
import org.apache.cxf.phase.PhaseInterceptor;
import org.apache.cxf.transport.ConduitInitiatorManager;
import org.apache.cxf.transport.DestinationFactoryManager;
import org.apache.cxf.transport.local.LocalTransportFactory;
import org.apache.cxf.wsdl.WSDLConstants;
import org.jetbrains.annotations.NotNull;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.core.api.util.ClassUtils;
import org.mule.soap.api.SoapVersion;
import org.mule.soapkit.soap.api.server.SoapServer;
import org.mule.soapkit.soap.api.server.SoapServerConfiguration;
import org.mule.soapkit.soap.api.server.SoapServerFactory;
import org.mule.soapkit.soap.server.interceptor.AttachmentOutputInterceptor;
import org.mule.soapkit.soap.server.interceptor.CopyAttachmentInInterceptor;
import org.mule.soapkit.soap.server.interceptor.CopyAttachmentOutInterceptor;
import org.mule.soapkit.soap.server.interceptor.CustomNamespacesInterceptor;
import org.mule.soapkit.soap.server.interceptor.InputSoapHeadersInterceptor;
import org.mule.soapkit.soap.server.interceptor.OutputPayloadInterceptor;
import org.mule.soapkit.soap.server.interceptor.OutputSoapHeadersInterceptor;
import org.mule.soapkit.soap.server.interceptor.RemoveCheckClosingTagsInterceptor;
import org.mule.soapkit.soap.server.interceptor.SeekBopInterceptor;
import org.mule.soapkit.soap.server.interceptor.StaxSchemaValidationInInterceptor;
import org.mule.soapkit.soap.server.interceptor.WriteXmlDeclarationOutInterceptor;
import org.mule.soapkit.soap.server.transport.ProxyTransport;

import javax.xml.namespace.QName;
import java.util.List;

import static java.lang.Boolean.parseBoolean;
import static java.lang.System.getProperty;

/**
 * {@link SoapServerFactory} implementation that creates {@link SoapCxfServer} instances.
 *
 * @since 1.1
 */
public class SoapCxfServerFactory implements SoapServerFactory {

  public static final String MULE_SOAPKIT_HEADER_VALIDATION = "mule.soapkit.headerValidation";

  private final Bus bus;

  public SoapCxfServerFactory() {
    bus = ClassUtils.withContextClassLoader(SoapCxfServerFactory.class.getClassLoader(),
                                            () -> new SpringBusFactory().createBus((String) null, true));

    // Local Transport (Documentation @ http://cxf.apache.org/docs/local-transport.html)
    final LocalTransportFactory transport = new ProxyTransport();

    final DestinationFactoryManager dfm = bus.getExtension(DestinationFactoryManager.class);

    dfm.registerDestinationFactory(LocalTransportFactory.TRANSPORT_ID, transport);
    dfm.registerDestinationFactory("http://cxf.apache.org/transports/http", transport);
    dfm.registerDestinationFactory("http://cxf.apache.org/transports/http/configuration", transport);

    dfm.registerDestinationFactory("http://schemas.xmlsoap.org/soap/http", transport);
    dfm.registerDestinationFactory("http://schemas.xmlsoap.org/wsdl/soap/http", transport);
    dfm.registerDestinationFactory("http://schemas.xmlsoap.org/wsdl/http/", transport);
    dfm.registerDestinationFactory("http://www.w3.org/2003/05/soap/bindings/HTTP/", transport);
    dfm.registerDestinationFactory("http://cxf.apache.org/transports/jms", transport);
    dfm.registerDestinationFactory("http://mule.codehaus.org/cxf", transport);

    final ConduitInitiatorManager extension = bus.getExtension(ConduitInitiatorManager.class);
    extension.registerConduitInitiator(LocalTransportFactory.TRANSPORT_ID, transport);
    extension.registerConduitInitiator("http://schemas.xmlsoap.org/soap/http", transport);
    extension.registerConduitInitiator("http://cxf.apache.org/transports/http", transport);
    extension.registerConduitInitiator("http://cxf.apache.org/transports/http/configuration",
                                       transport);

    extension.registerConduitInitiator("http://schemas.xmlsoap.org/wsdl/soap/", transport);
    extension.registerConduitInitiator("http://schemas.xmlsoap.org/soap/http/", transport);
    extension.registerConduitInitiator("http://schemas.xmlsoap.org/wsdl/soap/http", transport);
    extension.registerConduitInitiator("http://schemas.xmlsoap.org/wsdl/http/", transport);
    extension.registerConduitInitiator("http://www.w3.org/2003/05/soap/bindings/HTTP/", transport);
    extension.registerConduitInitiator("http://cxf.apache.org/bindings/xformat", transport);
    extension.registerConduitInitiator("http://cxf.apache.org/transports/jms", transport);
    extension.registerConduitInitiator("http://mule.codehaus.org/cxf", transport);

    try {
      // Force the HTTP transport to load if available, otherwise it could
      // overwrite our namespaces later
      extension.getConduitInitiator("http://schemas.xmlsoap.org/soap/http");
    } catch (final BusException ignore) {
    }
  }

  /**
   * Creates a new instance of a {@link SoapCxfServer} for the given address ans soap version.
   *
   * @throws ConnectionException if the client couldn't be created.
   */
  @Override
  public SoapServer create(final SoapServerConfiguration config) throws ConnectionException {
    return createSoapCxfServer(config);
  }

  private SoapCxfServer createSoapCxfServer(final SoapServerConfiguration config) {
    final Server server = createCxfServer(config);

    return SoapCxfServer.create(server, config.getPortModel(), config.isMtomEnabled(), config.isValidationEnabled(),
                                config.getValidationErrorLevel());
  }

  @NotNull
  private Server createCxfServer(final SoapServerConfiguration config) {
    final ServerFactoryBean factory = new ServerFactoryBean();

    factory.setBus(bus);
    factory.setDataBinding(new StaxDataBinding());
    factory.getFeatures().add(new StaxDataBindingFeature());

    /*
    final Map<String, Object> props = new HashMap<>();
    props.put(Message.MTOM_ENABLED, config.isMtomEnabled() ? Boolean.TRUE : Boolean.FALSE);
    factory.setProperties(props);
    */
    final SoapVersion soapVersion = config.getVersion();
    if (soapVersion != null)
      factory.setBindingId(getBindingIdForSoapVersion(soapVersion));
    factory.setServiceClass(ProxyService.class);

    final ProxyServiceFactoryBean serviceFactory = new ProxyServiceFactoryBean(config);

    final QName service = new QName(config.getNamespace(), config.getService());
    serviceFactory.setServiceName(service);

    factory.setServiceFactory(serviceFactory);

    // In Proxy Interceptors (Ctx to Mule)
    factory.getInInterceptors().add(new InputSoapHeadersInterceptor());
    factory.getInInterceptors().add(new CopyAttachmentInInterceptor());

    // Out Proxy Interceptors (Mule to Ctx)
    factory.getOutInterceptors().add(new OutputSoapHeadersInterceptor());
    factory.getOutInterceptors().add(new CopyAttachmentOutInterceptor());
    factory.getOutInterceptors().add(new OutputPayloadInterceptor());
    factory.getOutInterceptors().add(new WriteXmlDeclarationOutInterceptor(config.isResponseXMLDeclarationEnabled()));
    factory.getOutInterceptors().add(new CustomNamespacesInterceptor(config.getNamespacePrefixes()));

    // This interceptor analyze Payload and set Bop if is not set
    final SeekBopInterceptor seekBopInterceptor = new SeekBopInterceptor();
    seekBopInterceptor.addBefore(StaxDataBindingInterceptor.class.getName());
    factory.getInInterceptors().add(seekBopInterceptor);

    if (config.isPayloadBody() && config.isValidationEnabled()) {
      if (isHeaderValidationEnabled())
        factory.getInInterceptors().add(new SoapHeaderInterceptor());

      factory.getInInterceptors().add(new StaxSchemaValidationInInterceptor());

      // This is a Hack because when validation is Enabled ReadHeadersInterceptor add ReadHeadersInterceptor.CheckClosingTagsInterceptor
      // during handleMessage() method. (it is added when ReadHeadersInterceptor is executed) and CheckClosingTagsInterceptor consume the XMLStreamReader.
      // RemoveCheckClosingTagsInterceptor removes CheckClosingTagsInterceptor prevent XMLStreamReader from being consumed.
      factory.getInInterceptors().add(new RemoveCheckClosingTagsInterceptor());
    }
    factory.setInvoker(new SoapCxfInvoker());

    final Server server = factory.create();

    // Binding is created using SoapBindingFactory and this factory add AttachmentOutInterceptor that always serialize
    // attachments in mtom.
    final Binding binding = server.getEndpoint().getBinding();
    removeInterceptor(binding.getOutInterceptors(), AttachmentOutInterceptor.class);
    removeInterceptor(binding.getOutFaultInterceptors(), AttachmentOutInterceptor.class);

    binding.getOutInterceptors().add(new AttachmentOutputInterceptor());
    binding.getOutFaultInterceptors().add(new AttachmentOutputInterceptor());

    return server;
  }

  private static void removeInterceptor(final List<Interceptor<? extends Message>> interceptors, final Class<?> clazz) {

    for (Interceptor<?> i : interceptors) {
      if (i instanceof PhaseInterceptor) {
        final PhaseInterceptor<Message> p = (PhaseInterceptor<Message>) i;

        if (p.getClass().equals(clazz)) {
          interceptors.remove(p);
          break;
        }
      }
    }
  }

  private String getBindingIdForSoapVersion(final SoapVersion soapVersion) {
    return soapVersion.getNumber().equals("1.2") ? WSDLConstants.NS_SOAP12 : WSDLConstants.NS_SOAP11;
  }

  private static boolean isHeaderValidationEnabled() {
    String headerValiationEnabled = getProperty(MULE_SOAPKIT_HEADER_VALIDATION);

    if (headerValiationEnabled == null)
      return true;

    return parseBoolean(headerValiationEnabled);
  }
}
