/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.soapkit.soap.interceptor;

import com.google.common.collect.ImmutableMap;
import org.apache.cxf.binding.soap.SoapFault;
import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.message.Attachment;
import org.apache.cxf.message.Exchange;
import org.apache.cxf.message.Message;
import org.apache.cxf.phase.AbstractPhaseInterceptor;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.soap.api.message.SoapAttachment;

import javax.activation.DataHandler;
import java.io.IOException;
import java.util.Collection;

import static org.apache.cxf.interceptor.Fault.FAULT_CODE_SERVER;
import static org.mule.soapkit.soap.SoapConstants.MULE_ATTACHMENTS_KEY;

/**
 * CXF out interceptor that collects the received Mtom SOAP attachments in the response, transforms it to message parts and stores
 * them in the response message {@link Exchange} as {@link SoapAttachment}.
 * if no attachments are returned an empty map is set.
 *
 * @since 1.1
 */
public class OutputMtomSoapAttachmentsInterceptor extends AbstractPhaseInterceptor<Message> {

  // Copied from SOAP Service, Phase added in constructor
  public OutputMtomSoapAttachmentsInterceptor(String phase) {
    super(phase);
  }

  @Override
  public void handleMessage(Message message) throws Fault {
    final ImmutableMap.Builder<String, SoapAttachment> result = ImmutableMap.builder();
    Collection<Attachment> attachments = message.getAttachments();
    if (attachments != null && !attachments.isEmpty()) {
      attachments.forEach(a -> result.put(getName(a), getSoapAttachment(a)));
    }
    message.getExchange().put(MULE_ATTACHMENTS_KEY, result.build());
  }

  private String getName(Attachment attachment) {
    DataHandler dataHandler = attachment.getDataHandler();
    return dataHandler.getName() != null ? dataHandler.getName() : attachment.getId();
  }

  private SoapAttachment getSoapAttachment(Attachment attachment) {
    DataHandler dataHandler = attachment.getDataHandler();
    try {
      MediaType contentType = MediaType.parse(dataHandler.getContentType());
      return new SoapAttachment(dataHandler.getInputStream(), contentType.toString());
    } catch (IOException e) {
      throw new SoapFault("Error copying received attachment [" + getName(attachment) + "]", e, FAULT_CODE_SERVER);
    }
  }
}
