/**
 * (c) 2003-2015 MuleSoft, Inc. The software in this package is
 * published under the terms of the CPAL v1.0 license, a copy of which
 * has been included with this distribution in the LICENSE.md file.
 */

package org.mule.modules.zendesk;

import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import javax.ws.rs.core.MediaType;

import org.apache.commons.lang.Validate;
import org.mule.api.ConnectionException;
import org.mule.api.ConnectionExceptionCode;
import org.mule.api.annotations.Configurable;
import org.mule.api.annotations.Connect;
import org.mule.api.annotations.ConnectionIdentifier;
import org.mule.api.annotations.Connector;
import org.mule.api.annotations.Disconnect;
import org.mule.api.annotations.Processor;
import org.mule.api.annotations.ValidateConnection;
import org.mule.api.annotations.param.ConnectionKey;
import org.mule.api.annotations.param.Default;
import org.mule.api.annotations.param.Optional;
import org.mule.commons.jersey.JerseyUtil;
import org.mule.commons.jersey.provider.GsonProvider;
import org.mule.modules.utils.pagination.PaginatedIterable;
import org.mule.modules.zendesk.jersey.GsonFactory;
import org.mule.modules.zendesk.jersey.ZendeskException;
import org.mule.modules.zendesk.jersey.ZendeskResponseHandler;
import org.mule.modules.zendesk.model.ActivityStream;
import org.mule.modules.zendesk.model.Attachment;
import org.mule.modules.zendesk.model.Automation;
import org.mule.modules.zendesk.model.Category;
import org.mule.modules.zendesk.model.EntityType;
import org.mule.modules.zendesk.model.FileUpload;
import org.mule.modules.zendesk.model.Forum;
import org.mule.modules.zendesk.model.ForumSubscription;
import org.mule.modules.zendesk.model.Group;
import org.mule.modules.zendesk.model.GroupMembership;
import org.mule.modules.zendesk.model.JobStatus;
import org.mule.modules.zendesk.model.Locale;
import org.mule.modules.zendesk.model.Macro;
import org.mule.modules.zendesk.model.Organization;
import org.mule.modules.zendesk.model.OrganizationRelated;
import org.mule.modules.zendesk.model.Request;
import org.mule.modules.zendesk.model.SatisfactionRating;
import org.mule.modules.zendesk.model.SharingAgreement;
import org.mule.modules.zendesk.model.SuspendedTicket;
import org.mule.modules.zendesk.model.Ticket;
import org.mule.modules.zendesk.model.TicketAudit;
import org.mule.modules.zendesk.model.TicketComment;
import org.mule.modules.zendesk.model.TicketField;
import org.mule.modules.zendesk.model.TicketMetric;
import org.mule.modules.zendesk.model.Topic;
import org.mule.modules.zendesk.model.TopicComment;
import org.mule.modules.zendesk.model.TopicSubscription;
import org.mule.modules.zendesk.model.TopicVote;
import org.mule.modules.zendesk.model.Trigger;
import org.mule.modules.zendesk.model.User;
import org.mule.modules.zendesk.model.UserIdentity;
import org.mule.modules.zendesk.model.UserRelated;
import org.mule.modules.zendesk.model.View;
import org.mule.modules.zendesk.model.ViewExecute;
import org.mule.modules.zendesk.model.responses.ApplyMacroResponse;
import org.mule.modules.zendesk.model.responses.EntityResponse;
import org.mule.modules.zendesk.model.responses.SearchResponse;
import org.mule.util.StringUtils;

import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.config.ClientConfig;
import com.sun.jersey.api.client.config.DefaultClientConfig;
import com.sun.jersey.api.client.filter.HTTPBasicAuthFilter;
import com.sun.jersey.api.client.filter.LoggingFilter;
import com.sun.jersey.api.json.JSONConfiguration;

/**
 * Zendesk is a customer support platform that supports more than 25,000 businesses and 60 million customers in 140 countries around the globe. Many of these businesses use the
 * Zendesk API to automate and enhance their customer support with Zendesk.
 *
 * @author MuleSoft, Inc.
 */
@SuppressWarnings("unused")
@Connector(name = "zendesk", schemaVersion = "1.0", friendlyName = "Zendesk")
public class ZendeskConnector {

    /**
     * Zendesk API endpoint. E.g.: https://{subdomain}.zendesk.com/api/v2
     */
    @Configurable
    private String apiUrl;

    /**
     * Jersey client
     */
    private Client client;

    private JerseyUtil jerseyUtil;

    private WebResource apiResource;

    /**
     * Connect to Zendesk
     *
     * @param username
     *            Zendesk username
     * @param password
     *            Zendesk password
     * @throws ConnectionException
     *             when the service is unreachable or the credentials are not valid
     */
    @Connect
    public void connect(@ConnectionKey String username, String password) throws ConnectionException {

        ClientConfig clientConfig = new DefaultClientConfig();
        clientConfig.getFeatures().put(JSONConfiguration.FEATURE_POJO_MAPPING, Boolean.TRUE);
        clientConfig.getSingletons().add(new GsonProvider(GsonFactory.get()));

        setClient(Client.create(clientConfig));
        getClient().addFilter(new HTTPBasicAuthFilter(username, password));
        
        this.initJerseyUtil();

        setApiResource(getClient().resource(getApiUrl()));
        try {
            if (this.getCurrentUser().getEmail().equals("invalid@example.com")) {
                throw new ConnectionException(ConnectionExceptionCode.INCORRECT_CREDENTIALS, null, "Please check your credentials and try again.");
            }
        } catch (Exception e) {
            throw new ConnectionException(ConnectionExceptionCode.INCORRECT_CREDENTIALS, null,
                    "The service couldn't be validated. Please check your URL and credentials and try again");
        }

    }

    /**
     * Initialize JerseyUtils
     */
    private void initJerseyUtil() {
        JerseyUtil.Builder builder = JerseyUtil.builder();
        builder.setResponseHandler(new ZendeskResponseHandler());
        setJerseyUtil(builder.build());
    }

    /**
     * Disconnect
     */
    @Disconnect
    public void disconnect() {
        setClient(null);
    }

    /**
     * Are we connected
     */
    @ValidateConnection
    public boolean isConnected() {
        return getClient() != null;
    }

    /**
     * Are we connected
     */
    @ConnectionIdentifier
    public String connectionId() {
        return "{ User: \"" + getCurrentUser().getEmail() + " \"}";
    }

    /**
     * Returns current user information
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-current-user}
     *
     * @return current user
     */
    @Processor
    public User getCurrentUser() {
        return this.jerseyUtil.get(this.apiResource.path("users").path("me.json"), EntityResponse.class, 200, 201).getUser();
    }

    /**
     * Creates a new Zendesk ticket
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/tickets.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-ticket}
     *
     * @param ticket
     *            ticket to create
     * @return new Ticket
     */
    @Processor
    public Ticket createTicket(@Default("#[payload]") Ticket ticket) {
        return this.jerseyUtil.post(this.apiResource.path("tickets.json").entity(ticket), EntityResponse.class, 201).getTicket();
    }

    /**
     * Get list of tickets
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/tickets.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-tickets}
     *
     * @return list of tickets
     */
    @Processor
    public Iterable<Ticket> listTickets() {
        return createPaginationResults(EntityType.TICKET, this.jerseyUtil.get(this.apiResource.path("tickets.json"), EntityResponse.class, 200));
    }

    /**
     * Get ticket
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/tickets.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-ticket}
     *
     * @param ticketId
     *            ticket id
     * @return ticket information
     */
    @Processor
    public Ticket getTicket(Long ticketId) {
        return this.jerseyUtil.get(this.apiResource.path("tickets").path(ticketId + ".json"), EntityResponse.class, 200).getTicket();
    }

    /**
     * Update a ticket
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/tickets.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:update-ticket}
     *
     * @param ticket
     *            ticket to update
     * @return updated ticket
     */
    @Processor
    public Ticket updateTicket(@Default("#[payload]") Ticket ticket) {
        return this.jerseyUtil.put(this.apiResource.path("tickets").path(ticket.getId() + ".json").accept(MediaType.APPLICATION_JSON).entity(ticket), EntityResponse.class, 200).getTicket();
    }

    /**
     * Delete a ticket
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/tickets.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:delete-ticket}
     *
     * @param ticket
     *            ticket to delete
     * @return deleted ticket
     */
    @Processor
    public Ticket deleteTicket(@Default("#[payload]") Ticket ticket) {
        this.jerseyUtil.delete(this.apiResource.path("tickets").path(ticket.getId() + ".json").entity(ticket), String.class, 204);
        return ticket;
    }

    /**
     * List Ticket audits for a given ticket
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/ticket_audits.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-ticket-audits}
     *
     * @param ticketId
     *            ticket id
     * @return tickets audits list
     */
    @Processor
    public Iterable<TicketAudit> listTicketAudits(Long ticketId) {
        return createPaginationResults(EntityType.AUDIT,
                this.jerseyUtil.get(this.apiResource.path("tickets").path(ticketId.toString()).path("audits.json"), EntityResponse.class, 200));
    }

    /**
     * Get a ticket audit
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/ticket_audits.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-ticket-audit}
     *
     * @param ticketId
     *            ticket id
     * @param auditId
     *            audit id
     * @return ticket audit information
     */
    @Processor
    public TicketAudit getTicketAudit(Long ticketId, Long auditId) {
        return this.jerseyUtil.get(this.apiResource.path("tickets").path(ticketId.toString()).path("audits").path(auditId + ".json"), EntityResponse.class, 200).getAudit();
    }

    /**
     * List Ticket audits for a given ticket
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/ticket_fields.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-ticket-fields}
     *
     * @return tickets fields list
     */
    @Processor
    public Iterable<TicketField> listTicketFields() {
        return createPaginationResults(EntityType.TICKET_FIELD, this.jerseyUtil.get(this.apiResource.path("ticket_fields.json"), EntityResponse.class, 200));
    }

    /**
     * Create a new ticket field
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/ticket_fields.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-ticket-field}
     *
     * @param ticketField
     *            Ticket Field object to create
     * @return tickets field
     */
    @Processor
    public TicketField createTicketField(@Default("#[payload]") TicketField ticketField) {
        return this.jerseyUtil.post(this.apiResource.path("ticket_fields.json").entity(ticketField), EntityResponse.class, 201).getTicketField();
    }

    /**
     * Update ticket field
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/ticket_fields.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:update-ticket-field}
     *
     * @param ticketField
     *            Ticket Field to be updated
     * @return tickets field
     */
    @Processor
    public TicketField updateTicketField(@Default("#[payload]") TicketField ticketField) {
        return this.jerseyUtil.put(this.apiResource.path("ticket_fields").path(ticketField.getId() + ".json").accept(MediaType.APPLICATION_JSON).entity(ticketField), EntityResponse.class, 200).getTicketField();
    }

    /**
     * It allows moving data from legacy systems into Zendesk. Zendesk does not run triggers or the likes during bulk imports like these.
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/ticket_import.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:bulk-import-ticket}
     *
     * @param ticket
     *            ticket to be created
     * @return ticket created
     */
    @Processor
    public Ticket bulkImportTicket(@Default("#[payload]") Ticket ticket) {
        return this.jerseyUtil.put(this.apiResource.path("imports").path("tickets.json").entity(ticket), EntityResponse.class, 201).getTicket();
    }

    /**
     * List ticket metrics
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/ticket_metrics.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-ticket-metrics}
     *
     * @return ticket metrics list
     */
    @Processor
    public Iterable<TicketMetric> listTicketMetrics() {
        return createPaginationResults(EntityType.TICKET_METRIC, this.jerseyUtil.get(this.apiResource.path("ticket_metrics.json"), EntityResponse.class, 200));
    }

    /**
     * Get ticket metric
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/ticket_metrics.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-ticket-metrics}
     *
     * @param ticketId
     *            ticket to get metrics
     * @return ticket metrics
     */
    @Processor
    public TicketMetric getTicketMetrics(Long ticketId) {
        return this.jerseyUtil.get(this.apiResource.path("tickets").path(ticketId.toString()).path("metrics.json"), EntityResponse.class, 200).getTicketMetric();
    }

    /**
     * List views
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/views.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-views}
     *
     * @return available views
     */
    @Processor
    public Iterable<View> listViews() {
        return createPaginationResults(EntityType.VIEW, this.jerseyUtil.get(this.apiResource.path("views.json"), EntityResponse.class, 200));
    }

    /**
     * List active views
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/views.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-active-views}
     *
     * @return active views
     */
    @Processor
    public Iterable<View> listActiveViews() {
        return createPaginationResults(EntityType.VIEW, this.jerseyUtil.get(this.apiResource.path("views").path("active.json"), EntityResponse.class, 200));
    }

    /**
     * List views
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/views.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-compact-views}
     *
     * @return A compacted list of shared and personal views available to the current user
     */
    @Processor
    public Iterable<View> listCompactViews() {
        return createPaginationResults(EntityType.VIEW, this.jerseyUtil.get(this.apiResource.path("views").path("compact.json"), EntityResponse.class, 200));
    }

    /**
     * Get a view information
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/views.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-view}
     *
     * @param viewId
     *            view information to be retrieved
     * @return view information
     */
    @Processor
    public View getView(Long viewId) {
        return this.jerseyUtil.get(this.apiResource.path("views").path(viewId + ".json"), EntityResponse.class, 200).getView();
    }

    /**
     * Create new view
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/views.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-view}
     *
     * @param view
     *            view to be created
     * @return new view
     */
    @Processor
    public View createView(@Default("#[payload]") View view) {
        return this.jerseyUtil.post(this.apiResource.path("views.json").entity(view), EntityResponse.class, 201).getView();
    }

    /**
     * Update a view
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/views.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:update-view}
     *
     * @param view
     *            view to be updated
     * @return updated view
     */
    @Processor
    public View updateView(@Default("#[payload]") View view) {
        return this.jerseyUtil.put(this.apiResource.path("views").path(view.getId() + ".json").accept(MediaType.APPLICATION_JSON).entity(view), EntityResponse.class, 200).getView();
    }

    /**
     * Get tickets from view
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/views.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-tickets-from-view}
     *
     * @param viewId
     *            view to get tickets from
     * @return list of tickets
     */
    @Processor
    public Iterable<Ticket> getTicketsFromView(Long viewId) {
        return createPaginationResults(EntityType.TICKET,
                this.jerseyUtil.get(this.apiResource.path("views").path(viewId.toString()).path("tickets.json"), EntityResponse.class, 200));
    }

    /**
     * Execute a view
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/views.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:execute-view}
     *
     * @param viewId
     *            view id to be executed
     * @return view execution response
     */
    @Processor
    public ViewExecute executeView(Long viewId) {
        return this.jerseyUtil.get(this.apiResource.path("views").path(viewId.toString()).path("execute.json"), ViewExecute.class, 200);
    }

    /**
     * Calculates the size of the view in terms of number of tickets the view will return
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/views.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-view-counts}
     *
     * @param viewIds
     *            list of view ids
     * @return view count response
     */
    @Processor
    public ViewExecute getViewCounts(List<String> viewIds) {
        return this.jerseyUtil.get(this.apiResource.path("views").path("counts_many.json").queryParam("ids", StringUtils.join(viewIds, ",")), ViewExecute.class, 200);
    }

    /**
     * Get users
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/users.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-users}
     *
     * @return list of users
     */
    @Processor
    public Iterable<User> listUsers() {
        return createPaginationResults(EntityType.USER, this.jerseyUtil.get(this.apiResource.path("users.json"), EntityResponse.class, 200));
    }

    /**
     * Get users per organization id
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/users.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-users-by-organization}
     *
     * @param organizationId
     *            organization id
     * @return list of users
     */
    @Processor
    public Iterable<User> listUsersByOrganization(Long organizationId) {
        return createPaginationResults(EntityType.USER,
                this.jerseyUtil.get(this.apiResource.path("organizations").path(organizationId.toString()).path("users.json"), EntityResponse.class, 200));
    }

    /**
     * Get user information
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/users.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-user}
     *
     * @param userId
     *            to get retrieved
     * @return user information
     */
    @Processor
    public User getUser(Long userId) {
        return this.jerseyUtil.get(this.apiResource.path("users").path(userId + ".json"), EntityResponse.class, 200).getUser();
    }

    /**
     * Get users per group id
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/users.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-users-by-group}
     *
     * @param groupId
     *            group id
     * @return user information
     */
    @Processor
    public Iterable<User> listUsersByGroup(Long groupId) {
        return createPaginationResults(EntityType.USER, this.jerseyUtil.get(this.apiResource.path("groups").path(groupId.toString()).path("users.json"), EntityResponse.class, 200));
    }

    /**
     * Get user related information
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/users.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-user-related}
     *
     * @param userId
     *            to get information
     * @return user related information
     */
    @Processor
    public UserRelated getUserRelated(Long userId) {
        return this.jerseyUtil.get(this.apiResource.path("users").path(userId.toString()).path("related.json"), EntityResponse.class, 200).getUserRelated();
    }

    /**
     * Create user
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/users.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-user}
     *
     * @param user
     *            to be created
     * @return new user information
     */
    @Processor
    public User createUser(@Default("#[payload]") User user) {
        return this.jerseyUtil.post(this.apiResource.path("users.json").entity(user), EntityResponse.class, 201).getUser();
    }

    /**
     * Update user
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/users.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:update-user}
     *
     * @param user
     *            to be updated
     * @return updated user information
     */
    @Processor
    public User updateUser(@Default("#[payload]") User user) {
        return this.jerseyUtil.put(this.apiResource.path("users").path(user.getId() + ".json").accept(MediaType.APPLICATION_JSON).entity(user), EntityResponse.class, 200).getUser();
    }

    /**
     * Delete user
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/users.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:delete-user}
     *
     * @param user
     *            to be deleted
     * @return deleted user information
     */
    @Processor
    public User deleteUser(@Default("#[payload]") User user) {
        return this.jerseyUtil.delete(this.apiResource.path("users").path(user.getId() + ".json"), EntityResponse.class, 200).getUser();
    }

    /**
     * List requests
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/requests.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-requests}
     *
     * @return list of requests
     */
    @Processor
    public Iterable<Request> listRequests() {
        return createPaginationResults(EntityType.REQUEST, this.jerseyUtil.get(this.apiResource.path("requests.json"), EntityResponse.class, 200));
    }

    /**
     * List requests by status. Available values are ccd, open, solved.
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/requests.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-requests-by-status}
     *
     * @param status
     *            request status. Available values are ccd, open, solved.
     * @return list of requests
     */
    @Processor
    public Iterable<Request> listRequestsByStatus(String status) {
        Validate.isTrue(Arrays.asList("ccd", "open", "solved").contains(status), "The status is not valid for the operation");

        return createPaginationResults(EntityType.REQUEST, this.jerseyUtil.get(this.apiResource.path("requests").path(status + ".json"), EntityResponse.class, 200));
    }

    /**
     * List requests by user
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/requests.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-requests-by-user}
     *
     * @param userId
     *            user id
     * @return list of requests
     */
    @Processor
    public Iterable<Request> listRequestsByUser(Long userId) {
        return createPaginationResults(EntityType.REQUEST,
                this.jerseyUtil.get(this.apiResource.path("users").path(userId.toString()).path("requests.json"), EntityResponse.class, 200));
    }

    /**
     * Get request information
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/requests.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-request}
     *
     * @param requestId
     *            request id
     * @return list of requests
     */
    @Processor
    public Request getRequest(Long requestId) {
        return this.jerseyUtil.get(this.apiResource.path("requests").path(requestId + ".json"), EntityResponse.class, 200).getRequest();
    }

    /**
     * Create request
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/requests.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-request}
     *
     * @param request
     *            request to be created
     * @return new request information
     */
    @Processor
    public Request createRequest(@Default("#[payload]") Request request) {
        return this.jerseyUtil.post(this.apiResource.path("requests.json").entity(request), EntityResponse.class, 201).getRequest();
    }

    /**
     * Update request
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/requests.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:update-request}
     *
     * @param request
     *            request to be updated
     * @return updated request information
     */
    @Processor
    public Request updateRequest(@Default("#[payload]") Request request) {
        return this.jerseyUtil.put(this.apiResource.path("requests").path(request.getId() + ".json").accept(MediaType.APPLICATION_JSON).entity(request), EntityResponse.class, 200).getRequest();
    }

    /**
     * List requests comments
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/requests.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-request-comments}
     *
     * @param requestId
     *            request id
     * @return list of comments for the given request
     */
    @Processor
    public Iterable<TicketComment> listRequestComments(Long requestId) {
        return createPaginationResults(EntityType.TICKET_COMMENT,
                this.jerseyUtil.get(this.apiResource.path("requests").path(requestId.toString()).path("comments.json"), EntityResponse.class, 200));
    }

    /**
     * Get a particular comment for a given request
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/requests.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-request-comment}
     *
     * @param requestId
     *            request id
     * @param commentId
     *            commentId
     * @return request comment information
     */
    @Processor
    public TicketComment getRequestComment(Long requestId, Long commentId) {
        return this.jerseyUtil.get(this.apiResource.path("requests").path(requestId.toString()).path("comments").path(commentId + ".json"), EntityResponse.class, 200).getComment();
    }

    /**
     * Return all user identities for a given user id
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/user_identities.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-user-identities}
     *
     * @param userId
     *            user id
     * @return list of user identities
     */
    @Processor
    public Iterable<UserIdentity> listUserIdentities(Long userId) {
        return createPaginationResults(EntityType.IDENTITY,
                this.jerseyUtil.get(this.apiResource.path("users").path(userId.toString()).path("identities.json"), EntityResponse.class, 200));
    }

    /**
     * Get an user identity
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/user_identities.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-user-identity}
     *
     * @param userId
     *            user id
     * @param userIdentityId
     *            user identity id
     * @return user identity information
     */
    @Processor
    public UserIdentity getUserIdentity(Long userId, Long userIdentityId) {
        return this.jerseyUtil.get(this.apiResource.path("users").path(userId.toString()).path("identities").path(userIdentityId + ".json"), EntityResponse.class, 200)
                .getIdentity();
    }

    /**
     * Create an user identity for a given user
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/user_identities.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-user-identity}
     *
     * @param userIdentity
     *            user identity to be created
     * @return user identity information
     */
    @Processor
    public UserIdentity createUserIdentity(@Default("#[payload]") UserIdentity userIdentity) {
        Validate.notNull(userIdentity.getUserId(), "UserID in identity cannot be null");

        return this.jerseyUtil.post(this.apiResource.path("users").path(userIdentity.getUserId().toString()).path("identities.json").entity(userIdentity), EntityResponse.class,
                201).getIdentity();
    }

    /**
     * Verify user identity
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/user_identities.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:verify-user-identity}
     *
     * @param userIdentity
     *            user identity to be verified
     * @return user identity information
     */
    @Processor
    public UserIdentity verifyUserIdentity(@Default("#[payload]") UserIdentity userIdentity) {

        Validate.notNull(userIdentity.getUserId(), "UserID in identity cannot be null");
        return this.jerseyUtil.put(
                this.apiResource.path("users").path(userIdentity.getUserId().toString()).path("identities").path(userIdentity.getId().toString()).path("verify.json"),
                EntityResponse.class, 200).getIdentity();
    }

    /**
     * Make user identity the primary
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/user_identities.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:make-user-identity-primary}
     *
     * @param userIdentity
     *            user identity to be set as primary
     * @return user identity information
     */
    @Processor
    public UserIdentity makeUserIdentityPrimary(@Default("#[payload]") UserIdentity userIdentity) {

        Validate.notNull(userIdentity.getUserId(), "UserID in identity cannot be null");
        return this.jerseyUtil.put(
                this.apiResource.path("users").path(userIdentity.getUserId().toString()).path("identities").path(userIdentity.getId().toString()).path("make_primary.json"),
                EntityResponse.class, 200).getIdentity();
    }

    /**
     * Request user id verification
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/user_identities.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:request-user-identity-verification}
     *
     * @param userIdentity
     *            user identity
     * @return user identity information
     */
    @Processor
    public UserIdentity requestUserIdentityVerification(@Default("#[payload]") UserIdentity userIdentity) {

        Validate.notNull(userIdentity.getUserId(), "UserID in identity cannot be null");
        return this.jerseyUtil
                .put(this.apiResource.path("users").path(userIdentity.getUserId().toString()).path("identities").path(userIdentity.getId().toString())
                        .path("request_verification.json"), EntityResponse.class, 200).getIdentity();
    }

    /**
     * Delete user identity
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/user_identities.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:delete-user-identity}
     *
     * @param userIdentity
     *            user identity to be deleted
     */
    @Processor
    public void deleteUserIdentity(@Default("#[payload]") UserIdentity userIdentity) {
        Validate.notNull(userIdentity.getUserId(), "UserID in identity cannot be null");
        this.jerseyUtil.delete(this.apiResource.path("users").path(userIdentity.getUserId().toString()).path("identities").path(userIdentity.getId() + ".json"), EntityResponse.class, 204);
    }

    /**
     * List groups
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/groups.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-groups}
     *
     * @return available groups
     */
    @Processor
    public Iterable<Group> listGroups() {
        return createPaginationResults(EntityType.GROUP, this.jerseyUtil.get(this.apiResource.path("groups.json"), EntityResponse.class, 200));
    }

    /**
     * List assignable groups
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/groups.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-assignable-groups}
     *
     * @return assignable groups
     */
    @Processor
    public Iterable<Group> listAssignableGroups() {
        return createPaginationResults(EntityType.GROUP, this.jerseyUtil.get(this.apiResource.path("groups.json").path("assignable.json"), EntityResponse.class, 200));
    }

    /**
     * Get group for given group id
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/groups.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-group}
     *
     * @param groupId
     *            group id
     * @return group information
     */
    @Processor
    public Group getGroup(Long groupId) {
        return this.jerseyUtil.get(this.apiResource.path("groups").path(groupId + ".json"), EntityResponse.class, 200).getGroup();
    }

    /**
     * Create group
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/groups.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-group}
     *
     * @param group
     *            group to be created
     * @return Created group
     */
    @Processor
    public Group createGroup(@Default("#[payload]") Group group) {
        return this.jerseyUtil.post(this.apiResource.path("groups.json").entity(group), EntityResponse.class, 201).getGroup();
    }

    /**
     * Update group
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/groups.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:update-group}
     *
     * @param group
     *            group to be update
     * @return Updated group
     */
    @Processor
    public Group updateGroup(@Default("#[payload]") Group group) {
        Validate.notNull(group.getId(), "GroupId cannot be null");
        return this.jerseyUtil.put(this.apiResource.path("groups").path(group.getId() + ".json").accept(MediaType.APPLICATION_JSON).entity(group), EntityResponse.class, 200).getGroup();
    }

    /**
     * Delete group
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/groups.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:delete-group}
     *
     * @param group
     *            group to be deleted
     */
    @Processor
    public void deleteGroup(@Default("#[payload]") Group group) {
        Validate.notNull(group.getId(), "GroupId cannot be null");
        this.jerseyUtil.delete(this.apiResource.path("groups").path(group.getId() + ".json").entity(group), ClientResponse.class, 204);
    }

    /**
     * List group memberships
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/group_memberships.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-group-memberships}
     *
     * @return group memberships list
     */
    @Processor
    public Iterable<GroupMembership> listGroupMemberships() {
        return createPaginationResults(EntityType.GROUP_MEMBERSHIP, this.jerseyUtil.get(this.apiResource.path("group_memberships.json"), EntityResponse.class, 200));
    }

    /**
     * List group memberships by user
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/group_memberships.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-group-memberships-by-user}
     *
     * @param userId
     *            user id
     * @return group memberships list
     */
    @Processor
    public Iterable<GroupMembership> listGroupMembershipsByUser(Long userId) {
        return createPaginationResults(EntityType.GROUP_MEMBERSHIP,
                this.jerseyUtil.get(this.apiResource.path("users").path(userId.toString()).path("group_memberships.json"), EntityResponse.class, 200));
    }

    /**
     * List group memberships by group
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/group_memberships.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-group-memberships-by-group}
     *
     * @param groupId
     *            user id
     * @return group memberships list
     */
    @Processor
    public Iterable<GroupMembership> listGroupMembershipsByGroup(Long groupId) {
        return createPaginationResults(EntityType.GROUP_MEMBERSHIP,
                this.jerseyUtil.get(this.apiResource.path("groups").path(groupId.toString()).path("memberships.json"), EntityResponse.class, 200));
    }

    /**
     * List group memberships by group
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/group_memberships.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-group-assignable-memberships}
     *
     * @return group memberships list
     */
    @Processor
    public Iterable<GroupMembership> listGroupAssignableMemberships() {
        return createPaginationResults(EntityType.GROUP_MEMBERSHIP,
                this.jerseyUtil.get(this.apiResource.path("group_memberships").path("assignable.json"), EntityResponse.class, 200));
    }

    /**
     * List assignable memberships by group
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/group_memberships.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-group-assignable-memberships-by-group}
     *
     * @param groupId
     *            user id
     * @return group memberships list
     */
    @Processor
    public Iterable<GroupMembership> listGroupAssignableMembershipsByGroup(Long groupId) {
        return createPaginationResults(EntityType.GROUP_MEMBERSHIP,
                this.jerseyUtil.get(this.apiResource.path("groups").path(groupId.toString()).path("memberships").path("assignable.json"), EntityResponse.class, 200));
    }

    /**
     * Get membership information
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/group_memberships.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-group-membership}
     *
     * @param groupMembershipId
     *            group membership id
     * @return group memberships information
     */
    @Processor
    public GroupMembership getGroupMembership(Long groupMembershipId) {
        return this.jerseyUtil.get(this.apiResource.path("group_memberships").path(groupMembershipId + ".json"), EntityResponse.class, 200).getGroupMembership();
    }

    /**
     * Get membership information
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/group_memberships.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-group-membership-by-user}
     *
     * @param groupMembershipId
     *            group membership
     * @param userId
     *            user
     * @return group memberships information
     */
    @Processor
    public GroupMembership getGroupMembershipByUser(Long groupMembershipId, Long userId) {
        return this.jerseyUtil.get(this.apiResource.path("users").path(userId.toString()).path("group_memberships").path(groupMembershipId + ".json"), EntityResponse.class, 200)
                .getGroupMembership();
    }

    /**
     * Create membership
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/group_memberships.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-group-membership}
     *
     * @param groupMembership
     *            group membership to be created
     * @return created group membership
     */
    @Processor
    public GroupMembership createGroupMembership(@Default("#[payload]") GroupMembership groupMembership) {
        return this.jerseyUtil.post(this.apiResource.path("group_memberships.json").entity(groupMembership), EntityResponse.class, 201).getGroupMembership();
    }

    /**
     * Delete membership
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/group_memberships.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:delete-group-membership}
     *
     * @param groupMembership
     *            group membership to be deleted
     */
    @Processor
    public void deleteGroupMembership(@Default("#[payload]") GroupMembership groupMembership) {
        this.jerseyUtil.delete(this.apiResource.path("group_memberships").path(groupMembership.getId() + ".json").entity(groupMembership), ClientResponse.class, 204);
    }

    /**
     * Set a group membership as default
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/group_memberships.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:set-group-membership-as-default}
     *
     * @param groupMembership
     *            group membership to be set as default
     * @return operation results
     */
    @Processor
    public Iterable<GroupMembership> setGroupMembershipAsDefault(@Default("#[payload]") GroupMembership groupMembership) {
        return createPaginationResults(
                EntityType.GROUP_MEMBERSHIP,
                this.jerseyUtil.put(this.apiResource.path("users").path(groupMembership.getUserId().toString()).path("group_memberships").path(groupMembership.getId().toString())
                        .path("make_default.json"), EntityResponse.class, 200));
    }

    /**
     * List organizations
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/organizations.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-organizations}
     *
     * @return operation results
     */
    @Processor
    public Iterable<Organization> listOrganizations() {
        return createPaginationResults(EntityType.ORGANIZATION, this.jerseyUtil.put(this.apiResource.path("organizations.json"), EntityResponse.class, 200));
    }

    /**
     * Get organization related information
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/organizations.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-organization-related}
     *
     * @param organizationId
     *            organization id
     * @return organization related information
     */
    @Processor
    public OrganizationRelated getOrganizationRelated(Long organizationId) {
        return this.jerseyUtil.get(this.apiResource.path("organizations").path(organizationId.toString()).path("related.json"), EntityResponse.class, 200).getOrganizationRelated();
    }

    /**
     * Get organization info
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/organizations.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-organization}
     *
     * @param organizationId
     *            organization id
     * @return organization information
     */
    @Processor
    public Organization getOrganization(Long organizationId) {
        return this.jerseyUtil.get(this.apiResource.path("organizations").path(organizationId + ".json"), EntityResponse.class, 200).getOrganization();
    }

    /**
     * Create organization
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/organizations.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-organization}
     *
     * @param organization
     *            organization to be created
     * @return new organization
     */
    @Processor
    public Organization createOrganization(@Default("#[payload]") Organization organization) {
        return this.jerseyUtil.post(this.apiResource.path("organizations.json").entity(organization), EntityResponse.class, 201).getOrganization();
    }

    /**
     * Update organization
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/organizations.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:update-organization}
     *
     * @param organization
     *            organization to be updated
     * @return updated organization
     */
    @Processor
    public Organization updateOrganization(@Default("#[payload]") Organization organization) {
        return this.jerseyUtil.put(this.apiResource.path("organizations").path(organization.getId() + ".json").accept(MediaType.APPLICATION_JSON).entity(organization), EntityResponse.class, 200).getOrganization();
    }

    /**
     * Update organization
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/organizations.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:delete-organization}
     *
     * @param organization
     *            organization to be updated
     */
    @Processor
    public void deleteOrganization(@Default("#[payload]") Organization organization) {
        this.jerseyUtil.delete(this.apiResource.path("organizations").path(organization.getId() + ".json"), EntityResponse.class, 204);
    }

    /**
     * Search organization
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/organizations.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:search-organization}
     *
     * @param searchTerm
     *            search term
     * @return list of organizations
     */
    @Processor
    public Iterable<Organization> searchOrganization(String searchTerm) {
        return createPaginationResults(EntityType.ORGANIZATION,
                this.jerseyUtil.get(this.apiResource.path("organizations").path("search.json").queryParam("external_id", searchTerm), EntityResponse.class, 200));
    }

    /**
     * List forums
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/forums.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-forums}
     *
     * @return list of forums
     */
    @Processor
    public Iterable<Forum> listForums() {
        return createPaginationResults(EntityType.FORUM, this.jerseyUtil.get(this.apiResource.path("forums.json"), EntityResponse.class, 200));
    }

    /**
     * List forums by category
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/forums.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-forums-by-category}
     *
     * @param categoryId
     *            category id
     * @return list of forums
     */
    @Processor
    public Iterable<Forum> listForumsByCategory(Long categoryId) {
        return createPaginationResults(EntityType.FORUM,
                this.jerseyUtil.get(this.apiResource.path("categories").path(categoryId.toString()).path("forums.json"), EntityResponse.class, 200));
    }

    /**
     * Get Forum
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/forums.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-forum}
     *
     * @param forumId
     *            forum id
     * @return list of forums
     */
    @Processor
    public Forum getForum(Long forumId) {
        return this.jerseyUtil.get(this.apiResource.path("forums").path(forumId + ".json"), EntityResponse.class, 200).getForum();
    }

    /**
     * Create Forum
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/forums.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-forum}
     *
     * @param forum
     *            forum to be created
     * @return created forum
     */
    @Processor
    public Forum createForum(@Default("#[payload]") Forum forum) {
        return this.jerseyUtil.post(this.apiResource.path("forums.json").entity(forum), EntityResponse.class, 201).getForum();
    }

    /**
     * Update Forum
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/forums.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:update-forum}
     *
     * @param forum
     *            forum to be updated
     * @return updated forum
     */
    @Processor
    public Forum updateForum(@Default("#[payload]") Forum forum) {
        return this.jerseyUtil.put(this.apiResource.path("forums").path(forum.getId() + ".json").accept(MediaType.APPLICATION_JSON).entity(forum), EntityResponse.class, 200).getForum();
    }

    /**
     * Delete Forum
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/forums.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:delete-forum}
     *
     * @param forum
     *            forum to be deleted
     */
    @Processor
    public void deleteForum(@Default("#[payload]") Forum forum) {
        this.jerseyUtil.delete(this.apiResource.path("forums").path(forum.getId() + ".json"), EntityResponse.class, 204);
    }

    /**
     * List forum subscriptions
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/forum_subscriptions.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-forum-subscriptions}
     *
     * @return list of forum subscriptions
     */
    @Processor
    public Iterable<ForumSubscription> listForumSubscriptions() {
        return createPaginationResults(EntityType.FORUM_SUBSCRIPTION, this.jerseyUtil.get(this.apiResource.path("forum_subscriptions.json"), EntityResponse.class, 200));
    }

    /**
     * List forum subscriptions by forum
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/forum_subscriptions.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-forum-subscriptions-by-forum}
     *
     * @param forumId
     *            forum id
     * @return list of forum subscriptions
     */
    @Processor
    public Iterable<ForumSubscription> listForumSubscriptionsByForum(Long forumId) {
        return createPaginationResults(EntityType.FORUM_SUBSCRIPTION,
                this.jerseyUtil.get(this.apiResource.path("forums").path(forumId.toString()).path("subscriptions.json"), EntityResponse.class, 200));
    }

    /**
     * Get forum subscription
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/forum_subscriptions.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-forum-subscription}
     *
     * @param forumSubscriptionId
     *            forum id
     * @return list of forum subscriptions
     */
    @Processor
    public ForumSubscription getForumSubscription(Long forumSubscriptionId) {
        return this.jerseyUtil.get(this.apiResource.path("forum_subscriptions").path(forumSubscriptionId + ".json"), EntityResponse.class, 200).getForumSubscription();
    }

    /**
     * Create forum subscription
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/forum_subscriptions.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-forum-subscription}
     *
     * @param forumSubscription
     *            forum subscription to be created
     * @return new forum subscription
     */
    @Processor
    public ForumSubscription createForumSubscription(@Default("#[payload]") ForumSubscription forumSubscription) {
        return this.jerseyUtil.post(this.apiResource.path("forum_subscriptions.json").entity(forumSubscription), EntityResponse.class, 201).getForumSubscription();
    }

    /**
     * Delete forum subscription
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/forum_subscriptions.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:delete-forum-subscription}
     *
     * @param forumSubscription
     *            forum subscription to be deleted
     */
    @Processor
    public void deleteForumSubscription(@Default("#[payload]") ForumSubscription forumSubscription) {
        this.jerseyUtil.delete(this.apiResource.path("forum_subscriptions").path(forumSubscription.getId() + ".json"), EntityResponse.class, 204);
    }

    /**
     * List categories
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/categories.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-categories}
     *
     * @return list of categories
     */
    @Processor
    public Iterable<Category> listCategories() {
        return createPaginationResults(EntityType.CATEGORY, this.jerseyUtil.get(this.apiResource.path("categories.json"), EntityResponse.class, 200));
    }

    /**
     * Get category
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/categories.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-category}
     *
     * @param categoryId
     *            category id
     * @return category information
     */
    @Processor
    public Category getCategory(Long categoryId) {
        return this.jerseyUtil.get(this.apiResource.path("categories").path(categoryId + ".json"), EntityResponse.class, 200).getCategory();
    }

    /**
     * Create category
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/categories.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-category}
     *
     * @param category
     *            category to be created
     * @return new user information
     */
    @Processor
    public Category createCategory(@Default("#[payload]") Category category) {
        return this.jerseyUtil.post(this.apiResource.path("categories.json").entity(category), EntityResponse.class, 201).getCategory();
    }

    /**
     * Update category
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/categories.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:update-category}
     *
     * @param category
     *            category to be updated
     * @return updated category information
     */
    @Processor
    public Category updateCategory(@Default("#[payload]") Category category) {
        return this.jerseyUtil.put(this.apiResource.path("categories").path(category.getId() + ".json").accept(MediaType.APPLICATION_JSON).entity(category), EntityResponse.class, 200).getCategory();
    }

    /**
     * Delete category
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/categories.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:delete-category}
     *
     * @param category
     *            category to be deleted
     */
    @Processor
    public void deleteCategory(@Default("#[payload]") Category category) {
        this.jerseyUtil.delete(this.apiResource.path("categories").path(category.getId() + ".json"), EntityResponse.class, 204);
    }

    /**
     * List topics
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topics.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-topics}
     *
     * @return list of topics
     */
    @Processor
    public Iterable<Topic> listTopics() {
        return createPaginationResults(EntityType.TOPIC, this.jerseyUtil.get(this.apiResource.path("topics.json"), EntityResponse.class, 200));
    }

    /**
     * List topics by forum
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topics.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-topics-by-forum}
     *
     * @param forumId
     *            forum Id
     * @return list of topics
     */
    @Processor
    public Iterable<Topic> listTopicsByForum(Long forumId) {
        return createPaginationResults(EntityType.TOPIC,
                this.jerseyUtil.get(this.apiResource.path("forums").path(forumId.toString()).path("topics.json"), EntityResponse.class, 200));
    }

    /**
     * List topics by user
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topics.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-topics-by-user}
     *
     * @param userId
     *            user Id
     * @return list of topics
     */
    @Processor
    public Iterable<Topic> listTopicsByUser(Long userId) {
        return createPaginationResults(EntityType.TOPIC, this.jerseyUtil.get(this.apiResource.path("users").path(userId.toString()).path("topics.json"), EntityResponse.class, 200));
    }

    /**
     * Get topic
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topics.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-topic}
     *
     * @param topicId
     *            topic id
     * @return topic info
     */
    @Processor
    public Topic getTopic(Long topicId) {
        return this.jerseyUtil.get(this.apiResource.path("topics").path(topicId + ".json"), EntityResponse.class, 200).getTopic();
    }

    /**
     * Create topic
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topics.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-topic}
     *
     * @param topic
     *            topic to be created
     * @return new topic
     */
    @Processor
    public Topic createTopic(@Default("#[payload]") Topic topic) {
        return this.jerseyUtil.post(this.apiResource.path("topics.json").entity(topic), EntityResponse.class, 201).getTopic();
    }

    /**
     * Update topic
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topics.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:update-topic}
     *
     * @param topic
     *            topic to be updated
     * @return update topic
     */
    @Processor
    public Topic updateTopic(@Default("#[payload]") Topic topic) {
        return this.jerseyUtil.put(this.apiResource.path("topics").path(topic.getId() + ".json").accept(MediaType.APPLICATION_JSON).entity(topic), EntityResponse.class, 200).getTopic();
    }

    /**
     * Creates a topic without sending out notifications. Allows setting created_at and updated_at.
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topics.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:import-topic}
     *
     * @param topic
     *            topic to be created
     * @return new topic
     */
    @Processor
    public Topic importTopic(@Default("#[payload]") Topic topic) {
        return this.jerseyUtil.post(this.apiResource.path("import").path("topics.json").entity(topic), EntityResponse.class, 201).getTopic();
    }

    /**
     * Delete topic
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topics.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:delete-topic}
     *
     * @param topic
     *            topic to be deleted
     */
    @Processor
    public void deleteTopic(@Default("#[payload]") Topic topic) {
        this.jerseyUtil.delete(this.apiResource.path("topics").path(topic.getId() + ".json"), EntityResponse.class, 204);
    }

    /**
     * List topic comments
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topic_comments.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-topic-comments}
     *
     * @param topicId
     *            topic id
     * @return list of topic comments
     */
    @Processor
    public Iterable<TopicComment> listTopicComments(Long topicId) {
        return createPaginationResults(EntityType.TOPIC_COMMENT,
                this.jerseyUtil.get(this.apiResource.path("topics").path(topicId.toString()).path("comments.json"), EntityResponse.class, 200));
    }

    /**
     * List topic comments by user
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topic_comments.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-topic-comments-by-user}
     *
     * @param userId
     *            user id
     * @return list of topic comments
     */
    @Processor
    public Iterable<TopicComment> listTopicCommentsByUser(Long userId) {
        return createPaginationResults(EntityType.TOPIC_COMMENT,
                this.jerseyUtil.get(this.apiResource.path("users").path(userId.toString()).path("topic_comments.json"), EntityResponse.class, 200));
    }

    /**
     * Get topic comment
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topic_comments.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-topic-comment}
     *
     * @param topicId
     *            topic id
     * @param topicCommentId
     *            topic comment id
     * @return topic comment information
     */
    @Processor
    public TopicComment getTopicComment(Long topicId, Long topicCommentId) {
        return this.jerseyUtil.get(this.apiResource.path("topics").path(topicId.toString()).path("comments").path(topicCommentId + ".json"), EntityResponse.class, 200)
                .getTopicComment();
    }

    /**
     * Create Topic comment
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topic_comments.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-topic-comment}
     *
     * @param topicComment
     *            topic comment to be created
     * @return created topic comment
     */
    @Processor
    public TopicComment createTopicComment(@Default("#[payload]") TopicComment topicComment) {
        return this.jerseyUtil.post(this.apiResource.path("topics").path(topicComment.getTopicId().toString()).path("comments.json").entity(topicComment), EntityResponse.class,
                201).getTopicComment();
    }

    /**
     * Creates a comment without sending out notifications. Allows setting created_at and updated_at.
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topic_comments.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:import-topic-comment}
     *
     * @param topicComment
     *            topic comment to be created
     * @return created topic comment
     */
    @Processor
    public TopicComment importTopicComment(@Default("#[payload]") TopicComment topicComment) {
        return this.jerseyUtil.post(this.apiResource.path("import").path("topics").path(topicComment.getTopicId().toString()).path("comments.json").entity(topicComment),
                EntityResponse.class, 201).getTopicComment();
    }

    /**
     * Update topic comment
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topic_comments.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:update-topic-comment}
     *
     * @param topicComment
     *            topic comment to be updated
     * @return updated topic comment
     */
    @Processor
    public TopicComment updateTopicComment(@Default("#[payload]") TopicComment topicComment) {
        return this.jerseyUtil.put(
                this.apiResource.path("topics").path(topicComment.getTopicId().toString()).path("comments").path(topicComment.getId() + ".json").accept(MediaType.APPLICATION_JSON).entity(topicComment),
                EntityResponse.class, 200).getTopicComment();
    }

    /**
     * Delete topic comment
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topic_comments.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:delete-topic-comment}
     *
     * @param topicComment
     *            topic comment to be deleted
     */
    @Processor
    public void deleteTopicComment(@Default("#[payload]") TopicComment topicComment) {
        this.jerseyUtil.delete(this.apiResource.path("topics").path(topicComment.getTopicId().toString()).path("comments").path(topicComment.getId() + ".json"), EntityResponse.class, 204);
    }

    /**
     * List topic subscriptions
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topic_subscriptions.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-topic-subscriptions}
     *
     * @return list of topic subscriptions
     */
    @Processor
    public Iterable<TopicSubscription> listTopicSubscriptions() {
        return createPaginationResults(EntityType.TOPIC_SUBSCRIPTION, this.jerseyUtil.get(this.apiResource.path("topic_subscriptions.json"), EntityResponse.class, 200));
    }

    /**
     * List topic subscriptions by topic
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topic_subscriptions.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-topic-subscriptions-by-topic}
     *
     * @param topicId
     *            topic id
     * @return list of topic subscriptions
     */
    @Processor
    public Iterable<TopicSubscription> listTopicSubscriptionsByTopic(Long topicId) {
        return createPaginationResults(EntityType.TOPIC_SUBSCRIPTION,
                this.jerseyUtil.get(this.apiResource.path("topics").path(topicId.toString()).path("subscriptions.json"), EntityResponse.class, 200));
    }

    /**
     * Get topic subscription
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topic_subscriptions.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-topic-subscription}
     *
     * @param topicSubscriptionId
     *            topic comment id
     * @return topic comment information
     */
    @Processor
    public TopicSubscription getTopicSubscription(Long topicSubscriptionId) {
        return this.jerseyUtil.get(this.apiResource.path("topic_subscriptions").path(topicSubscriptionId + ".json"), EntityResponse.class, 200).getTopicSubscription();
    }

    /**
     * Create Topic subscription
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topic_subscriptions.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-topic-subscription}
     *
     * @param topicSubscription
     *            topic subscription to be created
     * @return created topic subscription
     */
    @Processor
    public TopicSubscription createTopicSubscription(@Default("#[payload]") TopicSubscription topicSubscription) {
        return this.jerseyUtil.post(this.apiResource.path("topic_subscriptions.json").entity(topicSubscription), EntityResponse.class, 201).getTopicSubscription();
    }

    /**
     * Delete topic subscription
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topic_subscriptions.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:delete-topic-subscription}
     *
     * @param topicSubscription
     *            topic subscription to be deleted
     */
    @Processor
    public void deleteTopicSubscription(@Default("#[payload]") TopicSubscription topicSubscription) {
        this.jerseyUtil.delete(this.apiResource.path("topic_subscriptions").path(topicSubscription.getId() + ".json"), EntityResponse.class, 204);
    }

    /**
     * List topic votes
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topic_votes.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-topic-votes}
     *
     * @param topicId
     *            topic id
     * @return list of topic votes
     */
    @Processor
    public Iterable<TopicVote> listTopicVotes(Long topicId) {
        return createPaginationResults(EntityType.TOPIC_VOTE,
                this.jerseyUtil.get(this.apiResource.path("topics").path(topicId.toString()).path("votes.json"), EntityResponse.class, 200));
    }

    /**
     * List topic votes by user
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topic_votes.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-topic-votes-by-user}
     *
     * @param userId
     *            user id
     * @return list of topic votes
     */
    @Processor
    public Iterable<TopicVote> listTopicVotesByUser(Long userId) {
        return createPaginationResults(EntityType.TOPIC_VOTE,
                this.jerseyUtil.get(this.apiResource.path("users").path(userId.toString()).path("topic_votes.json"), EntityResponse.class, 200));
    }

    /**
     * Check topic for vote
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topic_votes.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:check-for-vote}
     *
     * @param topicId
     *            topic id
     * @return topic vote information
     */
    @Processor
    public TopicComment checkForVote(Long topicId) {
        return this.jerseyUtil.get(this.apiResource.path("topics").path(topicId.toString()).path("vote.json"), EntityResponse.class, 200).getTopicComment();
    }

    /**
     * Create Topic vote
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topic_votes.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-topic-vote}
     *
     * @param topicVote
     *            topic vote to be created
     * @return created topic vote
     */
    @Processor
    public TopicVote createTopicVote(@Default("#[payload]") TopicVote topicVote) {
        return this.jerseyUtil.post(this.apiResource.path("topics").path(topicVote.getTopicId().toString()).path("vote.json").entity(topicVote), EntityResponse.class, 201)
                .getTopicVote();
    }

    /**
     * Delete topic vote
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/topic_votes.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:delete-topic-vote}
     *
     * @param topicVote
     *            topic vote to be deleted
     */
    @Processor
    public void deleteTopicVote(@Default("#[payload]") TopicVote topicVote) {
        this.jerseyUtil.delete(this.apiResource.path("topics").path(topicVote.getTopicId().toString()).path("vote.json"), EntityResponse.class, 204);
    }

    /**
     * List activities
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/activity_stream.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-activities}
     *
     * @return list of activities
     */
    @Processor
    public Iterable<ActivityStream> listActivities() {
        return createPaginationResults(EntityType.ACTIVITY_STREAM, this.jerseyUtil.get(this.apiResource.path("activities.json"), EntityResponse.class, 200));
    }

    /**
     * Get activity
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/activity_stream.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-activity}
     *
     * @param activityId
     *            activity id
     * @return activity information
     */
    @Processor
    public ActivityStream getActivity(Long activityId) {
        return this.jerseyUtil.get(this.apiResource.path("activities").path(activityId + ".json"), EntityResponse.class, 200).getActivity();
    }

    /**
     * Upload File attachment
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/attachments.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:upload-attachment}
     *
     * @param content
     *            file to upload
     * @param mediaType
     *            file media type. E.g.: application/octet-stream
     * @param filename
     *            filename
     * @param token
     *            optional token
     * @return upload information
     */
    @Processor
    public FileUpload uploadAttachment(InputStream content, String mediaType, String filename, @Optional String token) {
        return this.jerseyUtil.post(this.apiResource.path("uploads.json").queryParam("filename", filename).entity(content, mediaType).accept(MediaType.APPLICATION_JSON), EntityResponse.class, 201).getUpload();
    }

    /**
     * Delete upload
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/attachments.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:delete-upload}
     *
     * @param token
     *            token for the attachment
     */
    @Processor
    public void deleteUpload(String token) {
        this.jerseyUtil.delete(this.apiResource.path("uploads").path(token + ".json"), EntityResponse.class, 204);
    }

    /**
     * Get attachment
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/attachments.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-attachment}
     *
     * @param attachmentId
     *            attachment id
     * @return attachment information
     */
    @Processor
    public Attachment getAttachment(Long attachmentId) {
        return this.jerseyUtil.get(this.apiResource.path("attachments").path(attachmentId + ".json"), EntityResponse.class, 200).getAttachment();
    }

    /**
     * Delete attachment
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/attachments.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:delete-attachment}
     *
     * @param attachmentId
     *            attachment id
     */
    @Processor
    public void deleteAttachment(Long attachmentId) {
        this.jerseyUtil.delete(this.apiResource.path("attachments").path(attachmentId + ".json"), EntityResponse.class, 204);
    }

    /**
     * List automations
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/automations.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-automations}
     *
     * @return list of automations
     */
    @Processor
    public Iterable<Automation> listAutomations() {
        return createPaginationResults(EntityType.AUTOMATION, this.jerseyUtil.get(this.apiResource.path("automations.json"), EntityResponse.class, 200));
    }

    /**
     * Get automation
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/automations.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-automation}
     *
     * @param automationId
     *            automation id
     * @return automation info
     */
    @Processor
    public Automation getAutomation(Long automationId) {
        return this.jerseyUtil.get(this.apiResource.path("automations").path(automationId + ".json"), EntityResponse.class, 200).getAutomation();
    }

    /**
     * List active automations
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/automations.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-active-automations}
     *
     * @return list of active automations
     */
    @Processor
    public Iterable<Automation> listActiveAutomations() {
        return createPaginationResults(EntityType.AUTOMATION, this.jerseyUtil.get(this.apiResource.path("automations").path("active.json"), EntityResponse.class, 200));
    }

    /**
     * Create automation
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/automations.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-automation}
     *
     * @param automation
     *            automation to be created
     * @return created automation info
     */
    @Processor
    public Automation createAutomation(@Default("#[payload]") Automation automation) {
        return this.jerseyUtil.post(this.apiResource.path("automations.json").entity(automation), EntityResponse.class, 201).getAutomation();
    }

    /**
     * Update automation
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/automations.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:update-automation}
     *
     * @param automation
     *            automation to be updated
     * @return updated automation info
     */
    @Processor
    public Automation updateAutomation(@Default("#[payload]") Automation automation) {
        return this.jerseyUtil.put(this.apiResource.path("automations").path(automation.getId() + ".json").accept(MediaType.APPLICATION_JSON).entity(automation), EntityResponse.class, 200).getAutomation();
    }

    /**
     * Show Job Status
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/job_statuses.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:show-job-status}
     *
     * @param jobStatusId
     *            job status id
     * @return job status info
     */
    @Processor
    public JobStatus showJobStatus(Long jobStatusId) {
        return this.jerseyUtil.get(this.apiResource.path("job_statuses").path(jobStatusId + ".json"), EntityResponse.class, 200).getJobStatus();
    }

    /**
     * List locales
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/locales.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-locales}
     *
     * @return locale list
     */
    @Processor
    public Iterable<Locale> listLocales() {
        return createPaginationResults(EntityType.LOCALE, this.jerseyUtil.get(this.apiResource.path("locales.json"), EntityResponse.class, 200));
    }

    /**
     * Get locale
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/locales.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-locale}
     *
     * @param localeId
     *            locale id
     * @return locale information
     */
    @Processor
    public Locale getLocale(Integer localeId) {
        return this.jerseyUtil.get(this.apiResource.path("locales").path(localeId + ".json"), EntityResponse.class, 200).getLocale();
    }

    /**
     * Get Current locale
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/locales.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-current-locale}
     *
     * @return current locale information
     */
    @Processor
    public Locale getCurrentLocale() {
        return this.jerseyUtil.get(this.apiResource.path("locales").path("current.json"), EntityResponse.class, 200).getLocale();
    }

    /**
     * List macros
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/macros.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-macros}
     *
     * @return macros list
     */
    @Processor
    public Iterable<Macro> listMacros() {
        return createPaginationResults(EntityType.MACRO, this.jerseyUtil.get(this.apiResource.path("macros.json"), EntityResponse.class, 200));
    }

    /**
     * List active macros
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/macros.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-active-macros}
     *
     * @return active macros list
     */
    @Processor
    public Iterable<Macro> listActiveMacros() {
        return createPaginationResults(EntityType.MACRO, this.jerseyUtil.get(this.apiResource.path("macros").path("active.json"), EntityResponse.class, 200));
    }

    /**
     * Get macro
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/macros.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-macro}
     *
     * @param macroId
     *            macro id
     * @return macros information
     */
    @Processor
    public Macro getMacro(Long macroId) {
        return this.jerseyUtil.get(this.apiResource.path("macros").path(macroId + ".json"), EntityResponse.class, 200).getMacro();
    }

    /**
     * Create new macro
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/macros.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-macro}
     *
     * @param macro
     *            macro to be created
     * @return new macro info
     */
    @Processor
    public Macro createMacro(@Default("#[payload]") Macro macro) {
        return this.jerseyUtil.post(this.apiResource.path("macros.json").entity(macro), EntityResponse.class, 201).getMacro();
    }

    /**
     * Update macro
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/macros.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:update-macro}
     *
     * @param macro
     *            macro to be updated
     * @return updated macro info
     */
    @Processor
    public Macro updateMacro(@Default("#[payload]") Macro macro) {
        return this.jerseyUtil.put(this.apiResource.path("macros").path(macro.getId() + ".json").accept(MediaType.APPLICATION_JSON).entity(macro), EntityResponse.class, 200).getMacro();
    }

    /**
     * Delete macro
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/macros.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:delete-macro}
     *
     * @param macro
     *            macro to be deleted
     */
    @Processor
    public void deleteMacro(@Default("#[payload]") Macro macro) {
        this.jerseyUtil.delete(this.apiResource.path("macros").path(macro.getId() + ".json").entity(macro), EntityResponse.class, 200);
    }

    /**
     * Apply macro
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/macros.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:apply-macro}
     *
     * @param macroId
     *            macro to be applied
     */
    @Processor
    public void applyMacro(Long macroId) {
        this.jerseyUtil.get(this.apiResource.path("macros").path(macroId.toString()).path("apply.json"), ApplyMacroResponse.class, 200);
    }

    /**
     * Apply macro to an specific ticket
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/macros.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:apply-macro-to-ticket}
     *
     * @param macroId
     *            macro to be applied
     * @param ticketId
     *            ticket to apply the macro
     */
    @Processor
    public void applyMacroToTicket(Long macroId, Long ticketId) {
        this.jerseyUtil.get(this.apiResource.path("tickets").path(ticketId.toString()).path("macros").path(macroId.toString()).path("apply.json"), ApplyMacroResponse.class, 200);
    }

    /**
     * List satisfaction ratings
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/satisfaction_ratings.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-satisfaction-ratings}
     *
     * @return satisfaction ratings list
     */
    @Processor
    public Iterable<SatisfactionRating> listSatisfactionRatings() {
        return createPaginationResults(EntityType.SATISFACTION_RATING, this.jerseyUtil.get(this.apiResource.path("satisfaction_ratings.json"), EntityResponse.class, 200));
    }

    /**
     * List received satisfaction ratings
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/satisfaction_ratings.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-received-satisfaction-ratings}
     *
     * @return satisfaction ratings list
     */
    @Processor
    public Iterable<SatisfactionRating> listReceivedSatisfactionRatings() {
        return createPaginationResults(EntityType.SATISFACTION_RATING,
                this.jerseyUtil.get(this.apiResource.path("satisfaction_ratings").path("received.json"), EntityResponse.class, 200));
    }

    /**
     * Get satisfaction rating info
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/satisfaction_ratings.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-satisfaction-rating}
     *
     * @param satisfactionRatingId
     *            satisfaction rating id
     * @return satisfaction ratings list
     */
    @Processor
    public SatisfactionRating getSatisfactionRating(Long satisfactionRatingId) {
        return this.jerseyUtil.get(this.apiResource.path("satisfaction_ratings").path(satisfactionRatingId + ".json"), EntityResponse.class, 200).getSatisfactionRating();
    }

    /**
     * Create satisfaction rating
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/satisfaction_ratings.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-satisfaction-rating}
     *
     * @param satisfactionRating
     *            satisfaction rating to be created
     * @return satisfaction ratings info
     */
    @Processor
    public SatisfactionRating createSatisfactionRating(@Default("#[payload]") SatisfactionRating satisfactionRating) {
        return this.jerseyUtil.post(this.apiResource.path("tickets").path(satisfactionRating.getTicketId().toString()).path("satisfaction_rating.json").entity(satisfactionRating),
                EntityResponse.class, 201).getSatisfactionRating();
    }

    /**
     * List sharing agreements
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/sharing_agreements.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-sharing-agreements}
     *
     * @return sharing agreements list
     */
    @Processor
    public Iterable<SharingAgreement> listSharingAgreements() {
        return createPaginationResults(EntityType.SHARING_AGREEMENT, this.jerseyUtil.get(this.apiResource.path("sharing_agreements.json"), EntityResponse.class, 200));
    }

    /**
     * List suspended tickets
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/suspended_tickets.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-suspended-tickets}
     *
     * @return suspended tickets list
     */
    @Processor
    public Iterable<SuspendedTicket> listSuspendedTickets() {
        return createPaginationResults(EntityType.SUSPENDED_TICKET, this.jerseyUtil.get(this.apiResource.path("suspended_tickets.json"), EntityResponse.class, 200));
    }

    /**
     * Get suspended ticket
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/suspended_tickets.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-suspended-ticket}
     *
     * @param suspendedTicketId
     *            suspended ticket id
     * @return suspended ticket info
     */
    @Processor
    public SuspendedTicket getSuspendedTicket(Long suspendedTicketId) {
        return this.jerseyUtil.get(this.apiResource.path("suspended_tickets").path(suspendedTicketId + ".json"), EntityResponse.class, 200).getSuspendedTicket();
    }

    /**
     * Recover a suspended ticket
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/suspended_tickets.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:recover-suspended-ticket}
     *
     * @param suspendedTicketId
     *            suspended ticket id
     */
    @Processor
    public void recoverSuspendedTicket(Long suspendedTicketId) {
        this.jerseyUtil.put(this.apiResource.path("suspended_tickets").path(suspendedTicketId.toString()).path("recover.json"), EntityResponse.class, 200);
    }

    /**
     * Delete a suspended ticket
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/suspended_tickets.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:delete-suspended-ticket}
     *
     * @param suspendedTicketId
     *            suspended ticket id
     */
    @Processor
    public void deleteSuspendedTicket(Long suspendedTicketId) {
        this.jerseyUtil.delete(this.apiResource.path("suspended_tickets").path(suspendedTicketId + ".json"), EntityResponse.class, 204);
    }

    /**
     * List triggers
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/triggers.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-triggers}
     *
     * @return list of triggers
     */
    @Processor
    public Iterable<Trigger> listTriggers() {
        return createPaginationResults(EntityType.TRIGGER, this.jerseyUtil.get(this.apiResource.path("triggers.json"), EntityResponse.class, 200));
    }

    /**
     * Get trigger
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/triggers.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:get-trigger}
     *
     * @param triggerId
     *            trigger id
     * @return trigger information
     */
    @Processor
    public Trigger getTrigger(Long triggerId) {
        return this.jerseyUtil.get(this.apiResource.path("triggers").path(triggerId + ".json"), EntityResponse.class, 200).getTrigger();
    }

    /**
     * List active triggers
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/triggers.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:list-active-triggers}
     *
     * @return list of active triggers
     */
    @Processor
    public Iterable<Trigger> listActiveTriggers() {
        return createPaginationResults(EntityType.TRIGGER, this.jerseyUtil.get(this.apiResource.path("triggers").path("active.json"), EntityResponse.class, 200));
    }

    /**
     * Create trigger
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/triggers.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:create-trigger}
     *
     * @param trigger
     *            trigger to be created
     * @return new trigger
     */
    @Processor
    public Trigger createTrigger(@Default("#[payload]") Trigger trigger) {
        return this.jerseyUtil.post(this.apiResource.path("triggers.json").entity(trigger), EntityResponse.class, 201).getTrigger();
    }

    /**
     * Update trigger
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/triggers.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:update-trigger}
     *
     * @param trigger
     *            trigger to be updated
     * @return updated trigger
     */
    @Processor
    public Trigger updateTrigger(@Default("#[payload]") Trigger trigger) {
        return this.jerseyUtil.put(this.apiResource.path("triggers").path(trigger.getId() + ".json").accept(MediaType.APPLICATION_JSON).entity(trigger), EntityResponse.class, 200).getTrigger();
    }

    /**
     * The search API is a unified search API that returns tickets, users, organizations, and forum topics. Define filters to narrow your search results according to result type,
     * date attributes, and object attributes such as ticket requester or tag.
     * <p>
     * <b>Important: This operation only returns one type of object at the time.</b> If you query more than one type you'll get a ZendeskException
     * </p>
     * <p>
     * <a href="http://developer.zendesk.com/documentation/rest_api/search.html">Check Zendesk API documentation</a>
     * </p>
     * <p/>
     * {@sample.xml ../../../doc/Zendesk-connector.xml.sample zendesk:search}
     *
     * @param type
     *            Entity type. It has to be the same as the entity queried
     * @param query
     *            string
     * @param sortBy
     *            Possible values are 'updated_at', 'created_at', 'priority', 'status', and 'ticket_type'
     * @param sortOrder
     *            One of 'relevance', 'asc', 'desc'. Defaults to 'relevance' when no 'order' criteria is requested.
     * @return iterable with search results
     */
    @Processor
    public Iterable search(EntityType type, String query, @Optional String sortBy, @Optional String sortOrder) {
        WebResource searchResource = this.apiResource.path("search.json").queryParam("query", query);

        if (sortBy != null) {
            searchResource.queryParam("sort_by", sortBy);
        }

        if (sortOrder != null) {
            searchResource.queryParam("sort_order", sortOrder);
        }

        return createPaginationResults(EntityType.SEARCH_RESULTS, this.jerseyUtil.get(searchResource, SearchResponse.class, 200));
    }

    public String getApiUrl() {
        return apiUrl;
    }

    public void setApiUrl(String apiUrl) {
        this.apiUrl = apiUrl;
    }

    public JerseyUtil getJerseyUtil() {
        return jerseyUtil;
    }

    public void setJerseyUtil(JerseyUtil jerseyUtil) {
        this.jerseyUtil = jerseyUtil;
    }

    public WebResource getApiResource() {
        return apiResource;
    }

    public void setApiResource(WebResource apiResource) {
        this.apiResource = apiResource;
    }

    public Client getClient() {
        return client;
    }

    public void setClient(Client client) {
        this.client = client;
    }

    @SuppressWarnings("unchecked")
    private <T> Iterable<T> createPaginationResults(final EntityType type, final EntityResponse paginatedResponse) {

        return new PaginatedIterable<T, EntityResponse>() {

            private EntityResponse first;

            @Override
            protected synchronized EntityResponse firstPage() {
                if (first == null) {
                    first = paginatedResponse;
                }
                return first;
            }

            @Override
            protected boolean hasNextPage(final EntityResponse currentPage) {
                return currentPage.getNextPage() != null;
            }

            @Override
            protected EntityResponse nextPage(final EntityResponse page) {
            	try {
					URI nextPageURI = new URI(page.getNextPage());
					return jerseyUtil.get(apiResource.uri(nextPageURI), EntityResponse.class, 200);
				} catch (URISyntaxException e) {
					throw new ZendeskException("Error attempting to get nextPage", e);
				}
            }

            @Override
            protected Iterator<T> pageIterator(final EntityResponse page) {
                try {
                    return ((List<T>) page.getClass().getMethod("get" + type.getNameForLists()).invoke(page)).iterator();
                } catch (Exception e) {
                    throw new ZendeskException(e);
                }
            }

            public boolean isEmpty() {
                return (this.size() == 0);
            }

            public int size() {
                return first.getCount();
            }
        };
    }
}
