/**
 * (c) 2003-2014 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */

package org.mule.modules.workday.professionalservices;

import org.mule.api.ConnectionException;
import org.mule.api.annotations.Connect;
import org.mule.api.annotations.ConnectionIdentifier;
import org.mule.api.annotations.ConnectivityTesting;
import org.mule.api.annotations.Connector;
import org.mule.api.annotations.Disconnect;
import org.mule.api.annotations.InvalidateConnectionOn;
import org.mule.api.annotations.Processor;
import org.mule.api.annotations.ValidateConnection;
import org.mule.api.annotations.display.FriendlyName;
import org.mule.api.annotations.param.ConnectionKey;
import org.mule.api.annotations.param.Default;
import org.mule.api.annotations.param.Optional;
import org.mule.api.annotations.param.RefOnly;
import org.mule.modules.workday.AbstractWorkdayModule;
import org.mule.modules.workday.api.exception.WorkdayCredentialsException;
import org.mule.modules.workday.api.exception.WorkdayException;

import com.workday.professionalservices.ExpenseReportAddUpdateType;
import com.workday.professionalservices.ExpenseReportCancelType;
import com.workday.professionalservices.ProfessionalServicesAutomationPort;

/**
 * The Professional Services Automation Web Service contains operations that expose Workday Financials Business Services data for integration with Professional Services Automation (PSA) systems.
 *
 * @author MuleSoft, Inc.
 */
@Connector(name = "wd-professional", schemaVersion = "2.0", friendlyName = "Workday Professional Services Automation", minMuleVersion = "3.4", connectivityTesting= ConnectivityTesting.DISABLED)
public class ProfessionalModule extends AbstractWorkdayModule {


    protected ProfessionalServicesAutomationPort client;
    protected String username;

    /**
     * Connects to Workday
     *
     * @param professionalServicesAutomationUser Workday user
     * @param professionalServicesAutomationPassword Workday password
     * @param professionalServicesAutomationEndpoint Workday endpoint
     * @param professionalServicesAutomationWsdlLocation Workday WSDL location
     * @throws org.mule.api.ConnectionException
     */
    @Connect
    public void connect(@FriendlyName("user") @ConnectionKey String professionalServicesAutomationUser,
                        @FriendlyName("password") String professionalServicesAutomationPassword,
                        @FriendlyName("endpoint") String professionalServicesAutomationEndpoint,
                        @FriendlyName("wsdl") @Optional @Default("https://community.workday.com/custom/developer/API/Professional_Services_Automation/v23.0/Professional_Services_Automation.wsdl") String professionalServicesAutomationWsdlLocation)
            throws ConnectionException {
        if (client == null)
        {
            setUsername(professionalServicesAutomationUser);
            CxfProfessionalServicesClient cxfProfessionalServicesClient = new CxfProfessionalServicesClient(professionalServicesAutomationUser, professionalServicesAutomationPassword,
                                                                            professionalServicesAutomationEndpoint, professionalServicesAutomationWsdlLocation);
            initClient(cxfProfessionalServicesClient, cxfProfessionalServicesClient.getServiceVersion());
        }
    }

    /**
     * Disconnects
     */
    @Disconnect
    public void disconnect() {
        setClient(null);
    }

    public ProfessionalServicesAutomationPort getClient() {
        return client;
    }

    /**
     * Connection identifier
     * @return connection identifier
     */
    @ConnectionIdentifier
    public String getConnectionIdentifier() {
        return "User: " + getUsername();
    }

    public String getUsername() {
        return username;
    }

    @Override
    protected void setClient(Object client) {
        this.client = (ProfessionalServicesAutomationPort) client;
    }

    public void setUsername(String username)
    {
        this.username = username;
    }

    /**
     * Validate connection
     * @return true if the connector is connected
     */
    @ValidateConnection
    public boolean validateConnection() {
        return getClient() != null;
    }

    /**
     * This web service operation will cancel an Expense Report.
     *
     * {@sample.xml ../../../doc/professional.xml.sample wd-professional:cancel-expense-report-old}
     *
     * @param expenseReportCancelRequest A request object containing a reference to the Expense Report that should be cancelled.
     * @throws WorkdayException If an error during the operation occurs this exception is thrown.
     */
    @Processor
    @InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
    public void cancelExpenseReportOld(@Optional @Default("#[payload]") @RefOnly ExpenseReportCancelType expenseReportCancelRequest)
            throws WorkdayException
    {
        client.cancelExpenseReportOld(expenseReportCancelRequest);
    }

        /**
     * This web service operation will add or update an Expense Report.
     *
     * {@sample.xml ../../../doc/professional.xml.sample wd-professional:add-update-expense-report}
     *
     * @param expenseReportAddUpdateRequest A request object used to add (or update, if already existing) a new Expense Report into the Workday system.
     * @throws WorkdayException If an error during the operation occurs this exception is thrown.
     */
    @Processor
    @InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
    public void addUpdateExpenseReport(@Optional @Default("#[payload]") @RefOnly ExpenseReportAddUpdateType expenseReportAddUpdateRequest)
            throws WorkdayException
    {
        client.addUpdateExpenseReport(expenseReportAddUpdateRequest);
    }

   
}
