/**
 * (c) 2003-2014 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */

package org.mule.modules.workday.financial;

import org.mule.api.ConnectionException;
import org.mule.api.annotations.Connect;
import org.mule.api.annotations.ConnectionIdentifier;
import org.mule.api.annotations.ConnectivityTesting;
import org.mule.api.annotations.Connector;
import org.mule.api.annotations.Disconnect;
import org.mule.api.annotations.InvalidateConnectionOn;
import org.mule.api.annotations.Processor;
import org.mule.api.annotations.ValidateConnection;
import org.mule.api.annotations.display.FriendlyName;
import org.mule.api.annotations.param.ConnectionKey;
import org.mule.api.annotations.param.Default;
import org.mule.api.annotations.param.Optional;
import org.mule.api.annotations.param.RefOnly;
import org.mule.modules.workday.AbstractWorkdayModule;
import org.mule.modules.workday.api.exception.WorkdayCredentialsException;
import org.mule.modules.workday.api.exception.WorkdayException;

import com.workday.financial.*;

/**
 * The Financial Management Web Service contains operations that expose Workday Financial Management Business Services data.
 * 
 * @author MuleSoft, Inc.
 */
@Connector(name = "wd-financial", schemaVersion = "2.0", friendlyName = "Workday Financial Management", minMuleVersion = "3.4", connectivityTesting=ConnectivityTesting.DISABLED)
public class FinancialModule extends AbstractWorkdayModule {

	/**
	 * The Workday account's username
	 */
	private String username;
	
	/**
	 * The client, mainly for mocking purposes.
	 */
	private FinancialManagementPort client;
	
	public void setUsername(String username) {
		this.username = username;
	}

	public void setClient(Object client) {
		this.client = (FinancialManagementPort) client;
	}

	public String getUsername() {
		return username;
	}
	
	/**
	 * Connects to Workday
	 * 
	 * @param financialUser
	 * 					Workday user
	 * @param financialPassword
	 * 					Workday password
	 * @param financialEndpoint
	 * 					Workday endpoint
	 * @param financialWsdlLocation
	 * 					Workday WSDL location
	 * @throws ConnectionException
	 */
	@Connect
	public void connect(
			@FriendlyName("user") @ConnectionKey String financialUser,
			@FriendlyName("password") String financialPassword,
			@FriendlyName("endpoint") String financialEndpoint,
			@FriendlyName("wsdl") @Optional @Default("https://community.workday.com/custom/developer/API/Financial_Management/v23.0/Financial_Management.wsdl") String financialWsdlLocation)
			throws ConnectionException {
		if (client == null) {
			setUsername(financialUser);
			CxfFinancialClient cxfFinancialClient = new CxfFinancialClient(financialUser, financialPassword, financialEndpoint, financialWsdlLocation);
			initClient(cxfFinancialClient, cxfFinancialClient.getServiceVersion());
		}
	}
	

	/**
	 * Disconnects
	 */
	@Disconnect
	public void disconnect() {
		setClient(null);
	}
	

	/**
	 * Validate connection
	 * 
	 * @return true if the connector is connected
	 */
	@ValidateConnection
	public boolean validateConnection() {
		return getClient() != null;
	}

	/**
	 * Connection identifier
	 * 
	 * @return connection identifier
	 */
	@ConnectionIdentifier
	public String getConnectionIdentifier() {
		return "User: " + getUsername();
	}

	public FinancialManagementPort getClient() {
		return client;
	}

	public void setClient(FinancialManagementPort client) {
		this.client = client;
	}
	
	/**
	 * This service operation will get Beginning Balance Journals for the specified criteria. 
	 * 
	 * Beginning Balance Journal data includes Company the journal is form, journal Currency, Currency Rate Type, Ledger Year, a Memo, Journal Source,
	 * Book Code and Journal Entry Line data. Journal Entry Line data includes Inter Company Affiliate, Ledger Account, Debit Amount, Credit Amount,
	 * Currency, Currency Rate, Ledger Debit Amount, Ledger Credit Amount, Line Memo, and Worktags. The request criteria can be for a single 
	 * transaction based on Reference, or all transactions can be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-beginning-balance-journals}
	 *
	 * @param getBeginningBalanceJournalsRequest A request object containing request references, criterias and filters, and a response group
	 * 	used to specify which Beginning Balance Journal data to retrieve.
	 * @return A response object containing Beginning Balance Journal response elements including received criteria, references, 
	 * data based on criteria and grouping, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetBeginningBalanceJournalsResponseType getBeginningBalanceJournals(@Optional @Default("#[payload]") @RefOnly GetBeginningBalanceJournalsRequestType getBeginningBalanceJournalsRequest) 
			throws WorkdayException
	{
		return client.getBeginningBalanceJournals(getBeginningBalanceJournalsRequest);
	}

	/**
	 * This service operation can be used to retrieve Regions for the specified criteria.
	 * 
	 * Region Data include Effective Date, Organization Data, Organization Subtype and Organization Container. Organization Data include Id, 
	 * Organization Name, Organization Code, Include Organization Code in Name, Organization Active, Availability Date, Organization Visibility, 
	 * External URL and Integration ID Data. Integration ID Data include ID ID Include System Id and External ID The request criteria can be 
	 * for a single Region based on Reference or all Regions if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-regions}
	 *
	 * @param getRegionsRequest A request object that contains request criterias, filters and references and a response group for retrieving Region data.
	 * @return A response object containing a list of Regions from Workday and metadata about the retrieved data.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetRegionsResponseType getRegions(@Optional @Default("#[payload]") @RefOnly GetRegionsRequestType getRegionsRequest)
			throws WorkdayException
	{
		return client.getRegions(getRegionsRequest);
	}

	/**
	 * This service operation will add or update Effort Certification Change Reason Codes.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-effort-certification-change-reason-code}
	 *
	 * @param putEffortCertificationChangeReasonCodeRequest A request object which contains all the necessary data to add or update Effort Certification Change Reason Codes.
	 * @return A response object that contains information about the Effort Certification Change Reason Codes that were added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutEffortCertificationChangeReasonCodeResponseType putEffortCertificationChangeReasonCode(@Optional @Default("#[payload]") @RefOnly PutEffortCertificationChangeReasonCodeRequestType putEffortCertificationChangeReasonCodeRequest) 
			throws WorkdayException
	{
		return client.putEffortCertificationChangeReasonCode(putEffortCertificationChangeReasonCodeRequest);
	}

	/**
	 * This service operation will add or update the Custom Worktags for the Custom Worktag Configuration.
	 * 
	 * Custom Worktags are user-defined values that can be used to tag transactions with data that is not represented by the Workday-delivered transaction Worktags.
	 * Custom Worktags must be enabled and mapped to a Custom Worktag Configuration. Custom Worktag data includes the Configuration Reference, 
	 * the Worktags Reference ID and Value for that Configuration.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-custom-worktag}
	 *
	 * @param putCustomWorktagRequest A request object that contains a reference to the Worktag to update, along with the data pertaining to the Worktag.
	 * @return A response object containing data about the Custom Worktag that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutCustomWorktagResponseType putCustomWorktag(@Optional @Default("#[payload]") @RefOnly PutCustomWorktagRequestType putCustomWorktagRequest)
			throws WorkdayException
	{
		return client.putCustomWorktag(putCustomWorktagRequest);
	}

	/**
	 * This service operation with add or update Surveys. You may only update a Survey currently in the system if it has no Survey Responses to the questions.
	 * If you wish to update a Survey that has been responded to, please make a new Survey.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-survey}
	 *
	 * @param putSurveyRequest A request object containing all the data for the survey.
	 * @return A response object that contains a reference to the survey that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutSurveyResponseType putSurvey(@Optional @Default("#[payload]") @RefOnly PutSurveyRequestType putSurveyRequest)
			throws WorkdayException
	{
		return client.putSurvey(putSurveyRequest);
	}

	/**
	 * This service operation will get Currency Rate Types for the specified criteria. 
	 * 
	 * Currency Rate Type data includes the Currency Rate Type ID, Description, and if it is the Default Rate Type. 
	 * The request criteria can be for a single transaction based on Reference, or all transaction can be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-currency-rate-types}
	 *
	 * @param getCurrencyRateTypesRequest A request object containing a request reference, filter and criteria 
	 * 				along with a response group for retrieving Currency Rate Type data.
	 * @return A response object containing a list of Currency Rate Types along with metadata about the response.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetCurrencyRateTypesResponseType getCurrencyRateTypes(@Optional @Default("#[payload]") @RefOnly GetCurrencyRateTypesRequestType getCurrencyRateTypesRequest) 
			throws WorkdayException
	{
		return client.getCurrencyRateTypes(getCurrencyRateTypesRequest);
	}

	/**
	 * This service operation will add or update Spend Categories.Spend Categories are a classification for items and services that drives the accounting in Workday. 
	 * Whenever an accounting entry is made, Workday looks first at the spend category. All items-expense items, purchase items-and services are associated with a Spend Category. 
	 * Sales items are associated with its analog: Revenue Category. This service returns the spend categories that are applied to every business asset-that is, expense item, purchase item, and service-in Workday.
	 * You can define categories to fit your enterprise's needs. You can also followed a standard like the UN/SPSC coding hierarchy. 
	 * 
	 * Spend Category data for this service includes Spend Category ID, Spend Category Name, Procurement Usage flag, Expense Usage flag, Track Items flag, Accounting Treatment,
	 * Depreciation Profile, and UNSPSC Code Range. Business Asset and Resource are synonymous and refer to the same business object.
	 * 
	 * Spend Category and Resource Category are synonymous and refer to the same business object.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-resource-category}
	 *
	 * @param putResourceCategoryRequest A request object containing a reference to the Spend Category to update and all Spend Category data items.
	 * 				The reference is optional and should not be used when wanting to add a new Resource Category.
	 * @return A response object containing data about the Resource Category that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutResourceCategoryResponseType putResourceCategory(@Optional @Default("#[payload]") @RefOnly PutResourceCategoryRequestType putResourceCategoryRequest) 
			throws WorkdayException
	{
		return client.putResourceCategory(putResourceCategoryRequest);
	}

	/**
	 * This service operation will add or update Projects used as worktag values only. 
	 * 
	 * Basic Project data includes Project ID, Project Name, Inactive status. These basic projects can only be used as worktag values and will not be usable
	 * from many financial transactions such as Customer Invoices, Project Time Sheets. The request criteria can be for a single transaction based on
	 * Reference, or all transaction can be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-basic-projects}
	 *
	 * @param getBasicProjectsRequest A request object used to retrieve Basic Projects. This object contains request references, filters and criterias
	 * 				along with a response group.
	 * @return A response object containing a list of Basic Projects along with received criteria and references.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetBasicProjectsResponseType getBasicProjects(@Optional @Default("#[payload]") @RefOnly GetBasicProjectsRequestType getBasicProjectsRequest) 
			throws WorkdayException
	{
		return client.getBasicProjects(getBasicProjectsRequest);
	}

	/**
	 * This web service returns data for 1042-S Income Codes.
	 * 
	 * Income Code for 1042-S data includes Withholding Spend Categories or Spend Category Hierarchies for all Income Codes, plus: Income Code, Description, 
	 * Spend Categories or Spend Category Hierarchies. A single Investor can be returned by supplying a reference ID or Income Code, a number of Income
	 *  codes by providing Spend Categories and/or Hierarchies or all 1042-S Income Codes can be returned if no criteria is provided.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get1042-s-income-codes}
	 *
	 * @param get1042SIncomeCodesRequest A request object containing request reference, criteria, filter to retrieve 1042-S income code data.
	 * @return A response object containing a list of 1042-S Income Codes that adhere to the specified criteria.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public Get1042SIncomeCodesResponseType get1042SIncomeCodes(@Optional @Default("#[payload]") @RefOnly Get1042SIncomeCodesRequestType get1042SIncomeCodesRequest) 
			throws WorkdayException
	{
		return client.get1042SIncomeCodes(get1042SIncomeCodesRequest);
	}

	/**
	 * This service operation will get Payment Messages for the specified criteria.
	 * 
	 * The request criteria can be for a single payment message based on Reference, or all payment messages for the specified criteria.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-payment-messages-financial}
	 *
	 * @param getPaymentMessagesRequest A request object containing request references, criterias, response filters and response groups.
	 * @return A response object containing a list of Payment Messages, including metadata about the response.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetPaymentMessagesResponseType getPaymentMessagesFinancial(@Optional @Default("#[payload]") @RefOnly GetPaymentMessagesRequestType getPaymentMessagesRequest) 
			throws WorkdayException
	{
		return client.getPaymentMessages(getPaymentMessagesRequest);
	}

	/**
	 * This service operation will retrieve a specified Effort Certifying Text (if specified) or will return all Effort Certifying Texts.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-effort-certifying-texts}
	 *
	 * @param getEffortCertifyingTextsRequest A request object containing request references, criterias, response filters and response groups to retrieve
	 * 				a list of Effort Certifying Texts.
	 * @return A response object containing Effort Certifying Text data.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetEffortCertifyingTextsResponseType getEffortCertifyingTexts(@Optional @Default("#[payload]") @RefOnly GetEffortCertifyingTextsRequestType getEffortCertifyingTextsRequest) 
			throws WorkdayException
	{
		return client.getEffortCertifyingTexts(getEffortCertifyingTextsRequest);
	}

	/**
	 * This service operation will get Spend Categories for the specified criteria. Spend Categories are a classification for items and services that drives the accounting in Workday. 
	 * Whenever an accounting entry is made, Workday looks first at the spend category. All items-expense items, purchase items-and services are associated with a Spend Category. 
	 * 
	 * Sales items are associated with its analog: Revenue Category. Spend Category data for this service includes Spend Category ID, Spend Category Name, Procurement Usage flag, 
	 * Expense Usage flag, Track Items flag, Accounting Treatment, Depreciation Profile, and UNSPSC Code Range. The request criteria can be for a single transaction based on Reference, 
	 * or all transaction can be retrieved if no criteria is specified. Spend Category and Resource Category are synonymous and refer to the same business object.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-resource-categories}
	 *
	 * @param getResourceCategoriesRequest A request object containing request references, criterias, filters and response groups.
	 * @return A response object containing Spend Category resource objects, including received criteria, references, data based on the criteria given in the request, grouping,
	 * 			results filter and a result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetResourceCategoriesResponseType getResourceCategories(@Optional @Default("#[payload]") @RefOnly GetResourceCategoriesRequestType getResourceCategoriesRequest) 
			throws WorkdayException
	{
		return client.getResourceCategories(getResourceCategoriesRequest);
	}

	/**
	 * This service operation will get Statistics for the specified criteria.
	 * 
	 * Statistics data includes the Statistic Definition, Fiscal Period, Memo, reference to the Statistics line values (Company, Worktags and Line Value), and file attachment. 
	 * The request criteria can be for a single object based on Reference, or all data will be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-statistics}
	 *
	 * @param getStatisticsRequest A request object that is used to retrieve statistics from Workday based on references, criterias, response filters and response groups.
	 * @return A response object containing Spend Category response elements including received criteria, references, data based on criteria and grouping, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetStatisticsResponseType getStatistics(@Optional @Default("#[payload]") @RefOnly GetStatisticsRequestType getStatisticsRequest) 
			throws WorkdayException
	{
		return client.getStatistics(getStatisticsRequest);
	}

	/**
	 * This service operation will add or update Sales Items used as worktag values only. 
	 * 
	 * Basic Sales Item data includes Sales Item ID, Item Name, Inactive status. These basic sales items can only be used as worktag values and will not be usable from many 
	 * financial transactions such as Customer Invoices, Customer Contracts, Expense Reports, or Cash Sale transactions. The request criteria can be for a single transaction 
	 * based on Reference, or all transaction can be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-basic-sales-items}
	 *
	 * @param getBasicSalesItemsRequest A request object containing request reference, criteria, filter and response group for getting Sales Item data.
	 * @return A response object containing Sales Item response elements including received criteria, references, data based on criteria and grouping, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetBasicSalesItemsResponseType getBasicSalesItems(@Optional @Default("#[payload]") @RefOnly GetBasicSalesItemsRequestType getBasicSalesItemsRequest) 
			throws WorkdayException
	{
		return client.getBasicSalesItems(getBasicSalesItemsRequest);
	}

	/**
	 * This service operation will add or update a Payment Term. 
	 * 
	 * Data includes Payment Terms ID, Payment Terms Name, Due Days, Payment Discount Days, Payment Discount Percent, and Grace Days.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-payment-term}
	 *
	 * @param putPaymentTermRequest A request object containing Payment Term reference to add or update Payment Term data
	 * @return A response object containing a reference to the Payment Term that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutPaymentTermResponseType putPaymentTerm(@Optional @Default("#[payload]") @RefOnly PutPaymentTermRequestType putPaymentTermRequest) 
			throws WorkdayException
	{
		return client.putPaymentTerm(putPaymentTermRequest);
	}

	/**
	 * This web service operation will add or update Custom Validation Rule.
	 * 
	 * A Custom Validation Rule contains a Label and Classification. Each Custom Valiation Rule can contain many Condition Rules, 
	 * which are made up of Condition Items that make up the logic used in the Custom Validations for each Rule type.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-custom-validation-rule}
	 *
	 * @param putCustomValidationRuleRequest A request object used to add or update a Custom Validation Rule.
	 * @return A response object containing a reference to the Custom Validation Rule that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutCustomValidationRuleResponseType putCustomValidationRule(@Optional @Default("#[payload]") @RefOnly PutCustomValidationRuleRequestType putCustomValidationRuleRequest)
			throws WorkdayException
	{
		return client.putCustomValidationRule(putCustomValidationRuleRequest);
	}

	/**
	 * This service operation will get Tax Categories for the specified criteria.
	 * 
	 * Tax Category data includes Tax Category ID and Name. The request criteria can be for a single transaction based
	 * on Reference, or all transactions can be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-tax-categories}
	 *
	 * @param getTaxCategoriesRequest A request object containing a request reference, criteria, filter and response group used to retrieve Tax Category data.
	 * @return A response object containing Tax Category response elements including received criteria, references, data based on criteria and grouping, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetTaxCategoriesResponseType getTaxCategories(@Optional @Default("#[payload]") @RefOnly GetTaxCategoriesRequestType getTaxCategoriesRequest) 
			throws WorkdayException
	{
		return client.getTaxCategories(getTaxCategoriesRequest);
	}

	/**
	 * This service operation will get basic Customers that can be used as worktags only.
	 * 
	 * Basic Customer data includes Reference ID, Customer Name, and Inactive boolean. The request criteria can be for a single transaction based on Reference,
	 * or all transaction can be retrieved if no criteria is specified. Basic Customers can not be used on financial transactions such as Customer Invoices,
	 * Customer Contracts, or billing transactions. Use the Get Customers web service for retrieving complete Customer definitions used by Financial transactions.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-basic-customers}
	 *
	 * @param getBasicCustomersRequest A request object used to retrieve Basic Customers containing request references, request criterias, response filters and response groups.
	 * @return A response object containing data pertaining to Basic Customers along with metadata.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetBasicCustomersResponseType getBasicCustomers(@Optional @Default("#[payload]") @RefOnly GetBasicCustomersRequestType getBasicCustomersRequest) 
			throws WorkdayException
	{
		return client.getBasicCustomers(getBasicCustomersRequest);
	}

	/**
	 * A web service call that returns only non-basic Grants and their associated information.
	 * 
	 * The request criteria can be for a single transaction based on the Reference ID. If a Reference ID is not specified, the operation retrieves all transactions.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-grants}
	 *
	 * @param getGrantsRequest A request object containing request references, criterias, filters and response groups used to retrieve Grant data
	 * @return A response element containing Grant response data along with metadata (received criteria, references, and response filters).
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetGrantsResponseType getGrants(@Optional @Default("#[payload]") @RefOnly GetGrantsRequestType getGrantsRequest)
			throws WorkdayException
	{
		return client.getGrants(getGrantsRequest);
	}

	/**
	 * This service operation will get Payment Type data for the specified criteria.
	 * 
	 * Data includes ID, Payment Type Name, and Payment Method. The request criteria can be for a specific Payment Type based on reference(s),
	 * Payment Type Name (partial match ok), and Payment Method criteria. If no criteria are specified, all Payment Types will be retrieved.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-payment-types}
	 *
	 * @param getPaymentTypesRequest A request object used to retrieve Payment Types by specifying response filters, groups, request filters and references.
	 * @return A response object containing a list of Payment Type objects, including metadata about the response such 
	 * 			as received criteria, references, data based on criteria and grouping, and response filter.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetPaymentTypesResponseType getPaymentTypes(@Optional @Default("#[payload]") @RefOnly GetPaymentTypesRequestType getPaymentTypesRequest) 
			throws WorkdayException
	{
		return client.getPaymentTypes(getPaymentTypesRequest);
	}

	/**
	 * This service operation will get all the Custom Worktags and the Custom Worktag Configuration. Custom Worktags are user-defined values that can be used
	 * to tag transaction with data that is not represented by the Workday delivered transaction Worktags.
	 * 
	 * Custom Worktags must be enabled and mapped to a Custom Worktag Configuration. Custom Worktag data includes the Configuration Reference, the Worktags Reference ID and Value for that Configuration.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-custom-worktags}
	 *
	 * @param getCustomWorktagsRequest A request object used to retrieve Custom Worktags by specifying request references, criterias, filters and response groups.
	 * @return A response object containing Custom Worktag data retrieved from Workday, along with metadata about the retrieved data.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetCustomWorktagsResponseType getCustomWorktags(@Optional @Default("#[payload]") @RefOnly GetCustomWorktagsRequestType getCustomWorktagsRequest)
			throws WorkdayException
	{
		return client.getCustomWorktags(getCustomWorktagsRequest);
	}

	/**
	 * This service operation will add or update Statistic Definitions. 
	 * 
	 * Data includes: Name of the definition, Description, Fiscal Schedule, Required Dimensions, a boolean attribute that indicates whether 
	 * or not the definition requires Company, Statistic Type and whether or not the definition is Inactive.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-statistic-definition}
	 *
	 * @param putStatisticDefinitionRequest A request object used to add or update Statistic Definitions. The latter can be done by passing in the ID of an existing definition.
	 * @return A response object containing a reference to the Statistic Definition that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutStatisticDefinitionResponseType putStatisticDefinition(@Optional @Default("#[payload]") @RefOnly PutStatisticDefinitionRequestType putStatisticDefinitionRequest)
			throws WorkdayException
	{
		return client.putStatisticDefinition(putStatisticDefinitionRequest);
	}

	/**
	 * Adds or updates 1099 MISC Adjustments and submits them to the 1099 MISC Adjustment business process.
	 * 
	 * 1099 MISC Adjustment Data contains Company, 1099 Payee, Adjustment Date, Memo, and Line Data. Line data includes the Spend Category, amount, and memo.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:submit1099-m-i-s-c-adjustment}
	 *
	 * @param submit1099MISCAdjustmentRequest A request object containing data for creating or updating a 1099 MISC Adjustment and submitting for business processing.
	 * @return A response object containing a reference to the 1099 MISC Adjustment that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public Submit1099MISCAdjustmentResponseType submit1099MISCAdjustment(@Optional @Default("#[payload]") @RefOnly Submit1099MISCAdjustmentRequestType submit1099MISCAdjustmentRequest) 
			throws WorkdayException
	{
		return client.submit1099MISCAdjustment(submit1099MISCAdjustmentRequest);
	}

	/**
	 * This service operation will modify an existing Effort Certification Text (if a valid ID is provided) or will create a new Effort Certification Text.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-effort-certifying-text}
	 *
	 * @param putEffortCertifyingTextRequest A request object containing data used to add or update Effort Certification Texts on Workday.
	 * @return A response object that contains a reference to the Effort Certification Text that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutEffortCertifyingTextResponseType putEffortCertifyingText(@Optional @Default("#[payload]") @RefOnly PutEffortCertifyingTextRequestType putEffortCertifyingTextRequest) 
			throws WorkdayException
	{
		return client.putEffortCertifyingText(putEffortCertifyingTextRequest);
	}

	/**
	 * This service operation will add or update Program Hierarchies. This operation will allow the addition of
	 * an existing Program Hierarchy as the superior and the addition of existing Programs as subordinates.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-program-hierarchy}
	 *
	 * @param putProgramHierarchyRequest A request object containing a reference to the Program Hierarchy to update and Program Hierarchy data items.
	 * @return A response object containing a reference to the Program Hierarchy that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutProgramHierarchyResponseType putProgramHierarchy(@Optional @Default("#[payload]") @RefOnly PutProgramHierarchyRequestType putProgramHierarchyRequest) 
			throws WorkdayException
	{
		return client.putProgramHierarchy(putProgramHierarchyRequest);
	}

	/**
	 * This service operation will get Business Units. This service will return the Business Unit and its associated information. The request 
	 * criteria can be for a single transaction based on Reference, or all transactions can be retrieved if no Reference is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-business-units}
	 *
	 * @param getBusinessUnitsRequest A request object containing request reference, criteria, filter and response group used to retrieve Business Unit data.
	 * @return A response object containing Business Unit response objects, including received criteria, references, result filters and a result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetBusinessUnitsResponseType getBusinessUnits(@Optional @Default("#[payload]") @RefOnly GetBusinessUnitsRequestType getBusinessUnitsRequest) 
			throws WorkdayException
	{
		return client.getBusinessUnits(getBusinessUnitsRequest);
	}

	/**
	 * This service operation will submit Business Plan Amendment Requests.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:submit-business-plan-amendment}
	 *
	 * @param submitBusinessPlanAmendmentRequest A request object used to submit Business Plan Amendment requests.
	 * @return A response object containing a reference to the Business Plan Amendment that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public SubmitBusinessPlanAmendmentResponseType submitBusinessPlanAmendment(@Optional @Default("#[payload]") @RefOnly SubmitBusinessPlanAmendmentRequestType submitBusinessPlanAmendmentRequest)
			throws WorkdayException
	{
		return client.submitBusinessPlanAmendment(submitBusinessPlanAmendmentRequest);
	}

	/**
	 * A web service that is used to add or update Grants used as worktag values only. To add or update a complete Grant definition, use the Put Grant web service.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-basic-grant}
	 *
	 * @param putBasicGrantRequest A request object that contains data necessary to add or update a Basic Grant.
	 * @return A response object containing a reference to the Basic Grant that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutBasicGrantResponseType putBasicGrant(@Optional @Default("#[payload]") @RefOnly PutBasicGrantRequestType putBasicGrantRequest) throws WorkdayException
	{
		return client.putBasicGrant(putBasicGrantRequest);
	}

	/**
	 * This service operation will add or update Business Unit Hierarchies. This operation will allow the addition of 
	 * an existing Business Unit Hierarchy as the superior and the addition of existing Funds as subordinates.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-business-unit-hierarchy}
	 *
	 * @param putBusinessUnitHierarchyRequest A request object containing all Business Unit Hierarchy data objects and a 
	 * 				Business Unit Hierarchy reference object used to update an existing Business Unit Hierarchy
	 * @return A response object containing a reference to the Business Unit Hierarchy that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutBusinessUnitHierarchyResponseType putBusinessUnitHierarchy(@Optional @Default("#[payload]") @RefOnly PutBusinessUnitHierarchyRequestType putBusinessUnitHierarchyRequest) 
			throws WorkdayException
	{
		return client.putBusinessUnitHierarchy(putBusinessUnitHierarchyRequest);
	}

	/**
	 * A web service operation that is used to retrieve all or a some Gift Hierarchies.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-gift-hierarchies}
	 *
	 * @param getGiftHierarchiesRequest A request object containing request criterias, filters, references and response groups used to specify which Gift Hierarchies to retrieve.
	 * @return A response object that contains the instances returned by this operation and their accompanying data.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetGiftHierarchiesResponseType getGiftHierarchies(@Optional @Default("#[payload]") @RefOnly GetGiftHierarchiesRequestType getGiftHierarchiesRequest) 
			throws WorkdayException
	{
		return client.getGiftHierarchies(getGiftHierarchiesRequest);
	}

	/**
	 * This web service operation is designed to get an instance or return all Effort Certification Eligibility Rules (Condition Rules) without dependencies.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-effort-certification-eligibility-rules-without-dependencies}
	 *
	 * @param getEffortCertificationEligibilityRulesWithoutDependenciesRequest A request object containing Effort Certification Eligibility Rule response objects including received
	 * 					criteria, references, data based on criteria and grouping, results filter and result summary.
	 * @return A response object containing Effort Certification Eligibility Rule response elements including received criteria, references, data based on criteria and grouping, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetEffortCertificationEligibilityRulesResponseType getEffortCertificationEligibilityRulesWithoutDependencies(
			@Optional @Default("#[payload]") @RefOnly GetEffortCertificationEligibilityRulesWithoutDependenciesRequestType getEffortCertificationEligibilityRulesWithoutDependenciesRequest)
			throws WorkdayException
	{
		return client.getEffortCertificationEligibilityRulesWithoutDependencies(getEffortCertificationEligibilityRulesWithoutDependenciesRequest);
	}

	/**
	 * This service operation will add or update Programs. 
	 * 
	 * The Program data includes a Name, ID, Include Program ID in Name, Program Type and the ability to create an Inactive Program.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-program}
	 *
	 * @param putProgramRequest A request object containing all Program data items and a reference to a Program that should be updated.
	 * 			The latter should only be used when updating an existing Program.
	 * @return A response object containing a reference to the Program object that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutProgramResponseType putProgram(@Optional @Default("#[payload]") @RefOnly PutProgramRequestType putProgramRequest) 
			throws WorkdayException
	{
		return client.putProgram(putProgramRequest);
	}

	/**
	 * This service operation will get Journal Entries for the specified criteria.
	 * 
	 * Journal data includes the Journal Entry Reference, Journal Entry Status, Company, Currency, Ledger Period, Accounting Date, Journal Source, Transaction Date, 
	 * Operational Transaction, Memo, Book Code, Journal Entry Line data and Attachment Data. Journal Entry Line data includes Inter Company Affiliate, Ledger Account,
	 * Debit Amount, Credit Amount, Currency, Ledger Debit Amount, Ledger Credit Amount, Memo, and Worktags. The request criteria can be for a single transaction
	 *  based on Reference, or all transaction can be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-journals}
	 *
	 * @param getJournalsRequest A request object used to retrieve Journals, optionally specifiying criterias, references, filters and response groups.
	 * @return A response object containing a list of Journals that adhere to the given criteria, along with metadata about the response objects.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetJournalsResponseType getJournals(@Optional @Default("#[payload]") @RefOnly GetJournalsRequestType getJournalsRequest) 
			throws WorkdayException
	{
		return client.getJournals(getJournalsRequest);
	}

	/**
	 * This service operation will get loans. This service will return the Loan and its associated information. The request criteria can be for a single instance
	 * based on Reference, or all instances of loans can be retrieved if no Reference is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-loans}
	 *
	 * @param getLoansRequest A request object containing request reference, criteria, filter and response group for getting Loan data.
	 * @return A response object containing Loan response objects including received criteria, references, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown.
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetLoansResponseType getLoans(@Optional @Default("#[payload]") @RefOnly GetLoansRequestType getLoansRequest) 
			throws WorkdayException
	{
		return client.getLoans(getLoansRequest);
	}

	/**
	 * This service operation will modify an existing Effort Certification Type (if a valid ID is provided) or will create a new Effort Certification Type.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-effort-certification-type}
	 *
	 * @param putEffortCertificationTypeRequest A request object containing all Effort Certification Type data items and a reference to a Effort Certification Type 
	 * 			that should be updated.	The latter should only be used when updating an existing Effort Certification Type.
	 * @return A response object containing a reference to the Effort Certification Type that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutEffortCertificationTypeResponseType putEffortCertificationType(@Optional @Default("#[payload]") @RefOnly PutEffortCertificationTypeRequestType putEffortCertificationTypeRequest)
			throws WorkdayException
	{
		return client.putEffortCertificationType(putEffortCertificationTypeRequest);
	}

	/**
	 * This service operation will get Fund Hierarchies. This service will return the Fund Hierarchies and its associated information. The request criteria
	 * can be for a single transaction based on Reference, or all transactions can be retrieved if no Reference is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-fund-hierarchies}
	 *
	 * @param getFundHierarchiesRequest A request object containing request reference, criteria, filter and response group for getting Fund Hierarchy data
	 * @return A response object containing Fund Hierarchy response objects including received criteria, references, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetFundHierarchiesResponseType getFundHierarchies(@Optional @Default("#[payload]") @RefOnly GetFundHierarchiesRequestType getFundHierarchiesRequest) 
			throws WorkdayException
	{
		return client.getFundHierarchies(getFundHierarchiesRequest);
	}

	/**
	 * This web service operation is used to retrieve information for search settings. 
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-search-settings-financial}
	 *
	 * @param getSearchSettingsRequest A request object used to retrieve search settings from Workday, possibly filtering out unwanted settings.
	 * @return A response object containing search setup data.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetSearchSettingsResponseType getSearchSettingsFinancial(@Optional @Default("#[payload]") @RefOnly GetSearchSettingsRequestType getSearchSettingsRequest) 
			throws WorkdayException
	{
		return client.getSearchSettings(getSearchSettingsRequest);
	}

	/**
	 * This service operation will add or update Statistics.
	 * 
	 * Data includes: reference to the related Statistic Definition, Fiscal Period, Memo, the Statistics line values (Company, Worktags and Line Value), and optional file attachment.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-statistic}
	 *
	 * @param putStatisticRequest A request object containing Statistic data that will be assigned to the Statistic object being added or updated. A reference to an existing Statistic
	 * 					can also be set to update it with the specified Statistic data.
	 * @return A response object containing a reference to the Statistic that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown.
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutStatisticResponseType putStatistic(@Optional @Default("#[payload]") @RefOnly PutStatisticRequestType putStatisticRequest) 
			throws WorkdayException
	{
		return client.putStatistic(putStatisticRequest);
	}

	/**
	 * This service operation will add or update Sales Items used as worktag values only.
	 * 
	 * Basic Sales Item data includes Sales Item ID, Item Name, and optional Inactive attribute. 
	 * Use the Put Sales Item web service to add or update Sales Items used for any type of financial transaction.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-basic-sales-item}
	 *
	 * @param putBasicSalesItemRequest A request object containing Sales Item reference for update and basic Sales Item data required when used as a worktag.
	 * @return A response object containing a reference to the Sales Item that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutSalesItemResponseType putBasicSalesItem(@Optional @Default("#[payload]") @RefOnly PutBasicSalesItemRequestType putBasicSalesItemRequest)
			throws WorkdayException
	{
		return client.putBasicSalesItem(putBasicSalesItemRequest);
	}

	/**
	 * This service operation will get Programs. This service will return the Program and its associated information. 
	 * The request criteria can be for a single transaction based on Reference, or all transactions can be retrieved if no Reference is specified.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-programs}
	 *
	 * @param getProgramsRequest A request object containing request reference, criteria, filter and response group for getting Program data.
	 * @return A response object containing Program response objects including received criteria, references, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetProgramsResponseType getPrograms(@Optional @Default("#[payload]") @RefOnly GetProgramsRequestType getProgramsRequest)
			throws WorkdayException
	{
		return client.getPrograms(getProgramsRequest);
	}

	/**
	 * This web service operation is used to put search configurations into Workday.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-search-settings-financial}
	 *
	 * @param putSearchSettingsRequest A request object that contains Search Settings data to be placed into Workday.
	 * @return A response object containing tenant setup search data.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutSearchSettingsResponseType putSearchSettingsFinancial(@Optional @Default("#[payload]") @RefOnly PutSearchSettingsRequestType putSearchSettingsRequest) throws WorkdayException
	{
		return client.putSearchSettings(putSearchSettingsRequest);
	}

	/**
	 * This service operation will add or update Gifts.
	 * 
	 * The Fund data includes a Name, ID, Include Gift ID in Name, Gift Type, Gift Purpose, and the ability to create an Inactive Gift.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:submit-gift}
	 *
	 * @param submitGiftRequest A request object containing Gift reference for update and all Gift data items.
	 * @return A response object containing a reference to the Gift that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public SubmitGiftResponseType submitGift(@Optional @Default("#[payload]") @RefOnly SubmitGiftRequestType submitGiftRequest) throws WorkdayException
	{
		return client.submitGift(submitGiftRequest);
	}

	/**
	 * This Web Service will put Ledger Account Summaries or Ledger Accounts for a specified Ledger Account Summary  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-ledger-account-summary}
	 *
	 * @param putLedgerAccountSummaryRequest A request object containing Ledger Account Summary Data and an optional reference to an existing Ledger Account Summary for updating purposes.
	 * @return A response object containing a reference to the Ledger Account Summary that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutLedgerAccountSummaryResponseType putLedgerAccountSummary(@Optional @Default("#[payload]") @RefOnly PutLedgerAccountSummaryRequestType putLedgerAccountSummaryRequest)
			throws WorkdayException
	{
		return client.putLedgerAccountSummary(putLedgerAccountSummaryRequest);
	}

	/**
	 * This service operation will add or update Currency Rate Type.
	 * 
	 * Currency Rate Type data includes the Currency Rate Type ID, Description, and if it is the Default Rate Type.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-currency-rate-type}
	 *
	 * @param putCurrencyRateTypeRequest A request object containing Currency Rate Type reference for update and all Currency Rate Type data items.
	 * @return A response object containing a reference to the Currency Rate Type that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutCurrencyRateTypeResponseType putCurrencyRateType(@Optional @Default("#[payload]") @RefOnly PutCurrencyRateTypeRequestType putCurrencyRateTypeRequest) 
			throws WorkdayException
	{
		return client.putCurrencyRateType(putCurrencyRateTypeRequest);
	}

	/**
	 * This service operation will get Recurring Journal Templates for the specified criteria. Recurring Journal Templates allow you to create a journal
	 * template that can be used later to manually periodically create journals based off that template.
	 * 
	 * Recurring Journal Template data includes Recurring Journal Template ID, Template Name, Currency, Currency Rate Type, Ledger Reference, Template Start Date, 
	 * Template End Date, flag indicating if journals created from a template should be automatically submitted, control total amount, journal source, company,
	 * and Journal Line Data. Journal Line Data includes Line Ordering, inter-company affiliate, ledger account, debit amount, credit amount, currency, 
	 * ledger debit amount, ledger credit amount, memo and worktags.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-recurring-journal-templates}
	 *
	 * @param getRecurringJournalTemplatesRequest A request object containing Journal Template response elements including received 
	 * 					criteria, references, data based on criteria and grouping, results filter and result summary
	 * @return A response object containing Recurring Journal Template response elements including received criteria, references, data based on criteria and grouping, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetRecurringJournalTemplatesResponseType getRecurringJournalTemplates(@Optional @Default("#[payload]") @RefOnly GetRecurringJournalTemplatesRequestType getRecurringJournalTemplatesRequest) 
			throws WorkdayException
	{
		return client.getRecurringJournalTemplates(getRecurringJournalTemplatesRequest);
	}

	/**
	 * This Web Service will return Ledger Account Summaries, included Ledger Accounts and indirectly included accounts depending on Response Group setting. 
	 * To Return All Ledger Accounts (indirectly included) set the Response Group 'Include All Ledger Account Data'  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-ledger-account-summaries}
	 *
	 * @param getLedgerAccountSummaryRequest A request object used to specify criterias, references, filters and response groups.
	 * @return A response object containing a list of Ledger Account Summaries, along with metadata about the returned results.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetLedgerAccountSummariesResponseType getLedgerAccountSummaries(@Optional @Default("#[payload]") @RefOnly GetLedgerAccountSummaryRequestType getLedgerAccountSummaryRequest)
			throws WorkdayException
	{
		return client.getLedgerAccountSummaries(getLedgerAccountSummaryRequest);
	}

	/**
	 * This service operation will retrieve a specified Effort Certification Change Reason Code (if specified) or will return all Effort Certification Change Reason Codes.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-effort-certification-change-reason-codes}
	 *
	 * @param getEffortCertificationChangeReasonCodesRequest A request object containing Effort Certification Reason Code request data.
	 * @return A response object containing a list of Effort Certification Reason Codes, along with metadata pertaining to the response
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetEffortCertificationChangeReasonCodesResponseType getEffortCertificationChangeReasonCodes(@Optional @Default("#[payload]") @RefOnly GetEffortCertificationChangeReasonCodesRequestType getEffortCertificationChangeReasonCodesRequest)
			throws WorkdayException
	{
		return client.getEffortCertificationChangeReasonCodes(getEffortCertificationChangeReasonCodesRequest);
	}

	/**
	 * This service operation will add or update Customer Category.
	 * 
	 *  Customer Category includes the Customer Category ID and Customer Category Name. Customer categories are used to help determine accounting for transactions beyond what is defined
	 *  in the posting rules. When processing a transaction, Workday finds the category on a customer and applies the accounting based on the category. You can't override this category
	 *  for individual transactions. You can also use customer categories when selecting or searching for customers. Each customer must have one customer category.
	 *  You assign a customer to customer categories when creating or editing a customer.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-customer-category}
	 *
	 * @param putCustomerCategoryRequest A request object containing all Custoemr Category data items. 
	 * 				It also contains a Customer Category reference field used for updating existing objects.
	 * @return A response object containing a reference to the Customer Category object that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutCustomerCategoryResponseType putCustomerCategory(@Optional @Default("#[payload]") @RefOnly PutCustomerCategoryRequestType putCustomerCategoryRequest) 
			throws WorkdayException
	{
		return client.putCustomerCategory(putCustomerCategoryRequest);
	}

	/**
	 * This service operation will get Get Business Entity Contacts for the specified criteria.
	 * 
	 * The data includes the Business Entity Contact ID, the Business Entity the Contact is for (Supplier, Customer, Financial Institution or Tax Authority), the Personal Data 
	 * including Name Detail data and Contact Information data. Name Detail data includes formatted Name, Country, Prefix, First Name, Middle Name, Last Name, Secondary Last Name, 
	 * and Name Suffix. Contact Information data includes Address, Phone, Email, Instance Messenger and Web Address data. The request criteria can be for a single transaction based 
	 * on Reference, or all transactions can be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-business-entity-contacts-financial}
	 *
	 * @param getBusinessEntityContactsRequest A request object containing request reference, criterias, filters and response groups for retrieving Business Entity Contact data.
	 * @return A response object containing Business Entity Contact response elements including received criteria, references, data based on criteria and grouping, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetBusinessEntityContactsResponseType getBusinessEntityContactsFinancial(@Optional @Default("#[payload]") @RefOnly GetBusinessEntityContactsRequestType getBusinessEntityContactsRequest)
			throws WorkdayException
	{
		return client.getBusinessEntityContacts(getBusinessEntityContactsRequest);
	}

	/**
	 * Adds or updates Grant Hierarchies. In addition, this operation allows adding an existing Grant Hierarchy as the superior and adding existing Grants as subordinates.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-grant-hierarchy}
	 *
	 * @param putGrantHierarchyRequest A request object containing all Grant Hierarchy data items.
	 * 					It also contains a Grant Hierarchy reference field used for updating existing objects.
	 * @return A response object containing a reference to the Grant Hierarchy object that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutGrantHierarchyResponseType putGrantHierarchy(@Optional @Default("#[payload]") @RefOnly PutGrantHierarchyRequestType putGrantHierarchyRequest) 
			throws WorkdayException
	{
		return client.putGrantHierarchy(putGrantHierarchyRequest);
	}

	/**
	 * Adds or updates Gift Hierarchy. To add or update a complete Gift definition, use the Submit Gift web service.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-gift-hierarchy}
	 *
	 * @param putGiftHierarchyRequest A request object that holds data necessary for adding or updating a Gift Hierarchy.
	 * @return A response object containing a reference to the Gift Hierarchy object that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutGiftHierarchyResponseType putGiftHierarchy(@Optional @Default("#[payload]") @RefOnly PutGiftHierarchyRequestType putGiftHierarchyRequest)
			throws WorkdayException
	{
		return client.putGiftHierarchy(putGiftHierarchyRequest);
	}

	/**
	 * This service operation will get Program Hierarchies. This service will return the Program Hierarchies and its associated information. 
	 * The request criteria can be for a single transaction based on Reference, or all transactions can be retrieved if no Reference is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-program-hierarchies}
	 *
	 * @param getProgramHierarchiesRequest A request object containing request reference, criteria, filter and response group for getting Program Hierarchy data.
	 * @return A response object containing Program Hierarchy response elements including received criteria, references, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetProgramHierarchiesResponseType getProgramHierarchies(@Optional @Default("#[payload]") @RefOnly GetProgramHierarchiesRequestType getProgramHierarchiesRequest)
			throws WorkdayException
	{
		return client.getProgramHierarchies(getProgramHierarchiesRequest);
	}

	/**
	 * This service operation will get Fund Types. This service will return the Fund Type and its associated information. 
	 * The request criteria can be for a single transaction based on Reference, or all transactions can be retrieved if no Reference is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-fund-types}
	 *
	 * @param getFundTypesRequest A request object containing request reference, criteria, filter and response group for getting Fund Type data
	 * @return A response object containing Fund Type response elements including received references, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetFundTypesResponseType getFundTypes(@Optional @Default("#[payload]") @RefOnly GetFundTypesRequestType getFundTypesRequest) 
			throws WorkdayException
	{
		return client.getFundTypes(getFundTypesRequest);
	}

	/**
	 * Adds or updates an Accounting Journal and submits it to the Accounting Journals business process. This import is asynchronous and the response 
	 * will only be an acknowledgement or receipt. Use the Get Import Processes web service operation to determine the status of the journal process, 
	 * and the Get Import Process Messages web service operation for status, completion, and error messages. You'll find those operations in the 
	 * Integrations web service. Journals can be entered for month end adjustments, depreciation, or to record accrued liability for purchase items 
	 * that have been received but not paid for. Journal entries for accruals can automatically be reversed. Workday creates journals behind the scenes 
	 * for operational transactions, and uses posting rules to interpret the business events into debits and credits and accounts. Journal entries are
	 * posted to a ledger defined for each company. 
	 * 
	 * Accounting Journal data includes Add Only option, Business Processing options, Accounting Journal ID, Submit for Approval flag, Locked in Workday
	 * flag, Closed Period flag, Journal Number, Company, Currency, Ledger Type, Accounting Date, Journal Source, Memo, Adjustment Journal flag,
	 * Create Reversal flag, Control Total Amount, Currency Rate Type, Book Code, Journal Entry Line data and Attachment data. 
	 * 
	 * Journal Entry Line data includes Inter Company Affiliate, Ledger Account, Debit Amount, Credit Amount, Currency, Currency Rate, Ledger Debit Amount, 
	 * Ledger Credit Amount, Line Memo, and Worktags. The Get Journals web service operation will return both Accounting and Operational Journals.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:import-accounting-journal}
	 *
	 * @param importAccountingJournalRequest A request object containing data necessary for creating an Accounting Journal and submitting for business processing.
	 * @return A response object containing a reference to the Import Process and a reference to the Header Instance objects.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutImportProcessResponseType importAccountingJournal(@Optional @Default("#[payload]") @RefOnly ImportAccountingJournalRequestType importAccountingJournalRequest) 
			throws WorkdayException
	{
		return client.importAccountingJournal(importAccountingJournalRequest);
	}

	/**
	 * This web service returns data for Investors.
	 * 
	 * Investor data includes Name, ID, Default Payment Type, Contact Information, Contacts, Attachments, Alternate Names. A single Investor can be returned
	 * by supplying a reference ID, or all Investors can be returned if no criteria is provided.  
	 * 
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-investors}
	 *
	 * @param getInvestorsRequest A request object containing request reference, criteria, filter for getting Investor data.
	 * @return A response object containing response data for Investors meeting request criteria.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetInvestorsResponseType getInvestors(@Optional @Default("#[payload]") @RefOnly GetInvestorsRequestType getInvestorsRequest) 
			throws WorkdayException
	{
		return client.getInvestors(getInvestorsRequest);
	}

	/**
	 * This service operation will get Supplier Categories for the specified criteria. Supplier Category data includes the Supplier Category ID and the Supplier Category Name.
	 * Supplier Categories are used to maintain suppliers and apply accounting rules to procurement transactions. You can also use Supplier categories when selecting or 
	 * searching for Suppliers. All Suppliers must have one and only one category. The request criteria can be for a 
	 * single transaction based on Reference, or all transaction can be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-supplier-categories}
	 *
	 * @param getSupplierCategoriesRequest A request object containing request reference, criteria, filter and response group for getting Supplier Category data.
	 * @return A response object containing Supplier Category response objects including received criteria, references, data based on criteria and grouping, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetSupplierCategoriesResponseType getSupplierCategories(@Optional @Default("#[payload]") @RefOnly GetSupplierCategoriesRequestType getSupplierCategoriesRequest) 
			throws WorkdayException
	{
		return client.getSupplierCategories(getSupplierCategoriesRequest);
	}

	/**
	 * Retrieves 1099 MISC Adjustments for the specified criteria. 
	 * 
	 * 1099 MISC Adjustment Data contains Company, 1099 Payee, Adjustment Date, and additional Line Data, which includes the adjustment Spend Category, adjustment amount, and memo. 
	 * The request criteria can limit the retrieved data to a single transaction based on Reference, Company or Company Hierarchy, Start and End Dates. 
	 * If no request criteria is specified, then the operation retrieves all transactions.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get1099-m-i-s-c-adjustments}
	 *
	 * @param get1099MISCAdjustmentsRequest A request object containing request reference, criteria, filter and response group for getting 1099 MISC Adjustment data.
	 * @return A response object containing 1099 MISC Adjustment response objects including received criteria, references, data.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown.
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public Get1099MISCAdjustmentsResponseType get1099MISCAdjustments(@Optional @Default("#[payload]") @RefOnly Get1099MISCAdjustmentsRequestType get1099MISCAdjustmentsRequest) 
			throws WorkdayException
	{
		return client.get1099MISCAdjustments(get1099MISCAdjustmentsRequest);
	}

	/**
	 * This service operation will add or update Receivable Writeoff Categories. Receivable Writeoff Categories allow you to designate categories to describe receivables that you write off.
	 * 
	 *  eceivable Writeoff Category data includes the Receivable Writeoff Category ID and the Receivable Writeoff Category Name. 
	 *  In v14, Receivable Writeoff Category was renamed to Receivable Writeoff Reason. This name change is only reflected in the Workday application. The web service API still uses the name Receivable Writeoff Category.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-receivable-writeoff-category}
	 *
	 * @param putReceivableWriteoffCategoryRequest A request object that contains all Receivable Writeoff Category data items.
	 * 				It also contains a field which references an existing Receivable Writeoff Category object to apply updates.
	 * @return A response object containing a reference to the Receivable Writeoff Category that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutReceivableWriteoffCategoryResponseType putReceivableWriteoffCategory(@Optional @Default("#[payload]") @RefOnly PutReceivableWriteoffCategoryRequestType putReceivableWriteoffCategoryRequest) 
			throws WorkdayException
	{
		return client.putReceivableWriteoffCategory(putReceivableWriteoffCategoryRequest);
	}

	/**
	 * This service operation is used to add or update Award Personnel Responsibilities. 
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-award-personnel-responsibility}
	 *
	 * @param putAwardPersonnelResponsibilityRequest A request object which contains all Award Personnel Responsibility data.
	 * 				It also contains a field which references an existing Award Personnel Responsibility object to apply updates. 
	 * @return A response object containing a reference to the Award Personnel Responsibility that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutAwardPersonnelResponsibilityResponseType putAwardPersonnelResponsibility(@Optional @Default("#[payload]") @RefOnly PutAwardPersonnelResponsibilityRequestType putAwardPersonnelResponsibilityRequest)
			throws WorkdayException
	{
		return client.putAwardPersonnelResponsibility(putAwardPersonnelResponsibilityRequest);
	}

	/**
	 * DEPRECATED: This operation has been replaced by Get Workday Companies. 
	 * 
	 * This service operation will get Companies for the specified criteria. 
	 * 
	 * Company data includes the Company Name, Business Entity Tax ID and Address Data. Address Data includes Country, Date Last Changed, Address Lines, Municipality, 
	 * Submunicipality, Region, Subregion, Postal Code, Usage Data (Public, Usage Type, Use For and Comments) and Effective Date. The request criteria can be for a 
	 * single transaction based on Reference, or all transactions can be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-companies}
	 *
	 * @param getCompaniesRequest A request object containing request reference, criteria, filter and response group for getting Company data
	 * @return A response object containing Company response objects including received criteria, references, data based on criteria and grouping, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	@Deprecated
	public GetCompaniesResponseType getCompanies(@Optional @Default("#[payload]") @RefOnly GetCompaniesRequestType getCompaniesRequest) 
			throws WorkdayException
	{
		return client.getCompanies(getCompaniesRequest);
	}

	/**
	 * Adds or updates Gifts used as worktag values only. To add or update a complete Gift definition, use the Submit Gift web service.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-basic-gift}
	 *
	 * @param putBasicGiftRequest A request object which contains all Gift data.
	 * 				It also contains a field which references an existing Gift object to apply updates. 
	 * @return A response object containing a reference to the Gift that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutBasicGiftResponseType putBasicGift(@Optional @Default("#[payload]") @RefOnly PutBasicGiftRequestType putBasicGiftRequest) 
			throws WorkdayException
	{
		return client.putBasicGift(putBasicGiftRequest);
	}

	/**
	 * This service operation will get Cost Centers for the specified criteria. 
	 * 
	 * Cost Center Data include Effective Date, Organization Data, Organization Subtype, Organization Container and Worktags. Organization Data include Id, 
	 * Organization Name, Organization Code, Include Organization Code in Name, Organization Active, Availability Date, Organization Visibility, External URL 
	 * and Integration ID Data. Integration ID Data include ID ID Include System Id and External ID The request criteria can be for a single Cost Center based 
	 * on Reference or all Cost Centers if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-cost-centers}
	 *
	 * @param getCostCentersRequest A request object containing request reference, criteria, filter for getting Cost Center data.
	 * @return A response object containing a list of Cost Centers along with metadata about the response.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetCostCentersResponseType getCostCenters(@Optional @Default("#[payload]") @RefOnly GetCostCentersRequestType getCostCentersRequest) 
			throws WorkdayException
	{
		return client.getCostCenters(getCostCentersRequest);
	}

	/**
	 * This service operation will get Tax Applicability data for the specified criteria. 
	 * 
	 * Data includes Tax ID, Tax Applicability Code, Taxable, and Recoverable. The request criteria can be for a specific Tax Applicability based on reference(s),
	 * Tax Applicability Code (partial match ok), Taxable?, and Recoverable? criteria. If no criteria are specified, all Tax Applicabilities will be retrieved.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-tax-applicabilities}
	 *
	 * @param getTaxApplicabilitiesRequest A request object containing request reference, criteria, filter and response group for getting Tax Applicability data.
	 * @return A response object containing Tax Applicability response objects including received criteria, references, data based on criteria and grouping, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetTaxApplicabilitiesResponseType getTaxApplicabilities(@Optional @Default("#[payload]") @RefOnly GetTaxApplicabilitiesRequestType getTaxApplicabilitiesRequest)
			throws WorkdayException
	{
		return client.getTaxApplicabilities(getTaxApplicabilitiesRequest);
	}

	/**
	 * This web service will get 1099 Misc data. The web service can either references to specific 1099 Misc instances or a reference
	 * to 1099 Electronic Filing Run and Company which will then return all related 1099 Misc instances.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get1099-m-i-s-cs}
	 *
	 * @param get1099MiscsRequest A request object containing references, criteria, filter and response group for specifying which instances to return in the response.
	 * @return A response object containing a list of 1099 Misc Response objects, along with metadata about the returned list.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public Get1099MiscsResponseType get1099MISCs(@Optional @Default("#[payload]") @RefOnly Get1099MiscsRequestType get1099MiscsRequest) 
			throws WorkdayException
	{
		return client.get1099MISCs(get1099MiscsRequest);
	}

	/**
	 * This service operation will add or update Object Class Sets. This operation will allow creation of object classes as part of the Object Class Set. 
	 * An Object Class Set and Object Class have to be unique and an Object Class cannot repeat within a Object Class Set or be part of multiple Object Class Sets.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-object-class-set}
	 *
	 * @param putObjectClassSetRequest A request object containing Object Class Set reference for update and all Object CLass Set data items.
	 * @return A response object containing a reference to the Object Class Set that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutObjectClassSetResponseType putObjectClassSet(@Optional @Default("#[payload]") @RefOnly PutObjectClassSetRequestType putObjectClassSetRequest)
			throws WorkdayException
	{
		return client.putObjectClassSet(putObjectClassSetRequest);
	}

	/**
	 * This service operation will retrieve a specified Effort Certification Type (if specified) or will return all Effort Certification Types.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-effort-certification-types}
	 *
	 * @param getEffortCertificationTypesRequest A request object containing references, criteria, filter and response group for specifying which instances to return in the response.
	 * @return A response object containing a list of Effort Certification Types, along with metadata about the returned list.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetEffortCertificationTypesResponseType getEffortCertificationTypes(@Optional @Default("#[payload]") @RefOnly GetEffortCertificationTypesRequestType getEffortCertificationTypesRequest) 
			throws WorkdayException
	{
		return client.getEffortCertificationTypes(getEffortCertificationTypesRequest);
	}

	/**
	 * This service operation will add or update Fund Hierarchies. This operation will allow the addition of an 
	 * existing Fund Hierarchy as the superior and the addition of existing Funds as subordinates.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-fund-hierarchy}
	 *
	 * @param putFundHierarchyRequest A request object which contains all Fund Hierarchy data.
	 * 				It also contains a field which references an existing Fund Hierarchy object to apply updates. 
	 * @return A response object containing a reference to the Fund Hierarchy object that was created.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutFundHierarchyResponseType putFundHierarchy(@Optional @Default("#[payload]") @RefOnly PutFundHierarchyRequestType putFundHierarchyRequest) 
			throws WorkdayException
	{
		return client.putFundHierarchy(putFundHierarchyRequest);
	}

	/**
	 * This web service allows you to add or update 1042-S Income Codes as well as Withholding Spend Categories or Hierarchies. 
	 * 
	 * Income Code for 1042-S data includes Withholding Spend Categories or Spend Category Hierarchies for all
	 * Income Codes, plus: Income Code, Description, Spend Categories or Spend Category Hierarchies.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put1042-s-income-code}
	 *
	 * @param put1042SIncomeCodeRequest A request object which contains all 1042-S Income Code data.
	 * 				It also contains a field which references an existing 1042-S Income Code object to apply updates. 
	 * @return A response object containing a reference to the 1042-S Income Code object that was created.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public Put1042SIncomeCodeResponseType put1042SIncomeCode(@Optional @Default("#[payload]") @RefOnly Put1042SIncomeCodeRequestType put1042SIncomeCodeRequest) 
			throws WorkdayException
	{
		return client.put1042SIncomeCode(put1042SIncomeCodeRequest);
	}

	/**
	 * This service operation will add or update Supplier Categories. 
	 * 
	 * Supplier Category data includes the Supplier Category ID and the Supplier Category Name. Supplier Categories are used to maintain suppliers
	 * and apply accounting rules to procurement transactions. You can also use Supplier categories when selecting or searching for Suppliers. 
	 * All Suppliers must have one and only one category.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-supplier-category}
	 *
	 * @param putSupplierCategoryRequest A request object which contains all Supplier Category data.
	 * 				It also contains a field which references an existing Supplier Category object to apply updates. 
	 * @return A response object containing a reference to the Supplier Category object that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutSupplierCategoryResponseType putSupplierCategory(@Optional @Default("#[payload]") @RefOnly PutSupplierCategoryRequestType putSupplierCategoryRequest) 
			throws WorkdayException
	{
		return client.putSupplierCategory(putSupplierCategoryRequest);
	}

	/**
	 * Cancel Accounting Journal - This will kick off the Unpost Journal Business Process and once that is complete, it will automatically cancel the journal.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:cancel-accounting-journal}
	 *
	 * @param cancelAccountingJournalRequest A request object containing a reference to the Accounting Journal that should be cancelled.
	 * @return A response object containing a reference to the Accounting Journal that was cancelled.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public CancelAccountingJournalResponseType cancelAccountingJournal(@Optional @Default("#[payload]") @RefOnly CancelAccountingJournalRequestType cancelAccountingJournalRequest) 
			throws WorkdayException
	{
		return client.cancelAccountingJournal(cancelAccountingJournalRequest);
	}

	/**
	 * Retrieves Gifts when they are used as worktag values only.
	 * 
	 * Basic Grant data includes Gift ID, Gift Name, Include Gift ID in Name, Gift Type, Gift Purpose, Related Worktags, and Inactive status.
	 * The request criteria can be for a single transaction based on Reference ID. If no criteria is specified, the operation retrieves all transactions.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-basic-gifts}
	 *
	 * @param getBasicGiftsRequest A request object that contains request reference, criteria, filter and response group for getting basic gift worktags data
	 * @return A response object a list of Basic Gift objects, along with metadata about the returned list.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetBasicGiftsResponseType getBasicGifts(@Optional @Default("#[payload]") @RefOnly GetBasicGiftsRequestType getBasicGiftsRequest) 
			throws WorkdayException
	{
		return client.getBasicGifts(getBasicGiftsRequest);
	}

	/**
	 * This service operation will get Tax Codes for the specified criteria. Tax codes group together multiple tax rates. For example, you could create
	 * a tax code for San Francisco, and include city, state, and county sales tax rates. 
	 * 
	 * Tax Code data includes the Tax Code ID, Tax Code Name, and the Tax Rate. The request criteria can be for a single
	 * transaction based on Reference, or all transactions can be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-tax-codes}
	 *
	 * @param getTaxCodesRequest A request object containing request reference, criteria, filter and response group for getting Tax Code data
	 * @return A response object containing Tax Code response elements including received criteria, references, data based on criteria and grouping, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetTaxCodesResponseType getTaxCodes(@Optional @Default("#[payload]") @RefOnly GetTaxCodesRequestType getTaxCodesRequest)
			throws WorkdayException
	{
		return client.getTaxCodes(getTaxCodesRequest);
	}

	/**
	 * This service operation will add or update Budget Details, also referred to as Business Plan Details or Business Plan Entries. Budget data includes the
	 * Company the Budget is for, the Budget Type and the Budget Year data. Budget Year data includes the Budget Fiscal Year, the Year Status and the Budget Period.
	 *  The Budget Period includes the Period status, and the Fiscal Period.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-business-plan-details}
	 *
	 * @param putBusinessPlanDetailsRequest A request object containing data for creating or updating Business Plan Details, also called the Business Plan Entry, data.
	 * 					A unique instance of the Business Plan Details is identified by the Reference ID or the Business Plan Period. 
	 * 					A Business Plan period is identified by the Business Plan (Business Plan type and the Company) 
	 * 					and the Fiscal Period (Fiscal Year and Company's Fiscal Schedule and Fiscal Posting Interval ie: 2009-Jan).
	 * @return A response object containinga  reference to the Business Plan Details object that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutBusinessPlanDetailsResponseType putBusinessPlanDetails(@Optional @Default("#[payload]") @RefOnly PutBusinessPlanDetailsRequestType putBusinessPlanDetailsRequest)
			throws WorkdayException
	{
		return client.putBusinessPlanDetails(putBusinessPlanDetailsRequest);
	}

	/**
	 * This service operation will get Gifts. This service will return the Gift and its associated information. 
	 * The request criteria can be for a single transaction based on Reference, or all transactions can be retrieved if no Reference is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-gifts}
	 *
	 * @param getGiftsRequest A request object that contains request reference, criteria, filter and response group for getting Gift data
	 * @return A response object containing Gift response elements including received criteria, references, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetGiftsResponseType getGifts(@Optional @Default("#[payload]") @RefOnly GetGiftsRequestType getGiftsRequest) 
			throws WorkdayException
	{
		return client.getGifts(getGiftsRequest);
	}

	/**
	 * This service operation will add or update Payment Type. 
	 * 
	 * Data includes ID, Payment Type Name, and Payment Method.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-payment-type}
	 *
	 * @param putPaymentTypeRequest A request object which contains all Payment Type data items.
	 * 				It also contains a field which references an existing Payment Type object to apply updates.
	 * @return A response object containing a reference to the Payment Type that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown.
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutPaymentTypeResponseType putPaymentType(@Optional @Default("#[payload]") @RefOnly PutPaymentTypeRequestType putPaymentTypeRequest) 
			throws WorkdayException
	{
		return client.putPaymentType(putPaymentTypeRequest);
	}

	/**
	 * This service operation will add or update Beginning Balance Journals. 
	 * 
	 * Beginning Balance Journal data includes Company the journal is from, journal Currency, Currency Rate Type, Ledger Year, a Memo, Journal Source,
	 * Book Code and Journal Entry Line data. Journal Entry Line data includes Inter Company Affiliate, Ledger Account, Debit Amount, Credit Amount, 
	 * Currency, Currency Rate, Ledger Debit Amount, Ledger Credit Amount, Line Memo, and Worktags. The Beginning Balance journal will be created
	 * for the first period of the Ledger Year provided.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-beginning-balance-journal}
	 *
	 * @param putBeginningBalanceJournalRequest A request object which contains all Beginning Balance Journal data items.
	 * 				It also contains a field which references an existing Beginning Balance Journal object to apply updates.
	 * @return A response object containing a reference to the Beginning Balance Journal that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutBeginningBalanceJournalResponseType putBeginningBalanceJournal(@Optional @Default("#[payload]") @RefOnly PutBeginningBalanceJournalRequestType putBeginningBalanceJournalRequest) 
			throws WorkdayException
	{
		return client.putBeginningBalanceJournal(putBeginningBalanceJournalRequest);
	}

	/**
	 * This web service will get 1099 Misc data for a reference to 1099 Electronic Filing Run. 
	 * It will return data for that entire 1099 Electronic Filing Run, which includes summary information by Payor.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-company1099-m-i-s-c-data}
	 *
	 * @param get1099DataRequest A request object that contains references, criteria, filter and response group for specifying which instances to return in the response.
	 * @return A response object that contains the instances returned by the request and their accomanying data. 
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public Get1099ElectronicFilingRunResponseType getCompany1099MISCData(@Optional @Default("#[payload]") @RefOnly Get1099DataRequestType get1099DataRequest) 
			throws WorkdayException
	{
		return client.getCompany1099MISCData(get1099DataRequest);
	}

	/**
	 * This service operation will add or update Tax Authorities. This service operation will get Tax Authorities for the specified criteria. 
	 * Tax Authorities are business entities. Create a tax authority to pay sales taxes collected from customers. 
	 * 
	 * Tax Authority data includes Tax Authority ID, Tax Authority Reference ID, Tax Authority Name, and Business Entity data. Business Entity data
	 * includes Name, Tax ID, External ID, Contact Information data, and Business Entity Logo. Contact Information data includes Address, Phone,
	 * Email, Instant Messenger and Web Address data.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-tax-authority}
	 *
	 * @param putTaxAuthorityRequest A request object which contains all Tax Authority data items.
	 * 				It also contains a field which references an existing Tax Authority object to apply updates.
	 * @return A response object containing a reference to a Tax Authority object to which the data items should be applied.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutTaxAuthorityResponseType putTaxAuthority(@Optional @Default("#[payload]") @RefOnly PutTaxAuthorityRequestType putTaxAuthorityRequest)
			throws WorkdayException
	{
		return client.putTaxAuthority(putTaxAuthorityRequest);
	}

	/**
	 * This service operation will get Spend Category Hierarchies for the specified criteria. The request criteria can be for a single
	 * Spend Category Hierarchy based on Reference, Spend Category Hierarchy Name, or all Hierarchies can be retrieved if no Reference is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-spend-category-hierarchies}
	 *
	 * @param getSpendCategoryHierarchiesRequest A request object that contains request reference, criteria, filter and response group for getting Spend Category Hierarchy data
	 * @return A response object that contains Speed Category Hierarchy objects, along with meta data about the objects such as the received criteria,
	 * 						references, results filter and results summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetSpendCategoryHierarchiesResponseType getSpendCategoryHierarchies(@Optional @Default("#[payload]") @RefOnly GetSpendCategoryHierarchiesRequestType getSpendCategoryHierarchiesRequest) 
			throws WorkdayException
	{
		return client.getSpendCategoryHierarchies(getSpendCategoryHierarchiesRequest);
	}

	/**
	 * This web service operation is designed to get or return all Effort Certification Eligibility Rules (Condition Rules) and dependencies.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-effort-certification-eligibility-rules}
	 *
	 * @param getEffortCertificationEligibilityRulesRequest A request object that contains request reference, criteria, filter and response group for getting Effort Certification Eligibility Rules and dependencies.
	 * @return A response object containing Effort Certification Eligibility Rule response objects including received criteria, references, data based on criteria and grouping, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetEffortCertificationEligibilityRulesResponseType getEffortCertificationEligibilityRules(@Optional @Default("#[payload]") @RefOnly GetEffortCertificationEligibilityRulesRequestType getEffortCertificationEligibilityRulesRequest) 
			throws WorkdayException
	{
		return client.getEffortCertificationEligibilityRules(getEffortCertificationEligibilityRulesRequest);
	}

	/**
	 * Retrieves Grants when they are used as worktag values only.
	 * 
	 * Basic Grant data includes Grant ID, Grant Name, and Inactive status. The request criteria can be for a single transaction based on Reference ID. 
	 * If no criteria is specified, the operation retrieves all transactions.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-basic-grants}
	 *
	 * @param getBasicGrantsRequest A request object that contains a request reference, criteria, filter and response group for getting basic grant worktags data
	 * @return A response object containing a list of basic grant objects including received criteria and references.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetBasicGrantsResponseType getBasicGrants(@Optional @Default("#[payload]") @RefOnly GetBasicGrantsRequestType getBasicGrantsRequest) 
			throws WorkdayException
	{
		return client.getBasicGrants(getBasicGrantsRequest);
	}

	/**
	 * A web service which returns Revenue Category Hierarchies according to its reference id, name, or all of them should no search criteria be provided.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-revenue-category-hierarchies}
	 *
	 * @param getRevenueCategoryHierarchiesRequest A request object that contains a request reference, criteria, filter and response group for retrieving Revenue Category Hierarchies
	 * @return A response object containing a list of Revenue Category Hierarchies, including received criterias and references.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetRevenueCategoryHierarchiesResponseType getRevenueCategoryHierarchies(@Optional @Default("#[payload]") @RefOnly GetRevenueCategoryHierarchiesRequestType getRevenueCategoryHierarchiesRequest)
			throws WorkdayException
	{
		return client.getRevenueCategoryHierarchies(getRevenueCategoryHierarchiesRequest);
	}

	/**
	 * Returns Grant Hierarchies and its associated information. The request criteria can be for a single transaction based on Reference ID. 
	 * If a Reference ID is not specified, the operation retrieves all transactions.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-grant-hierarchies}
	 *
	 * @param getGrantHierarchiesRequest A request object that contains request reference, criteria, filter and response group for getting Grant Hierarchy data
	 * @return A response object that contains Grant Hierarchy response elements including received criteria, references, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetGrantHierarchiesResponseType getGrantHierarchies(@Optional @Default("#[payload]") @RefOnly GetGrantHierarchiesRequestType getGrantHierarchiesRequest) 
			throws WorkdayException
	{
		return client.getGrantHierarchies(getGrantHierarchiesRequest);
	}

	/**
	 * This service operation will add or update Funds.
	 * 
	 * The Fund data includes a Name, ID, Include Fund ID in Name, Fund Type and the ability to create an Inactive Fund.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-fund}
	 *
	 * @param putFundRequest A request object which contains all Fund data items.
	 * 				It also contains a field which references an existing Fund object to which the data items should be applied.
	 * @return A response object containing a reference to the Fund that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutFundResponseType putFund(@Optional @Default("#[payload]") @RefOnly PutFundRequestType putFundRequest)
			throws WorkdayException
	{
		return client.putFund(putFundRequest);
	}

	/**
	 * This service operation will get Statistic Definitions for the specified criteria. 
	 * 
	 * Statistics Definition data includes the Name of the definition, Description, Fiscal Schedule, Required Dimensions, 
	 * whether or not the definition requires Company, Statistic Type and whether or not the definition is Inactive. 
	 * The request criteria can be for a single object based on Reference, or all data will be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-statistic-definitions}
	 *
	 * @param getStatisticDefinitionsRequest A request object that contains request reference, criteria, filter and response group for retrieving a number of Statistic Definitions.
	 * @return A response object that contains a list of Statistic Definitions, along with metadata about the retrieved list.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetStatisticDefinitionsResponseType getStatisticDefinitions(@Optional @Default("#[payload]") @RefOnly GetStatisticDefinitionsRequestType getStatisticDefinitionsRequest)
			throws WorkdayException
	{
		return client.getStatisticDefinitions(getStatisticDefinitionsRequest);
	}

	/**
	 * This service operation will get Funds. This service will return the Fund and its associated information. 
	 * The request criteria can be for a single transaction based on Reference, or all transactions can be retrieved if no Reference is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-funds}
	 *
	 * @param getFundsRequest A request object that contains request reference, criteria, filter and response group for getting Fund data.
	 * @return A response object containing a list of Fund objects including received criteria, references, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetFundsResponseType getFunds(@Optional @Default("#[payload]") @RefOnly GetFundsRequestType getFundsRequest)
			throws WorkdayException
	{
		return client.getFunds(getFundsRequest);
	}

	/**
	 * This service operation will add or update Tax Codes. Tax codes group together multiple tax rates. 
	 * For example, you could create a taxcode for San Francisco, and include city, state, and county sales tax rates. 
	 * 
	 * Tax Code data includes the Tax Code ID, Tax Code Name, and the Tax Rate.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-tax-code}
	 *
	 * @param putTaxCodeRequest A request object which contains all Tax Code data items.
	 * 				It also contains a field which references an existing Tax Code object to which the data items should be applied.
	 * @return A response object that contains a reference to the Tax Code object that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutTaxCodeResponseType putTaxCode(@Optional @Default("#[payload]") @RefOnly PutTaxCodeRequestType putTaxCodeRequest)
			throws WorkdayException
	{
		return client.putTaxCode(putTaxCodeRequest);
	}

	/**
	 * This service operation will get Payment Terms data for the specified criteria. 
	 * 
	 * Data includes Payment Terms ID, Payment Terms Name, Due Days, Payment Discount Days, Payment Discount Percent, and Grace Days.
	 * The request criteria can be for specific Payment Terms based on reference(s), or Payment Terms Name (partial match ok) search criteria. 
	 * If no criteria are specified, all Payment Terms will be retrieved.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-payment-terms}
	 *
	 * @param getPaymentTermsRequest A request object that contains request reference, criteria, filter and response group for getting Payment Term data
	 * @return A response object containing Payment Term response objects including received criteria, references, data based on criteria and grouping, and response filter
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetPaymentTermsResponseType getPaymentTerms(@Optional @Default("#[payload]") @RefOnly GetPaymentTermsRequestType getPaymentTermsRequest) 
			throws WorkdayException
	{
		return client.getPaymentTerms(getPaymentTermsRequest);
	}

	/**
	 * Web Service to get position budgets created for Company, Business Plan Structure, Fiscal Year, Position, Employee and Version Status. 
	 * The web service will return all business plan entry lines from the previous versions of the position budget requested.
	 * If no Position Budget Reference or Request Criteria is entered, the web service will return all latest Approved, Draft or In Progress position budgets. 
	 * If a request criteria is entered, web service will return all position budgets matching the criteria. If a Position Budget reference is entered, 
	 * web service will return the single instance of position budget requested.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-position-budgets}
	 *
	 * @param getPositionBudgetsRequest A request object that contains request reference, criteria, filter and response group for getting Position Budget data
	 * @return A response object containing Position Budget response objects including received criteria, references, data based on criteria and grouping, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetPositionBudgetsResponseType getPositionBudgets(@Optional @Default("#[payload]") @RefOnly GetPositionBudgetsRequestType getPositionBudgetsRequest)
			throws WorkdayException
	{
		return client.getPositionBudgets(getPositionBudgetsRequest);
	}

	/**
	 * This service operation will update existing Budgets, also referred to as Budget Entries. 
	 * 
	 * Budget data includes the Company the Budget is for, the Budget Type and the Budget Year data. Budget Year data includes the Budget Fiscal Year, 
	 * the Year Status and the Budget Period. The Budget Period includes the Period status, and the Fiscal Period.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-budget-incremental}
	 *
	 * @param putBudgetIncrementalRequest A request object that contains data used to update a Budget object. A unique instance of the Budget is identified 
	 * 						by the Reference ID or the Budget Period. A Budget period is identified by the Budget (Budget type and the Company) 
	 * 						and the Fiscal Period (Fiscal Year and Company's Fiscal Schedule and Fiscal Posting Interval ie: 2009-Jan).
	 * @return A response object that contains a reference to the Budget object that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutBudgetIncrementalResponseType putBudgetIncremental(@Optional @Default("#[payload]") @RefOnly PutBudgetIncrementalRequestType putBudgetIncrementalRequest) 
			throws WorkdayException
	{
		return client.putBudgetIncremental(putBudgetIncrementalRequest);
	}

	/**
	 * This service operation will add or update Tax Rates. This service operation will get Tax Rates for the specified criteria. 
	 * Tax Rates designate a tax percentage to apply to a tax authority and tax category. 
	 * For example, you can establish a rate for California State sales tax of 6%. 
	 * 
	 * Tax Rate data includes Tax Rate ID, Tax Rate Name, Tax Rate, Tax Authority and Tax Category.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-tax-rate}
	 *
	 * @param putTaxRateRequest A request object which contains all Tax Rate data items.
	 * 				It also contains a field which references an existing Tax Rate object to which the data items should be applied.
	 * @return A response object that contains a reference to the Tax Rate object that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutTaxRateResponseType putTaxRate(@Optional @Default("#[payload]") @RefOnly PutTaxRateRequestType putTaxRateRequest) 
			throws WorkdayException
	{
		return client.putTaxRate(putTaxRateRequest);
	}

	/**
	 * This service operation will add or update Fund Types. 
	 * 
	 * The Fund Type data includes a Name, ID, Include Fund Type ID in Name, Inactive Fund Type flag and a Fund Restriction.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-fund-type}
	 *
	 * @param putFundTypeRequest A request object which contains all Fund values.
	 * 				It also contains a field which references an existing Fund object to which the data items should be applied.
	 * @return A response object that contains a reference to the Fund object that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutFundTypeResponseType putFundType(@Optional @Default("#[payload]") @RefOnly PutFundTypeRequestType putFundTypeRequest)
			throws WorkdayException
	{
		return client.putFundType(putFundTypeRequest);
	}

	/**
	 * This service operation will add or update Accounting Journals and submit to the Accounting Journals business process.
	 * Journals can be entered for month end adjustments, depreciation, or to record accrued liability for purchase items
	 * that have been received but not paid for. Journal entries for accruals can automatically be reversed. Workday creates 
	 * journals behind the scenes for operational transactions, and uses posting rules to interpret the business
	 * events into debits and credits and accounts. Journal entries are posted to a ledger defined for each company. 
	 * 
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:submit-accounting-journal}
	 *
	 * @param submitAccountingJournalRequest A request object which contains data necessary to submit an Accounting Journal to Workday.
	 * 				It also contains a field which references an existing Accounting Journal object to which the submitted data should be applied.
	 * @return A response object that contains a reference to the Accounting Journal that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public SubmitAccountingJournalResponseType submitAccountingJournal(@Optional @Default("#[payload]") @RefOnly SubmitAccountingJournalRequestType submitAccountingJournalRequest)
			throws WorkdayException
	{
		return client.submitAccountingJournal(submitAccountingJournalRequest);
	}

	/**
	 * A web service that is used to retrieve a number of Surveys from Workday.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-surveys}
	 *
	 * @param getSurveysRequest A request object that can be used to retrieve surveys from Workday.
	 * @return A response object containing a list of Surveys retrieved from Workday, along with metadata about the results in the response.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetSurveysResponseType getSurveys(@Optional @Default("#[payload]") @RefOnly GetSurveysRequestType getSurveysRequest)
			throws WorkdayException
	{
		return client.getSurveys(getSurveysRequest);
	}

	/**
	 * This service operation will add or update basic Customers that can be used as worktags only. 
	 * 
	 * Basic Customer data includes reference ID, Name and an Inactive boolean. Basic customers will be available as worktags, but can not be
	 * used on many financial transactions such as Customer Invoices that require fully configured customer.
	 * 
	 *  se the Get Customers and Put Customers web services for maintaining full customer information.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-basic-customer}
	 *
	 * @param putBasicCustomerRequest A request object which contains data necessary to add or update a Basic Customer.
	 * 				It also contains a field which references an existing Basic Customer used to specify to which the data in the request should be applied.
	 * @return A response object containing a reference to the Basic Customer that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutCustomerResponseType putBasicCustomer(@Optional @Default("#[payload]") @RefOnly PutBasicCustomerRequestType putBasicCustomerRequest)
			throws WorkdayException
	{
		return client.putBasicCustomer(putBasicCustomerRequest);
	}

	/**
	 * This service operation will get Payments for the specified criteria. 
	 * The request criteria can be for a single payment based on Reference, or all payments for a specified company.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-payments-financial}
	 *
	 * @param getPaymentsRequest A request object with which you can retrieve a list of Payments from Workday.
	 * @return A response object containing a list of Payments retrieved from Workday, along with metadata about the results in the response.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetPaymentsResponseType getPaymentsFinancial(@Optional @Default("#[payload]") @RefOnly GetPaymentsRequestType getPaymentsRequest) 
			throws WorkdayException
	{
		return client.getPayments(getPaymentsRequest);
	}

	/**
	 * This web service will kick off the business process to unpost an accounting journal. 
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:unpost-accounting-journal}
	 *
	 * @param unpostAccountingJournalRequest A request object that contains a reference to the Accounting Journal object which should be unposted.
	 * @return A response object containing a reference to the Accounting Journal that was unposted.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public CancelAccountingJournalResponseType unpostAccountingJournal(@Optional @Default("#[payload]") @RefOnly UnpostAccountingJournalRequestType unpostAccountingJournalRequest)
			throws WorkdayException
	{
		return client.unpostAccountingJournal(unpostAccountingJournalRequest);
	}

	/**
	 * This service operation will get Account Sets for the specified criteria. This service will return the Account Set, the Ledger Accounts 
	 * included in the Account Set, any included Account Sets, and the Chart of Accounts designation. The request criteria can be for a single 
	 * transaction based on Reference, or all transactions can be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-account-sets}
	 *
	 * @param getAccountSetsRequest A request object that contains request reference, criteria, filter and response group for getting Account Set data.
	 * @return A response object that contains a list of Account Set objects including received criteria, references, data based on criteria and grouping, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetAccountSetsResponseType getAccountSets(@Optional @Default("#[payload]") @RefOnly GetAccountSetsRequestType getAccountSetsRequest) 
			throws WorkdayException
	{
		return client.getAccountSets(getAccountSetsRequest);
	}

	/**
	 * This service operation will get Business Plan Details for the specified criteria. 
	 * 
	 * Business Plan data includes the Company the Business Plan is form, the Business Plan Type and the Business Plan Year data. Business Plan Year
	 * data includes the Business Plan Fiscal Year, the Year Status and the Business Plan Period. The Business Plan Period includes the Period status, 
	 * and the Fiscal Period. The request criteria can be for a single transaction based on Reference, or all transactions can be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-business-plan-details}
	 *
	 * @param getBusinessPlanDetailsRequest A request object that contains request reference, criteria, filter and response group for getting Business Plan Entry objects.
	 * @return A response object containing a list of Business Plan Entry objects including received criteria, references,
	 * 			data based on criteria and grouping, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetBusinessPlanDetailsResponseType getBusinessPlanDetails(@Optional @Default("#[payload]") @RefOnly GetBusinessPlanDetailsRequestType getBusinessPlanDetailsRequest)
			throws WorkdayException
	{
		return client.getBusinessPlanDetails(getBusinessPlanDetailsRequest);
	}

	/**
	 * This service operation will get Basic Suppliers that can be used as worktags only. 
	 * 
	 * Basic Supplier data includes a Reference ID, Supplier Name, and Inactive boolean. The request criteria can be for a single transaction based on Reference, or all
	 * transaction can be retrieved if no criteria is specified. Basic Suppliers can not be used on financial transactions such as Supplier Invoices or Supplier Payments.
	 * 
	 * Use the Get Suppliers web service for retrieving complete supplier definitions used by financial transactions.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-basic-suppliers}
	 *
	 * @param getBasicSuppliersRequest A request object that contains request reference, criteria, filter and response group for getting Basic Supplier objects.
	 * @return  A response object containing a list of Basic Supplier objects including received criteria, references,
	 * 			data based on criteria and grouping, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetBasicSuppliersResponseType getBasicSuppliers(@Optional @Default("#[payload]") @RefOnly GetBasicSuppliersRequestType getBasicSuppliersRequest) 
			throws WorkdayException
	{
		return client.getBasicSuppliers(getBasicSuppliersRequest);
	}

	/**
	 * This service operation will add or update Account Sets. This operation will allow creation of ledger accounts, and organize them into sets of accounts.
	 * You can include one account set in another account set. You can also designate an account set as a chart of accounts.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-account-set}
	 *
	 * @param putAccountSetRequest  A request object which contains data necessary to add or update an Account Set.
	 * 				It also contains a field which references an existing Account Set used to specify to which the data in the request should be applied.
	 * @return A response object which contains a reference to the Account Set that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutAccountSetResponseType putAccountSet(@Optional @Default("#[payload]") @RefOnly PutAccountSetRequestType putAccountSetRequest) 
			throws WorkdayException
	{
		return client.putAccountSet(putAccountSetRequest);
	}

	/**
	 * This web service operation will retrieve instances of Custom Validation Context. 
	 * 
	 * The response contains the reference to the Custom Validation Context returned, the Label, and Classification. Each Custom Valdiation Context can contain 
	 * many Custom Validations,, which are made up of Condition Items that make up the logic used in the Custom Validations for each Rule type.
	 * On the web service request, you can specify a specific reference to a Custom Validation Context to return.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-custom-validation-rules}
	 *
	 * @param getCustomValidationRulesRequest A request object that contains request reference, criteria, filter and response group for getting Custom Validation Rules.
	 * @return A response object containing a list of Custom Validation Rule objects including received criteria, references,
	 * 			data based on criteria and grouping, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetCustomValidationRulesResponseType getCustomValidationRules(@Optional @Default("#[payload]") @RefOnly GetCustomValidationRulesRequestType getCustomValidationRulesRequest)
			throws WorkdayException
	{
		return client.getCustomValidationRules(getCustomValidationRulesRequest);
	}

	/**
	 * This service operation will add or update Spend Category Hierarchies. This operation will allow for the addition of an existing
	 * Spend Category Hierarchies as the superiors/subordinates and addition of existing Spend Categories as subordinates.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-spend-category-hierarchy}
	 *
	 * @param putSpendCategoryHierarchyRequest A request object which contains data necessary to add or update a Spend Category Hierarchies.
	 * 				It also contains a field which references an existing Spend Category Hierarchies used to specify to which the data in the request should be applied.
	 * @return A response object that contains a reference to the Spend Category Hierarchy that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutSpendCategoryHierarchyResponseType putSpendCategoryHierarchy(@Optional @Default("#[payload]") @RefOnly PutSpendCategoryHierarchyRequestType putSpendCategoryHierarchyRequest) 
			throws WorkdayException
	{
		return client.putSpendCategoryHierarchy(putSpendCategoryHierarchyRequest);
	}

	/**
	 * This service operation will add or update Revenue Categories.
	 * 
	 * Revenue Category data includes Revenue Category ID, Name and Tax Applicability.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-revenue-category}
	 *
	 * @param putRevenueCategoryRequest A request object which contains data necessary to add or update a Revenue Category object.
	 * 				It also contains a field which references an existing Revenue Category used to specify to which the data in the request should be applied.
	 * @return A response object that contains a reference to the Revenue Category that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown.
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutRevenueCategoryResponseType putRevenueCategory(@Optional @Default("#[payload]") @RefOnly PutRevenueCategoryRequestType putRevenueCategoryRequest) 
			throws WorkdayException
	{
		return client.putRevenueCategory(putRevenueCategoryRequest);
	}

	/**
	 * This service operation will get Revenue Categories for the specified criteria.
	 * 
	 * Revenue Category data includes Revenue Category ID, Name and Tax Applicability. The request criteria can be for a 
	 * single transaction based on Reference, or all transaction can be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-revenue-categories}
	 *
	 * @param getRevenueCategoriesRequest A request object that contains request reference, criteria, filter and response group for getting Revenue Category data
	 * @return A response object containing a list of Revenue Category objects including received criteria, references, data based on criteria and grouping, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetRevenueCategoriesResponseType getRevenueCategories(@Optional @Default("#[payload]") @RefOnly GetRevenueCategoriesRequestType getRevenueCategoriesRequest)
			throws WorkdayException
	{
		return client.getRevenueCategories(getRevenueCategoriesRequest);
	}

	/**
	 * This service operation will add Translated Beginning Balance Amounts for the Beginning Balance Journal resolved from the 
	 * criteria of: Company, Fiscal Year, Ledger Type. The Currency specified determines which amounts will be updated. 
	 * The Beginning Balance Translation Amount Line Replacement Data updates values for lines that match the combination of Ledger Account, and Worktags.
	 * The Lines replace all lines defined for the beginning balance journal for the Currency that is specified on the request.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-beginning-balance-translation-amounts}
	 *
	 * @param putBeginningBalanceTranslationAmountsRequest A request object which contains data necessary to add or update a Balance Journal.
	 * 				It also contains a field which references an existing Balance Journal used to specify to which the data in the request should be applied.
	 * @return A response object containing a reference the the Beginning Balance Journal that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutBeginningBalanceTranslationAmountsResponseType putBeginningBalanceTranslationAmounts(@Optional @Default("#[payload]") @RefOnly PutBeginningBalanceTranslationAmountsRequestType putBeginningBalanceTranslationAmountsRequest) 
			throws WorkdayException
	{
		return client.putBeginningBalanceTranslationAmounts(putBeginningBalanceTranslationAmountsRequest);
	}

	/**
	 * This service operation will get Tax Authorities for the specified criteria. Tax Authorities are business entities.
	 * Create a tax authority to pay salestaxes collected from customers. 
	 * 
	 * Tax Authority data includes Tax Authority ID, Tax Authority Reference ID, Tax Authority Name, and Business Entity data. 
	 * Business Entity data includes Name, Tax ID, External ID, Contact Information data, and Business Entity Logo. 
	 * 
	 * Contact Information data includes Address, Phone, Email, Instant Messenger and Web Address data. The request criteria can be for a 
	 * single transaction based on Reference, or all transaction can be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-tax-authorities}
	 *
	 * @param getTaxAuthoritiesRequest A request object which contains request reference, criteria, filter and response group for getting Tax Authority data
	 * @return A response object containing a list of Tax Authority objects including received criteria, references, data based on criteria and grouping, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetTaxAuthoritiesResponseType getTaxAuthorities(@Optional @Default("#[payload]") @RefOnly GetTaxAuthoritiesRequestType getTaxAuthoritiesRequest)
			throws WorkdayException
	{
		return client.getTaxAuthorities(getTaxAuthoritiesRequest);
	}

	/**
	 * This service operation will add or update Currency Conversion Rate. 
	 * 
	 * Currency Conversion Rate data includes the Effective Timestamp, the From and Target Currencies, the Rate Type, and the Currency Rate.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-currency-conversion-rate}
	 *
	 * @param putCurrencyConversionRateRequest A request object contains a reference to an existing Currency Conversion Rate for update only purposes. 
	 * @return A response object containing a reference to the Currency Conversion Rate that was updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutCurrencyConversionRateResponseType putCurrencyConversionRate(@Optional @Default("#[payload]") @RefOnly PutCurrencyConversionRateRequestType putCurrencyConversionRateRequest)
			throws WorkdayException
	{
		return client.putCurrencyConversionRate(putCurrencyConversionRateRequest);
	}

	/**
	 * A web service which allows the entry of Revenue Category Hierarchies which embody Revenue Categories and a can contain child Revenue Category Hierarchies.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-revenue-category-hierarchy}
	 *
	 * @param putRevenueCategoryHierarchyRequest A request object which contains data necessary to add or update a Revenue Category Hierarchy.
	 * 				It also contains a field which references an existing Revenue Category Hierarchy used to specify to which the data in the request should be applied.
	 * @return A response object containing a reference to the Revenue Category Hierarchy that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutRevenueCategoryHierarchyResponseType putRevenueCategoryHierarchy(@Optional @Default("#[payload]") @RefOnly PutRevenueCategoryHierarchyRequestType putRevenueCategoryHierarchyRequest) 
			throws WorkdayException
	{
		return client.putRevenueCategoryHierarchy(putRevenueCategoryHierarchyRequest);
	}

	/**
	 * This service operation will add or update Business Units. 
	 * 
	 * The Business Unit data includes a Name, ID, Include Business Unit ID in Name and the ability to create an Inactive Business Unit.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-business-unit}
	 *
	 * @param putBusinessUnitRequest A request object which contains data necessary to add or update a Business Unit.
	 * 				It also contains a field which references an existing Business Unit used to specify to which the data in the request should be applied.
	 * @return A response object containing a reference to the Business Unit that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown.
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutBusinessUnitResponseType putBusinessUnit(@Optional @Default("#[payload]") @RefOnly PutBusinessUnitRequestType putBusinessUnitRequest) 
			throws WorkdayException
	{
		return client.putBusinessUnit(putBusinessUnitRequest);
	}

	/**
	 * This service operation will add or update Projects used as worktag values only.
	 * Use the Submit Workday Project web service to add or update complete Project definitions.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-basic-project}
	 *
	 * @param putBasicProjectRequest A request object which contains data necessary to add or update a Project.
	 * 				It also contains a field which references an existing Project used to specify to which the data in the request should be applied.
	 * @return A response object containing a reference to the Project that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutProjectResponseType putBasicProject(@Optional @Default("#[payload]") @RefOnly PutBasicProjectRequestType putBasicProjectRequest) 
			throws WorkdayException
	{
		return client.putBasicProject(putBasicProjectRequest);
	}

	/**
	 * This service operation will add or update Business Entity Contacts. 
	 * 
	 * The data includes the Business Entity Contact ID, the Business Entity the Contact is for (Supplier, Customer, Financial Institution or Tax Authority), 
	 * the Personal Data including Name Detail data and Contact Information data. Name Detail data includes formatted Name, Country, Prefix, First Name, 
	 * Middle Name, Last Name, Secondary Last Name, and Name Suffix. Contact Information data includes Address, Phone, Email, Instance Messenger and Web Address data.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-business-entity-contact-financial}
	 *
	 * @param putBusinessEntityContactRequest A request object which contains data necessary to add or update a Business Entity Contact.
	 * 				It also contains a field which references an existing Business Entity Contact used to specify to which the data in the request should be applied.
	 * @return A response object containing a reference to the Business Entity Contact that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutBusinessEntityContactResponseType putBusinessEntityContactFinancial(@Optional @Default("#[payload]") @RefOnly PutBusinessEntityContactRequestType putBusinessEntityContactRequest)
			throws WorkdayException
	{
		return client.putBusinessEntityContact(putBusinessEntityContactRequest);
	}

	/**
	 * Returns organization information for a type of organization. If the request does not specify an organization, the operation returns information for all organizations.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-organizations-financial}
	 *
	 * @param getOrganizationsRequest A request object which contains request reference, criteria, filter and response group for getting Organizations data.
	 * @return A response object that contains each Organization based on the Request References or Request Criteria. 
	 * 			The data returned is the current information as of the dates in the response filter, and does not include all historical information about the Organization.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetOrganizationsResponseType getOrganizationsFinancial(@Optional @Default("#[payload]") @RefOnly GetOrganizationsRequestType getOrganizationsRequest) 
			throws WorkdayException
	{
		return client.getOrganizations(getOrganizationsRequest);
	}

	/**
	 * This service operation will add or update Recurring Journal Templates for the specified criteria. Recurring Journal Templates allow
	 * you to create a journal template that can be used later to manually periodically create journals based off that template.
	 * 
	 * Recurring Journal Template data includes Recurring Journal Template ID, Template Name, Currency, Currency Rate Type, Ledger Reference, Template Start Date,
	 * Template End Date, flag indicating if journals created from a template should be automatically submitted, control total amount, journal source, company,
	 * and Journal Line Data. Journal Line Data includes Line Ordering, inter-company affiliate, ledger account, debit amount, credit amount, currency,
	 * ledger debit amount, ledger credit amount, memo and worktags.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-recurring-journal-template}
	 *
	 * @param putRecurringJournalTemplateRequest A request object which contains data necessary to add or update a Recurring Journal Template.
	 * 				It also contains a field which references an existing Recurring Journal Template used to specify to which the data in the request should be applied.
	 * @return A response object containing a reference to the Recurring Journal Template that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutRecurringJournalTemplateResponseType putRecurringJournalTemplate(@Optional @Default("#[payload]") @RefOnly PutRecurringJournalTemplateRequestType putRecurringJournalTemplateRequest)
			throws WorkdayException
	{
		return client.putRecurringJournalTemplate(putRecurringJournalTemplateRequest);
	}

	/**
	 * This service operation will get Receivable Writeoff Categories for the specified criteria. 
	 * Receivable Writeoff Categories allow you to designate categories to describe receivables that you write off. 
	 * 
	 * Receivable Writeoff Category data includes the Receivable Writeoff Category ID and the Receivable Writeoff Category Name. 
	 * 
	 * The request criteria can be for a single transaction based on Reference, or all transaction can be retrieved if no criteria is specified.
	 * In v14, Receivable Writeoff Category was renamed to Receivable Writeoff Reason. This name change is only reflected in the Workday application. 
	 * The web service API still uses the name Receivable Writeoff Category.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-receivable-writeoff-categories}
	 *
	 * @param getReceivableWriteoffCategoriesRequest A request object containing a request reference, criteria, filter and response group for getting Receivable Writeoff Category data
	 * @return A response object containing a list of Receivable Writeoff Category objects including received criteria, references, 
	 * 			data based on criteria and grouping, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown.
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetReceivableWriteoffCategoriesResponseType getReceivableWriteoffCategories(@Optional @Default("#[payload]") @RefOnly GetReceivableWriteoffCategoriesRequestType getReceivableWriteoffCategoriesRequest) 
			throws WorkdayException
	{
		return client.getReceivableWriteoffCategories(getReceivableWriteoffCategoriesRequest);
	}

	/**
	 * A web service used to retrieve Award Personnel Responsibilities from the Workday system. 
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-award-personnel-responsibilities}
	 *
	 * @param getAwardPersonnelResponsibilitiesRequest A request object used to retrieve Award Personnel Responsibilities.
	 * @return A response object containing a list of Award Personnel Responsibilities objects including received criteria, references, 
	 * 			data based on criteria and grouping, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetAwardPersonnelResponsibilitiesResponseType getAwardPersonnelResponsibilities(@Optional @Default("#[payload]") @RefOnly GetAwardPersonnelResponsibilitiesRequestType getAwardPersonnelResponsibilitiesRequest) 
			throws WorkdayException
	{
		return client.getAwardPersonnelResponsibilities(getAwardPersonnelResponsibilitiesRequest);
	}

	/**
	 * This service operation will add or update Loans. 
	 * 
	 * The Loan data includes a Loan Number, Customer, Vintage Date, First Referral Date, Property ID, Investor Type, Referral Type, Country Region and the ability to create an Inactive loan.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-loan}
	 *
	 * @param putLoanRequest A request object which contains data necessary to add or update a Loan.
	 * 				It also contains a field which references an existing Loan used to specify to which the data in the request should be applied.
	 * @return A response object containing a reference to the Loan object that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutLoanResponseType putLoan(@Optional @Default("#[payload]") @RefOnly PutLoanRequestType putLoanRequest)
			throws WorkdayException
	{
		return client.putLoan(putLoanRequest);
	}

	/**
	 * This service operation will get Object Class Sets for the specified criteria. This service will return the Object Class Set 
	 * and the Object Classes included in the Object Class Set. The request criteria can be for a single transaction based on Reference, 
	 * or all transactions can be retrieved if no reference is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-object-class-sets}
	 *
	 * @param getObjectClassSetsRequest A request object which Contains request reference, criteria, filter and response group for getting Object Class Set data.
	 * @return A response object containing a list of Object Class Set objects including received criteria, references, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown.
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetObjectClassSetsResponseType getObjectClassSets(@Optional @Default("#[payload]") @RefOnly GetObjectClassSetsRequestType getObjectClassSetsRequest) 
			throws WorkdayException
	{
		return client.getObjectClassSets(getObjectClassSetsRequest);
	}

	/**
	 * This service operation will add or update Tax Applicability. 
	 * 
	 * Data includes Tax Applicability ID, Tax Applicability Code, Taxable, and Recoverable.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-tax-applicability}
	 *
	 * @param putTaxApplicabilityRequest A request object which contains data necessary to add or update a Tax Applicability object.
	 * 				It also contains a field which references an existing Tax Applicability used to specify to which the data in the request should be applied.
	 * @return A response object containing a reference to the Tax Applicability object that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutTaxApplicabilityResponseType putTaxApplicability(@Optional @Default("#[payload]") @RefOnly PutTaxApplicabilityRequestType putTaxApplicabilityRequest)
			throws WorkdayException
	{
		return client.putTaxApplicability(putTaxApplicabilityRequest);
	}

	/**
	 * This service operation will add or update basic Suppliers that can be used as worktags only. 
	 * 
	 * Basic supplier data includes Supplier ID, Supplier Name and an Inactive boolean. Basic suppliers will be available as worktags, 
	 * but can not be used on many financial transactions such as Supplier Invoices that require full supplier information. 
	 * 
	 * Use the Get Suppliers and Put Suppliers web services for maintaining full supplier information.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-basic-supplier}
	 *
	 * @param putBasicSupplierRequest A request object which contains data necessary to add or update a Basic Supplier object.
	 * 				It also contains a field which references an existing Basic Supplier used to specify to which the data in the request should be applied.
	 * @return A response object containing a reference to the Basic Supplier that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutSupplierResponseType putBasicSupplier(@Optional @Default("#[payload]") @RefOnly PutBasicSupplierRequestType putBasicSupplierRequest) 
			throws WorkdayException
	{
		return client.putBasicSupplier(putBasicSupplierRequest);
	}

	/**
	 * This service operation will get Companies for the specified criteria.
	 * 
	 * The company data includes Company Tax ID, Industry Code, Organization data, Contact Information data, Accounting data, and company specific Sequence Generators.
	 * Contact Information data includes Address, Phone, Email, Instance Messenger and Web Address data. Accounting data includes Fiscal Schedule, Account Set, Currency,
	 * Account Control Rule Set, Account Posting rule Set, Account Translation Rule Set, and the option to Reverse Debit/Credit or Keep Debit/Credit and Reverse sign.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-workday-companies}
	 *
	 * @param getWorkdayCompaniesRequest A request object which contains request reference, criteria, filter and response group for getting a list of Company objects.
	 * @return A response object containing a list of Company objects including received criteria, references, data based on criteria and grouping, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetWorkdayCompaniesResponseType getWorkdayCompanies(@Optional @Default("#[payload]") @RefOnly GetWorkdayCompaniesRequestType getWorkdayCompaniesRequest) 
			throws WorkdayException
	{
		return client.getWorkdayCompanies(getWorkdayCompaniesRequest);
	}

	/**
	 * This service provides the ability to get all of the translated beginning balance amounts for a tenant. The Translated Beginning Balance Amounts are 
	 * broken out based on the translation currency of the lines for the combination of Company, Fiscal Year, Ledger Type, and Book Code.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-beginning-balance-translation-amounts}
	 *
	 * @param getBeginningBalanceTranslationAmountsRequest A request object which contains a list of response filters used to filter out unwanted Translated Beginning Balance Amounts.
	 * @return A response object containing Translated Beginning Balance Amounts response elements including received criteria, references, 
	 * 				data based on criteria and grouping, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetBeginningBalanceTranslationAmountsResponseType getBeginningBalanceTranslationAmounts(@Optional @Default("#[payload]") @RefOnly GetBeginningBalanceTranslationAmountsRequestType getBeginningBalanceTranslationAmountsRequest)
			throws WorkdayException
	{
		return client.getBeginningBalanceTranslationAmounts(getBeginningBalanceTranslationAmountsRequest);
	}

	/**
	 * This service operation will get Business Unit Hierarchies. This service will return the Business Unit Hierarchies and its associated information. 
	 * The request criteria can be for a single transaction based on Reference, or all transactions can be retrieved if no Reference is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-business-unit-hierarchies}
	 *
	 * @param getBusinessUnitHierarchiesRequest A request object which contains request reference, criteria, filter and response group for getting Business Unit Hierarchy data.
	 * @return A response object containing a list of Business Unit Hierarchy objects, along with received criteria, references, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetBusinessUnitHierarchiesResponseType getBusinessUnitHierarchies(@Optional @Default("#[payload]") @RefOnly GetBusinessUnitHierarchiesRequestType getBusinessUnitHierarchiesRequest) 
			throws WorkdayException
	{
		return client.getBusinessUnitHierarchies(getBusinessUnitHierarchiesRequest);
	}

	/**
	 * This web service operation will retrieve instances of Custom Validation Rules. The response contains the reference to the Custom Validation Rules returned, the Label, and Classification.
	 * Each Custom Valiation Rule can contain many Condition Rules, which are made up of Condition Items that make up the logic used in the Custom Validations for each Rule type. 
	 * On the web service request, you can specify a specific reference to a Custom Validation Rule to return. Condition Rules can refer to other Condition Rules, creating a nested structure of rules.
	 * This operation will ONLY return the top-most Condition Rules, and will not include any other rules referred to in the "Source Condition Rule" field.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-custom-validation-rules-without-dependencies}
	 *
	 * @param getCustomValidationRulesWithoutDependenciesRequest A request object which contains request reference, criteria, filter and response group used for retrieving Custom Validation Rules without Dependencies.
	 * @return A response object containing a list of Custom Validation Rules objects, along with received criteria, references, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown.
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetCustomValidationRulesResponseType getCustomValidationRulesWithoutDependencies(@Optional @Default("#[payload]") @RefOnly GetCustomValidationRulesWithoutDependenciesRequestType getCustomValidationRulesWithoutDependenciesRequest) 
			throws WorkdayException
	{
		return client.getCustomValidationRulesWithoutDependencies(getCustomValidationRulesWithoutDependenciesRequest);
	}

	/**
	 * This service operation will get Tax Rates for the specified criteria. Tax Rates designate a tax percentage to apply to a tax authority and tax category.
	 * For example, you can establish a rate for California State sales tax of 6%.
	 * 
	 * Tax Rate data includes Tax Rate ID, Tax Rate Name, Tax Rate, Tax Authority and Tax Category. The request criteria can be for a 
	 * single transaction based on Reference, or all transactions can be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-tax-rates}
	 *
	 * @param getTaxRatesRequest A request object which contains request reference, criteria, filter and response group used to retrieve Tax Rates.
	 * @return A response object containing a list of Tax Rate objects, along with received criteria, references, data based on criteria and grouping, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetTaxRatesResponseType getTaxRates(@Optional @Default("#[payload]") @RefOnly GetTaxRatesRequestType getTaxRatesRequest) 
			throws WorkdayException
	{
		return client.getTaxRates(getTaxRatesRequest);
	}

	/**
	 * This web service allows you to add or update an Investor object.
	 * 
	 * Data in the request includes Name, ID, Default Payment Type, Contact Information, Contacts, Attachments, Settlement Bank Account, and Alternate Names. 
	 * When creating a new Investor, an existing Customer, Supplier, Tax Authority, or Financial Institution can be provided. 
	 * If so, the existing Business Entity will be used in creating the new Investor.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-investor}
	 *
	 * @param putInvestorRequest A request object which contains data necessary to add or update an Investor object.
	 * 				It also contains a field which references an existing Investor used to specify to which the data in the request should be applied.
	 * @return A response object containing a reference to the Investor object that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutInvestorResponseType putInvestor(@Optional @Default("#[payload]") @RefOnly PutInvestorRequestType putInvestorRequest) 
			throws WorkdayException
	{
		return client.putInvestor(putInvestorRequest);
	}

	/**
	 * This service operation will add or update Tax Categories. 
	 * 
	 * Tax Category data includes Tax Category ID and Name.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-tax-category}
	 *
	 * @param putTaxCategoryRequest A request object which contains data necessary to add or update a Tax Category object.
	 * 				It also contains a field which references an existing Tax Category used to specify to which the data in the request should be applied.
	 * @return A response object containing a reference to the Tax Category that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutTaxCategoryResponseType putTaxCategory(@Optional @Default("#[payload]") @RefOnly PutTaxCategoryRequestType putTaxCategoryRequest) 
			throws WorkdayException
	{
		return client.putTaxCategory(putTaxCategoryRequest);
	}

	/**
	 * Adds or updates multiple Currency Conversion Rates in a single request. This operation is similar to the Put Currency Conversion Rate task, 
	 * which adds or updates only a single currency per request. 
	 * 
	 * Currency Conversion Rate data includes the Effective Timestamp, the From and Target Currencies, the Rate Type, and the Currency Rate.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-currency-conversion-rates}
	 *
	 * @param putCurrencyConversionRatesRequest A request object which contains data necessary to add or update a number of Currency Conversion Rate objects in bulk.
	 * @return A response object containing the web service version.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutCurrencyConversionRatesResponseType putCurrencyConversionRates(@Optional @Default("#[payload]") @RefOnly PutCurrencyConversionRatesRequestType putCurrencyConversionRatesRequest) 
			throws WorkdayException
	{
		return client.putCurrencyConversionRates(putCurrencyConversionRatesRequest);
	}

	/**
	 * This service operation will get Currency Conversion Rates for the specified criteria.
	 * 
	 * Currency Conversion Rate data includes the Effective Timestamp, the From and Target Currencies, the Rate Type, and the Currency Rate. The request criteria
	 * can be for a single transaction based on Reference, or all transactions can be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-currency-conversion-rates}
	 *
	 * @param getCurrencyConversionRatesRequest A request object which contains request reference, criteria, filter and response group for getting Currency Conversion Rate data.
	 * @return A response object containing a list of Currency Conversion Rate objects including received criteria, references, data based on criteria and grouping, results filter and result summary.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown.
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetCurrencyConversionRatesResponseType getCurrencyConversionRates(@Optional @Default("#[payload]") @RefOnly GetCurrencyConversionRatesRequestType getCurrencyConversionRatesRequest) 
			throws WorkdayException
	{
		return client.getCurrencyConversionRates(getCurrencyConversionRatesRequest);
	}

	/**
	 * This service operation will add or update Position Budgets. 
	 * 
	 * <ul>
	 * <li>
	 * Web service to submit position budgets for a company, business plan structure, fiscal year, fiscal period and position. If an approved position budget exists, this service will
	 * create an amendment for the existing budget. When creating an amendment, enter the total amounts of the new budget for the compensation and fringe benefit amounts. 
	 * </li>
	 * <li>
	 * If the latest position budget for the combination of company, business plan structure, fiscal year, fiscal period and position entered is In Progress or Draft, the service will update the existing position 
	 * budget and not create an amendment. If Add Only is true, the service will not create a new amendment, the existing In Progress or Draft budget will first have to be approved.  
	 * </li>
	 * </ul>
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:submit-position-budget}
	 *
	 * @param submitPositionBudgetRequest A request object which contains data necessary to add or update a Position Budget.
	 * 				It also contains a field which references an existing Position Budget used to specify to which the data in the request should be applied.
	 * 				A unique instance of the Position Budget is identified by a Company, Business Plan Structure, Fiscal Year, Fiscal Time Interval and Position.
	 * @return A response object containing a reference to the Position Budget that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public SubmitPositionBudgetResponseType submitPositionBudget(@Optional @Default("#[payload]") @RefOnly SubmitPositionBudgetRequestType submitPositionBudgetRequest) 
			throws WorkdayException
	{
		return client.submitPositionBudget(submitPositionBudgetRequest);
	}

	/**
	 * This web service allows you to update Tax Authority Form Type for a Contingent Worker.
	 * 
	 * Data in the request includes ID and Tax Authority Form Type Reference.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-contingent-worker-tax-authority-form-type}
	 *
	 * @param putContingentWorkerTaxAuthorityFormTypeRequest A request object which contains data necessary to add or update a Tax Authority Form Type for a Contingent Worker.
	 * 				It also contains a field which references an existing Tax Authority Form Type used to specify to which the data in the request should be applied.
	 * @return A response object containing a reference to the Tax Authority Form Type object that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutContingentWorkerTaxAuthorityFormTypeResponseType putContingentWorkerTaxAuthorityFormType(@Optional @Default("#[payload]") @RefOnly PutContingentWorkerTaxAuthorityFormTypeRequestType putContingentWorkerTaxAuthorityFormTypeRequest) 
			throws WorkdayException
	{
		return client.putContingentWorkerTaxAuthorityFormType(putContingentWorkerTaxAuthorityFormTypeRequest);
	}

	/**
	 * A web service used to add or update Grants. The operation also gives you the ability to create an Inactive Grant.  
	 * 
	 * Grant data includes a grant Name, ID, Include Grant ID in Name flag, Object Class Set, Facilities, and Administration Cost Schedule.
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-grant}
	 *
	 * @param putGrantRequest A request object which contains data necessary to add or update a Grant.
	 * 				It also contains a field which references an existing Grant used to specify to which the data in the request should be applied.
	 * @return A response object containing a reference to the Grant that was added or updated.
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutGrantResponseType putGrant(@Optional @Default("#[payload]") @RefOnly PutGrantRequestType putGrantRequest) 
			throws WorkdayException
	{
		return client.putGrant(putGrantRequest);
	}

	/**
	 * This service operation will get Account Sets for the specified criteria. This service will return the Account Set, the Ledger Accounts included in the Account Set,
	 * any included Account Sets, and the Chart of Accounts designation. The request criteria can be for a single transaction based on Reference, or all transactions can
	 * be retrieved if no criteria is specified. This Get request does not include the reflexive relationship to account set.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-account-sets-without-dependencies}
	 *
	 * @param getAccountSetsWithoutDependenciesRequest A request object which contains request reference, criteria, filter and response group for getting Account Set data.
	 * @return A response object containing a list of Account Set objects, along with metadata about the response such as received criteria, references, data based on criteria and grouping, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetAccountSetsResponseType getAccountSetsWithoutDependencies(@Optional @Default("#[payload]") @RefOnly GetAccountSetsWithoutDependenciesRequestType getAccountSetsWithoutDependenciesRequest)
			throws WorkdayException
	{
		return client.getAccountSetsWithoutDependencies(getAccountSetsWithoutDependenciesRequest);
	}

	/**
	 * This service operation will get Customer Categories Types for the specified criteria.
	 * 
	 *  ustomer Category includes the Customer Category ID and Customer Category Name. Customer categories are used to help determine accounting for transactions
	 *  beyond what is defined in the posting rules. When processing a transaction, Workday finds the category on a customer and applies the accounting based on the category. 
	 *  You can also use customer categories when selecting or searching for customers. Each customer must have one customer category. You assign a customer to a customer 
	 *  categories when creating or editing a customer. The request criteria can be for a single transaction based on Reference, or all transaction can be retrieved if no criteria is specified.  
	 *
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-customer-categories}
	 *
	 * @param getCustomerCategoriesRequest A request object which contains request reference, criteria, filter and response group for getting Customer Categories from Workday.
	 * @return A response object containing a list of Customer Category objects, along with metadata about the response such as received criteria, references, data based on criteria and grouping, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown 
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetCustomerCategoriesResponseType getCustomerCategories(@Optional @Default("#[payload]") @RefOnly GetCustomerCategoriesRequestType getCustomerCategoriesRequest) throws WorkdayException
	{
		return client.getCustomerCategories(getCustomerCategoriesRequest);
	}

	/**
	 * This service operation will get Loan Referral Types. This service will return the Loan Referral type and 
	 * its associated information. The request criteria can be for a single 
	 * transaction based on Reference, or all transactions can be retrieved if no Reference is specified.
	 * 
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-loan-referral-types}
	 * 
	 * @param getLoanReferralTypesRequest A request object that contains request reference, criteria, filter and response group for getting Loan Referral Type data
	 * @return Returns a response object containing Loan Referral Type response elements including received criteria, references, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)	
	public GetLoanReferralTypesResponseType getLoanReferralTypes(@Optional @Default("#[payload]") @RefOnly GetLoanReferralTypesRequestType getLoanReferralTypesRequest) 
			throws WorkdayException
	{
		return client.getLoanReferralTypes(getLoanReferralTypesRequest);
	}

	/**
	 * This service operation will add or update Loan Investor types.
	 * 
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-loan-investor-type}
	 * 
	 * @param putLoanInvestorTypeRequest A request object containing Loan Investor Type reference for update and all Loan Investor Type data items.
	 * @return Returns a response object containing Put Loan Investor Type Response Data
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)	
	public PutLoanInvestorTypeResponseType putLoanInvestorType(@Optional @Default("#[payload]") @RefOnly PutLoanInvestorTypeRequestType putLoanInvestorTypeRequest)
			throws WorkdayException
	{
		return client.putLoanInvestorType(putLoanInvestorTypeRequest);
	}

	/**
	 * This service operation will get Loan Investor Types. This service will return the Loan Investor type and its associated information. 
	 * The request criteria can be for a single transaction based on Reference, or all transactions can be retrieved if no Reference is specified.
	 * 
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:get-loan-investor-types}
	 * 
	 * @param getLoanInvestorTypesRequest A request object which contains request reference, criteria, filter and response group for getting Loan Investor Type data
	 * @return Returns a response object containing Loan Investor Type response elements including received criteria, references, results filter and result summary
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public GetLoanInvestorTypesResponseType getLoanInvestorTypes(@Optional @Default("#[payload]") @RefOnly GetLoanInvestorTypesRequestType getLoanInvestorTypesRequest)
			throws WorkdayException
	{
		return client.getLoanInvestorTypes(getLoanInvestorTypesRequest);
	}

	/**
	 * This service operation will add or update Loan Referral types
	 * 
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-loan-referral-type}
	 * 
	 * @param putLoanReferralTypeRequest A request object containing Loan Referral Type reference for update and all Loan Referral Type data items.
	 * @return Returns a response object containing Put Loan Referral Type Response Data
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutLoanReferralTypeResponseType putLoanReferralType(@Optional @Default("#[payload]") @RefOnly PutLoanReferralTypeRequestType putLoanReferralTypeRequest)
			throws WorkdayException
	{
		return client.putLoanReferralType(putLoanReferralTypeRequest);
	}
	
	/**
	 * This service operation will update existing Budgets asynchronously. Budget data includes the Company the Budget
	 * is for, the Budget Type and the Budget Year data. Budget Year data includes the Budget Fiscal Year, the Year Status
	 * and the Budget Period. The Budget Period includes the Period status, and the Fiscal Period.  
	 * 
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:import-budget-details-incremental}
	 * 
	 * @param importBudgetDetailsIncrementalHighVolumeRequest Put Import Process Request
	 * @return Put Import Process Response
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutImportProcessResponseType importBudgetDetailsIncremental(
			@Optional @Default("#[payload]") @RefOnly ImportBudgetDetailsIncrementalHighVolumeRequestType importBudgetDetailsIncrementalHighVolumeRequest)throws WorkdayException{
		return client.importBudgetDetailsIncremental(importBudgetDetailsIncrementalHighVolumeRequest);
	}

	/**
	 * For Third Party Tax Services to persist calculated tax information on Customer Invoice 
	 * 
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:put-third-party-calculated-tax-information}
	 * 
	 * @param putThirdPartyCalculatedTaxInformationRequest Put Third Party Calculated Tax Information Request
	 * @return Put Third Party Calculated Tax Information Response
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutThirdPartyCalculatedTaxInformationResponseType putThirdPartyCalculatedTaxInformation(
			@Optional @Default("#[payload]") @RefOnly PutThirdPartyCalculatedTaxInformationRequestType putThirdPartyCalculatedTaxInformationRequest)throws WorkdayException{
		return client.putThirdPartyCalculatedTaxInformation(putThirdPartyCalculatedTaxInformationRequest);
	}


	/**
	 * This service operation will add or update Budget Details asynchronously. Budget data includes 
	 * the Company the Budget is for, the Budget Type and the Budget Year data. Budget Year data includes 
	 * the Budget Fiscal Year, the Year Status and the Budget Period. The Budget Period includes the Period 
	 * status, and the Fiscal Period
	 * 
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:import-budget-details}
	 * 
	 * @param importBudgetDetailsRequest Import Budget Details Request
	 * @return Import Budget Details Response
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutImportProcessResponseType importBudgetDetails(
			@Optional @Default("#[payload]") @RefOnly ImportBudgetDetailsRequestType importBudgetDetailsRequest)throws WorkdayException{
		return client.importBudgetDetails(importBudgetDetailsRequest);
	}

	/**
	 * This service operation will import high volume Budget Amendments
	 * 
	 * {@sample.xml ../../../doc/financial.xml.sample wd-financial:import-budget-amendment}
	 * 
	 * @param importBudgetAmendmentRequestType Import Budget Amendment Request
	 * @return Import Budget Amendment Response
	 * @throws WorkdayException If an error during the operation occurs this exception is thrown
	 */
	@Processor
	@InvalidateConnectionOn(exception = WorkdayCredentialsException.class)
	public PutImportProcessResponseType importBudgetAmendment(
			@Optional @Default("#[payload]") @RefOnly ImportBudgetAmendmentRequestType importBudgetAmendmentRequestType)throws WorkdayException{
		return client.importBudgetAmendment(importBudgetAmendmentRequestType);
	}
}
