/**
 * (c) 2003-2012 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */

package org.mule.modules.sugarcrm.cxf.interceptors;

import org.mule.modules.utils.MuleSoftException;

import java.lang.reflect.Method;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;

import org.apache.cxf.interceptor.DocLiteralInInterceptor;
import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.message.Message;
import org.apache.cxf.phase.AbstractPhaseInterceptor;
import org.apache.cxf.phase.Phase;
import org.w3c.dom.Node;
import org.xmlsoap.schemas.soap.encoding.Array;

/**
 * Unmarshals to {@link JAXBElement}s the {@link Node}s left behind
 * 
 * @author Pablo Diez * 
 * @since 31/05/2012
 */
public class UnmarshalAnyElementsArrayInInterceptor extends AbstractPhaseInterceptor<Message>
{

    private Unmarshaller unmarshaller;

    public UnmarshalAnyElementsArrayInInterceptor()
    {
        super(Phase.UNMARSHAL);
        addAfter(DocLiteralInInterceptor.class.getName());
    }

    /** @see org.apache.cxf.interceptor.Interceptor#handleMessage(org.apache.cxf.message.Message) */
    @SuppressWarnings("unchecked")
    @Override
    public void handleMessage(Message message) throws Fault
    {
        try
        {
            List<Object> messageList = message.getContent(List.class);
            Object value = messageList.get(0);

            Object returnValue = value.getClass().getMethod("getReturn", null).invoke(value);
            replaceListsInObject(returnValue);
        }
        catch (Exception e)
        {
            // continue
        }
    }

    private void replaceListsInObject(Object object) 
    {
        Unmarshaller unmarshaller = createUnmarshaller();
        if (isArrayClass(object.getClass())){
            unMarshallListItems(unmarshaller, ((Array)object).getAny());
        } else {
            processBean(object, unmarshaller);
        }
    }

    private void processBean(Object object, Unmarshaller unmarshaller)
    {
        List<Method> methods = filterMethodsWithArrayReturn(object);
        for (Method method : methods)
        {
            try
            {
                unMarshallListItems(unmarshaller, ((Array) method.invoke(object)).getAny());
            }
            catch (Exception e)
            {
                throw MuleSoftException.soften(e);
            }
        }
    }

    private Unmarshaller createUnmarshaller()
    {
        try
        {
            return JAXBContext.newInstance("com.sugarcrm.sugarcrm:org.xmlsoap.schemas.soap.encoding:org.xmlsoap.schemas.wsdl").createUnmarshaller();
        }
        catch (JAXBException e)
        {
            throw MuleSoftException.soften(e);
        }
    }

    private List<Method> filterMethodsWithArrayReturn(Object object)
    {
        List<Method> methods = new LinkedList<Method>();
        for (Method m : object.getClass().getDeclaredMethods())
        {
            if (isArrayClass(m.getReturnType()))
            {
                methods.add(m);
            }
        }
        return methods;
    }

    private boolean isArrayClass(Class<?> type)
    {
        return Array.class.isAssignableFrom(type);
    }

    @SuppressWarnings("unchecked")
    private void unMarshallListItems(final Unmarshaller unmarshaller, final List<Object> anyList) 
    {
        Collection<Object> marshalledObjects = new LinkedList<Object>();
        for (Iterator<Object> iter = anyList.iterator(); iter.hasNext();)
        {
            Object value;
            try
            {
                value = ((JAXBElement<Object>) unmarshaller.unmarshal((Node) iter.next())).getValue();
                replaceListsInObject(value);
                marshalledObjects.add(value);
                iter.remove();
            }
            catch (JAXBException e)
            {
                throw MuleSoftException.soften(e);
            }
        }
        anyList.addAll(marshalledObjects);
    }

}
