/**
 * (c) 2003-2012 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */

package org.mule.modules.sugarcrm;

import org.mule.api.ConnectionException;
import org.mule.api.ConnectionExceptionCode;
import org.mule.api.annotations.Configurable;
import org.mule.api.annotations.Connect;
import org.mule.api.annotations.ConnectionIdentifier;
import org.mule.api.annotations.Connector;
import org.mule.api.annotations.Disconnect;
import org.mule.api.annotations.Processor;
import org.mule.api.annotations.ValidateConnection;
import org.mule.api.annotations.display.Password;
import org.mule.api.annotations.licensing.RequiresEnterpriseLicense;
import org.mule.api.annotations.param.ConnectionKey;
import org.mule.api.annotations.param.Default;
import org.mule.api.annotations.param.Optional;
import org.mule.modules.sugarcrm.api.SugarCrmClient;
import org.mule.modules.sugarcrm.api.SugarCrmClientSoap;
import org.mule.modules.sugarcrm.api.exceptions.InvalidLoginException;
import org.mule.modules.sugarcrm.utils.CryptoUtils;

import com.sugarcrm.sugarcrm.EntryValue;
import com.sugarcrm.sugarcrm.GetAvailableModulesRequestType;
import com.sugarcrm.sugarcrm.GetEntriesCountRequestType;
import com.sugarcrm.sugarcrm.GetEntriesCountResult;
import com.sugarcrm.sugarcrm.GetEntriesRequestType;
import com.sugarcrm.sugarcrm.GetEntryListRequestType;
import com.sugarcrm.sugarcrm.GetEntryRequestType;
import com.sugarcrm.sugarcrm.GetEntryResultVersion2;
import com.sugarcrm.sugarcrm.GetModuleFieldsRequestType;
import com.sugarcrm.sugarcrm.LoginRequestType;
import com.sugarcrm.sugarcrm.LogoutRequestType;
import com.sugarcrm.sugarcrm.ModuleList;
import com.sugarcrm.sugarcrm.NewModuleFields;
import com.sugarcrm.sugarcrm.NewSetEntriesResult;
import com.sugarcrm.sugarcrm.NewSetEntryResult;
import com.sugarcrm.sugarcrm.NewSetRelationshipListResult;
import com.sugarcrm.sugarcrm.ReturnSearchResult;
import com.sugarcrm.sugarcrm.SearchByModuleRequestType;
import com.sugarcrm.sugarcrm.SetEntriesRequestType;
import com.sugarcrm.sugarcrm.SetEntryRequestType;
import com.sugarcrm.sugarcrm.SetRelationshipRequestType;
import com.sugarcrm.sugarcrm.SetRelationshipResponseType;
import com.sugarcrm.sugarcrm.SetRelationshipsRequestType;
import com.sugarcrm.sugarcrm.SetRelationshipsResponseType;
import com.sugarcrm.sugarcrm.UserAuth;

import org.springframework.remoting.soap.SoapFaultException;

/**
 * Sugar is an affordable and easy to use customer relationship management (CRM)
 * platform, designed to help your business communicate with prospects, share sales
 * information, close deals and keep customers happy. As an open-source, web-based
 * CRM solution, Sugar is easy to customize and adapt to your changing needs. Ideal
 * for small and medium-sized companies, large enterprises and government
 * organizations, Sugar can run in the cloud or on-site.
 * 
 * @author MuleSoft, Inc.
 */
@Connector(name = "sugar", friendlyName = "Sugar")
@RequiresEnterpriseLicense
public class SugarConnector
{

    /**
     * Sugar's application client. By default uses {@link SugarCrmClientSoap}
     */
    private SugarCrmClient client;

    /**
     * Use this attribute to specify endpoint URL
     */
    @Configurable
    private String endpoint;

    private String sessionId;

    /**
     * Creates a new Sugar Connection
     * 
     * @param username Username used to initialize the session
     * @param password Password used to authenticate the user
     */
    @Connect
    public void connect(@ConnectionKey String username, @Password String password) throws ConnectionException
    {
        LoginRequestType request = new LoginRequestType();
        UserAuth userAuth = new UserAuth();
        userAuth.setUserName(username);
        userAuth.setPassword(CryptoUtils.getMd5(password));
        request.setUserAuth(userAuth);
        try
        {
            EntryValue response = getClient().login(request).getReturn();
            sessionId = response.getId();
        }
        catch (InvalidLoginException e)
        {
            throw new ConnectionException(ConnectionExceptionCode.INCORRECT_CREDENTIALS, e.getMessage(), "Invalid username or password");
        }
        catch (SoapFaultException e)
        {
            String causeMessage = e.getCause().getMessage();
            throw new ConnectionException(ConnectionExceptionCode.UNKNOWN, causeMessage, causeMessage);
        }
    }

    /**
     * End the current session
     */
    @Disconnect
    public void disconnect()
    {
        if (isConnected())
        {
            LogoutRequestType request = new LogoutRequestType();
            request.setSession(sessionId);
            getClient().logout(request);
            sessionId = null;
        }
    }

    /**
     * Validate is active connection
     */
    @ValidateConnection
    public boolean isConnected()
    {
        return sessionId != null;
    }

    /**
     * Returns the session id for the current connection
     * <p/>
     * 
     * @return the session id for the current connection
     */
    @ConnectionIdentifier
    public String getSessionId()
    {
        return sessionId;
    }

    /**
     * GetEntryList Processor.
     * <p>
     * Retrieves a list of SugarBeans.
     * <p/>
     * {@sample.xml ../../../doc/sugar-connector.xml.sample sugar:get-entry-list}
     * 
     * @param request a bean with module, query, order by and offset
     * @return {@link GetEntryResultVersion2} with entries selected
     */
    @Processor
    public com.sugarcrm.sugarcrm.GetEntryListResultVersion2 getEntryList(@Optional @Default("#[payload]") org.mule.modules.sugarcrm.requests.GetEntryListRequest request)
    {
        GetEntryListRequestType realRequest = request.getBean();
        realRequest.setSession(sessionId);
        return getClient().getEntryList(realRequest).getReturn();
    }

    /**
     * GetCountEntries Processor
     * <p>
     * Retrieves the specified number of records in a module.
     * <p/>
     * {@sample.xml ../../../doc/sugar-connector.xml.sample sugar:get-count-entries}
     * 
     * @param request a bean with module, query and deleted flag
     * @return {@link GetEntriesCountResult} with amount of records
     */
    @Processor
    public com.sugarcrm.sugarcrm.GetEntriesCountResult getCountEntries(@Optional @Default("#[payload]") org.mule.modules.sugarcrm.requests.GetEntriesCountRequest request)
    {
        GetEntriesCountRequestType realRequest = request.getBean();
        realRequest.setSession(sessionId);
        return getClient().getCountEntries(realRequest).getReturn();
    }

    /**
     * GetEntries Processor.
     * <p>
     * Retrieves multiple SugarBeans based on IDs. This API is not applicable to the
     * Reports module.
     * <p/>
     * {@sample.xml ../../../doc/sugar-connector.xml.sample sugar:get-entries}
     * 
     * @param request a bean with modules, ids and selected fields
     * @return {@link GetEntryResultVersion2} with entries selected
     */
    @Processor
    public com.sugarcrm.sugarcrm.GetEntryResultVersion2 getEntries(@Optional @Default("#[payload]") org.mule.modules.sugarcrm.requests.GetEntriesRequest request)
    {
        GetEntriesRequestType realRequest = request.getBean();
        realRequest.setSession(sessionId);
        return getClient().getEntries(realRequest).getReturn();
    }

    /**
     * SetEntry Processor.
     * <p>
     * Creates or updates a single SugarBean.
     * <p/>
     * {@sample.xml ../../../doc/sugar-connector.xml.sample sugar:set-entry}
     * 
     * @param request a bean with module and name-values for create or update
     *            SugarBean
     * @return {@link NewSetEntryResult} with a id
     */
    @Processor
    public com.sugarcrm.sugarcrm.NewSetEntryResult setEntry(@Optional @Default("#[payload]") org.mule.modules.sugarcrm.requests.SetEntryRequest request)
    {
        SetEntryRequestType realRequest = request.getBean();
        realRequest.setSession(sessionId);
        return getClient().setEntry(realRequest).getReturn();
    }

    /**
     * SetEntries Processor.
     * <p>
     * Creates or updates a list of SugarBeans.
     * <p/>
     * {@sample.xml ../../../doc/sugar-connector.xml.sample sugar:set-entries}
     * 
     * @param request a bean wit module and list of name-values for create or update
     *            a list of SugarBeans
     * @return {@link NewSetEntriesResult} a list of ids
     */
    @Processor
    public com.sugarcrm.sugarcrm.NewSetEntriesResult setEntries(@Optional @Default("#[payload]") org.mule.modules.sugarcrm.requests.SetEntriesRequest request)
    {
        SetEntriesRequestType realRequest = request.getBean();
        realRequest.setSession(sessionId);
        return getClient().setEntries(realRequest).getReturn();
    }

    /**
     * GetModuleFields Processor.
     * <p>
     * Retrieves variable definitions (vardefs) for fields of the specified
     * SugarBean.
     * <p/>
     * {@sample.xml ../../../doc/sugar-connector.xml.sample sugar:get-module-fields}
     * 
     * @param request a bean with module to obtains fields
     * @return {@link NewModuleFields} a list of fields in module
     */
    @Processor
    public com.sugarcrm.sugarcrm.NewModuleFields getModuleFields(@Optional @Default("#[payload]") org.mule.modules.sugarcrm.requests.GetModuleFieldsRequest request)
    {
        GetModuleFieldsRequestType realRequest = request.getBean();
        realRequest.setSession(sessionId);
        return getClient().getModuleFields(realRequest).getReturn();
    }

    /**
     * GetAvailableModules Processor.
     * <p>
     * Retrieves the list of modules available to the current user logged into the
     * system.
     * <p/>
     * {@sample.xml ../../../doc/sugar-connector.xml.sample sugar:get-available-modules}
     * 
     * @param request a bean with session id
     * @return {@link ModuleList} a list of availables modules
     */
    @Processor
    public com.sugarcrm.sugarcrm.ModuleList getAvailableModules(@Optional @Default("#[payload]") org.mule.modules.sugarcrm.requests.GetAvailableModulesRequest request)
    {
        GetAvailableModulesRequestType realRequest = request.getBean();
        realRequest.setSession(sessionId);
        return getClient().getAvailableModules(realRequest).getReturn();
    }

    /**
     * SearchByModule Processor.
     * <p>
     * Given a list of modules to search and a search string, 
     * return the id, module_name, along with the fields.
     * <br> 
     * Supported modules are Accounts, Bugs, Calls, Cases, Contacts, Leads, 
     * Opportunities, Projects, Project Tasks, and Quotes.
     * <p>
     * {@sample.xml ../../../doc/sugar-connector.xml.sample sugar:search-by-module}
     * 
     * @param request a bean with parameters
     * @return {@link ModuleList} a list of availables modules
     */
    @Processor
    public ReturnSearchResult searchByModule(@Optional @Default("#[payload]") org.mule.modules.sugarcrm.requests.SearchByModuleRequest request)
    {
        SearchByModuleRequestType realRequest = request.getBean();
        realRequest.setSession(sessionId);
        return getClient().searchByModule(realRequest).getReturn();
    }
    
    /**
     * SetRelationships Processor.
     * <p>
     * Sets multiple relationships between two SugarBeans.
     * 
     * {@sample.xml ../../../doc/sugar-connector.xml.sample sugar:set-relationships}
     * 
     * @param request a bean with parameters
     * @return {@link SetRelationshipsResponseType} a result operation
     */
    @Processor    
    public NewSetRelationshipListResult setRelationships(@Optional @Default("#[payload]") org.mule.modules.sugarcrm.requests.SetRelationshipsRequest request)
    {
        SetRelationshipsRequestType realRequest = request.getBean();
        realRequest.setSession(sessionId);
        return getClient().setRelationships(realRequest).getReturn();
    }

    /**
     * SetRelationship Processor.
     * <p>
     * Sets a single relationship between two SugarBeans.
     * 
     * {@sample.xml ../../../doc/sugar-connector.xml.sample sugar:set-relationship}
     * 
     * @param request a bean with parameters
     * @return {@link SetRelationshipResponseType} a result operation 
     */
    @Processor    
    public NewSetRelationshipListResult setRelationship(@Optional @Default("#[payload]") org.mule.modules.sugarcrm.requests.SetRelationshipRequest request)
    {
        SetRelationshipRequestType realRequest = request.getBean();
        realRequest.setSession(sessionId);
        return getClient().setRelationship(realRequest).getReturn();
    }
    
    /**
     * GetEntry Processor.
     * <p>
     * Retrieves a single SugarBean based on ID.
     * 
     * {@sample.xml ../../../doc/sugar-connector.xml.sample sugar:get-entry}
     * 
     * @param request a bean with parameters
     * @return {@link GetEntryResultVersion2} a result operation
     */
    @Processor
    public GetEntryResultVersion2 getEntry(@Optional @Default("#[payload]") org.mule.modules.sugarcrm.requests.GetEntryRequest request)
    {
        GetEntryRequestType realRequest = request.getBean();
        realRequest.setSession(sessionId);
        return getClient().getEntry(realRequest).getReturn();
    }

    /**
     * @return a client instance
     */
    private SugarCrmClient getClient()
    {
        if (client == null)
        {
            client = new SugarCrmClientSoap(endpoint);
        }
        return client;
    }

    /**
     * Setters/Getters
     */

    public String getEndpoint()
    {
        return endpoint;
    }

    public void setEndpoint(String endpoint)
    {
        this.endpoint = endpoint;
    }

    public void setClient(SugarCrmClient client)
    {
        this.client = client;
    }

}
