/**
 * Mule QuickBooks Connector
 *
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

/**
 * This file was automatically generated by the Mule Development Kit
 */
package org.mule.modules.quickbooks.online;

import javax.annotation.PostConstruct;
import javax.inject.Inject;

import oauth.signpost.exception.OAuthCommunicationException;
import oauth.signpost.exception.OAuthExpectationFailedException;
import oauth.signpost.exception.OAuthMessageSignerException;
import oauth.signpost.exception.OAuthNotAuthorizedException;
import oauth.signpost.signature.HmacSha1MessageSigner;
import org.apache.commons.lang.StringUtils;
import org.mule.api.MuleMessage;
import org.mule.api.annotations.Configurable;
import org.mule.api.annotations.Module;
import org.mule.api.annotations.Processor;
import org.mule.api.annotations.param.Default;
import org.mule.api.annotations.param.Optional;
import org.mule.api.annotations.param.OutboundHeaders;
import org.mule.api.config.MuleProperties;
import org.mule.api.store.ObjectDoesNotExistException;
import org.mule.api.store.ObjectStore;
import org.mule.api.store.ObjectStoreException;
import org.mule.modules.quickbooks.api.ObjectStoreHelper;
import org.mule.modules.quickbooks.api.exception.QuickBooksRuntimeException;
import org.mule.modules.quickbooks.api.model.AppMenuInformation;
import org.mule.modules.quickbooks.api.model.UserInformation;
import org.mule.modules.quickbooks.api.oauth.DefaultQuickbooksOAuthClient;
import org.mule.modules.quickbooks.api.oauth.OAuthCredentials;
import org.mule.modules.quickbooks.api.openid.DefaultOpenIDClient;
import org.mule.modules.quickbooks.api.openid.OpenIDCredentials;
import org.mule.modules.quickbooks.online.api.DefaultQuickBooksOnlineClient;
import org.mule.modules.quickbooks.online.api.QuickBooksOnlineClient;
import org.mule.modules.quickbooks.online.schema.Account;
import org.mule.modules.quickbooks.online.schema.Bill;
import org.mule.modules.quickbooks.online.schema.BillPayment;
import org.mule.modules.quickbooks.online.schema.CashPurchase;
import org.mule.modules.quickbooks.online.schema.Check;
import org.mule.modules.quickbooks.online.schema.CreditCardCharge;
import org.mule.modules.quickbooks.online.schema.Customer;
import org.mule.modules.quickbooks.online.schema.Estimate;
import org.mule.modules.quickbooks.online.schema.IdType;
import org.mule.modules.quickbooks.online.schema.Invoice;
import org.mule.modules.quickbooks.online.schema.Item;
import org.mule.modules.quickbooks.online.schema.Payment;
import org.mule.modules.quickbooks.online.schema.PaymentMethod;
import org.mule.modules.quickbooks.online.schema.SalesReceipt;
import org.mule.modules.quickbooks.online.schema.SalesTerm;
import org.mule.modules.quickbooks.online.schema.Vendor;
import org.openid4java.message.MessageException;

import java.util.List;
import java.util.Map;
/**
 * QuickBooks software provides an interface that allows you to use forms such as checks, deposit slips and invoices,
 * making the accounting process more comfortable for the average business owner or manager. By using the built-in
 * functions that pertain to your business, you are able to perform your company accounting by simply recording your
 * vendor activities, customer activities, banking transactions, payroll checks and taxes. QuickBooks handles the
 * accounting portion of each transaction behind the scenes.
 *
 * Read more: QuickBooks Accounting Tutorial | eHow.com http://www.ehow.com/way_5462311_quickbooks-accounting-tutorial.html#ixzz1csaydwxl
 * @author MuleSoft, inc.
 */
@SuppressWarnings("unused")
@Module(name = "quickbooks", schemaVersion= "4.0", friendlyName = "Quickbooks Online")
public class QuickBooksModule
{
    /**
     * API Key
     */
    @Configurable
    private String consumerKey;

    /**
     * API Secret
     */
    @Configurable
    private String consumerSecret;

    /**
     * Object store reference
     */
    @Configurable
    @Optional
    @Default(MuleProperties.DEFAULT_USER_OBJECT_STORE_NAME)
    private ObjectStore objectStore;

    /**
     * Object store helper
     */
    private ObjectStoreHelper objectStoreHelper;

    /**
     * Quick-Books client. By default uses DefaultQuickBooksOnlineClient class.
     */
    @Configurable
    @Optional
    private QuickBooksOnlineClient client;

    /**
     * The base uri of the quickbooks endpoint,
     * used to fetch the company uri. 
     * 
     * Quickbooks connector will first use this uri and the realmId to
     * get a second uri, called company uri, 
     * which is the actual quickbooks endpoint for the connector 
     *  
     */
    @Optional
    @Default("https://qbo.intuit.com/qbo1/rest/user/v2")
    @Configurable
    private String baseUri;

    /**
     * Prefix used for storing credentials in ObjectStore. It will be concatenated to the access token identifier.
     * <p>E.g. prefix: "qb_", user identifier (realmId): "12345", key for object store "qb_12345"</p>
     */
    @Configurable
    @Optional
    private String accessTokenIdentifierPrefix;

    /**
     * Specifies if the OpenID response will be verified. By default it is true.
     */
    @Configurable
    @Optional
    @Default("true")
    private boolean verifyOpenIdResponse;

    
    /**
     * Creates an Account.
     * The Account object represents the accounts that you keep to track your business.
     * Account is a component of a chart of accounts, and is part of a ledger.
     * You can use Account to record the total monetary amount that is allocated for a specific use.
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Account">Account Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:create-account}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param account   The Quickbooks account to be created.
     * @return The created Account.
     */
    @Processor
    public Account createAccount(String accessTokenId,
                                 @Optional @Default("#[payload]") Account account)
    {
        
        return client.create(getAccessTokenInformation(accessTokenId), account);
    }
    
    /**
     * Creates a Bill.
     * The Bill object represents an expense to the business.
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Bill">Bill Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:create-bill}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param bill The bill to be created
     * @return The created Bill.
     */
    @Processor
    public Bill createBill(String accessTokenId,
                           @Optional @Default("#[payload]") Bill bill)
    {
        return client.create(getAccessTokenInformation(accessTokenId), bill);
    }
    
    /**
     * Creates a BillPayment.
     * BillPayment represents the financial transaction of payment of bills that the business owner receives
     * from a vendor for goods or services purchased from the vendor.
     * QBO supports bill payments through a credit card or a bank account.
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/BillPayment">BillPayment Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:create-bill-payment}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param billPayment The bill payment object
     * @return The created BillPayment.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public BillPayment createBillPayment(String accessTokenId,
                                         @Optional @Default("#[payload]") BillPayment billPayment)
    {    
        return client.create(getAccessTokenInformation(accessTokenId), billPayment);
    }
    
    /**
     * Creates a CashPurchase.
     * CashPurchase represents an expense to the business as a cash transaction.
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/CashPurchase">CashPurchase Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:create-cash-purchase}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param cashPurchase The cash purchase to be created
     * @return The created CashPurchase.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public CashPurchase createCashPurchase(String accessTokenId,
                                           @Optional @Default("#[payload]") CashPurchase cashPurchase)
    {
        return client.create(getAccessTokenInformation(accessTokenId), cashPurchase);
    }
    
    /**
     * Creates a Check.
     * The Check object represents an expense to the business paid as a check transaction.
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Check">Check Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:create-check}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param check The check to be created
     * @return The created Check.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Check createCheck(String accessTokenId,
                             @Optional @Default("#[payload]") Check check)
    {
        return client.create(getAccessTokenInformation(accessTokenId), check);
    }
    
    /**
     * Creates a CreditCardCharge.
     * The CreditCardCharge object represents an expense to the business as a credit card charge 
     * transaction. CreditCardCharge must have the total expense equal to the total expense of 
     * line items.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/CreditCardCharge">CreditCardCharge Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:create-credit-card-charge}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param creditCardCharge The credit card charge to be created
     * @return The created CreditCardCharge.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public CreditCardCharge createCreditCardCharge(String accessTokenId,
                                                   @Optional @Default("#[payload]") CreditCardCharge creditCardCharge)
    {
        return client.create(getAccessTokenInformation(accessTokenId), creditCardCharge);
    }

    /**
     * Creates a Customer.
     * The Customer object represents the consumer of the service or the product that your business offers. 
     * QBO allows categorizing the customers in a way that is meaningful to the business.
     *
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Customer">Customer Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:create-customer}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param customer The customer to be created
     * @return The created Customer.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Customer createCustomer(String accessTokenId,
                                   @Optional @Default("#[payload]") Customer customer)
    {
        if(customer.getPaymentMethodId() != null && (customer.getPaymentMethodId().getValue().isEmpty() 
                || StringUtils.isEmpty(customer.getPaymentMethodId().getValue())))
        {
            customer.setPaymentMethodId(new IdType());
        }
        
        return client.create(getAccessTokenInformation(accessTokenId), customer);
    }
    
    /**
     * Creates an Estimate.
     * The Estimate object represents a proposal for a financial transaction from a business to a customer 
     * for goods or services proposed to be sold, including proposed pricing. It is also known as quote.
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Estimate">Estimate Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:create-estimate}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param estimate The estimate to be created
     * @return The created Estimate.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Estimate createEstimate(String accessTokenId,
                                   @Optional @Default("#[payload]") Estimate estimate)
    {
        return client.create(getAccessTokenInformation(accessTokenId), estimate);
    }
    
    /**
     * Creates an Invoice.
     * The Invoice object represents an invoice to a customer. Invoice could be based on salesterm 
     * with invoice and due dates for payment. Invoice supports tax, but as of now shipping charges 
     * are not supported. Invoice can be printed and emailed to a customer.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Invoice">Invoice Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:create-invoice}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param invoice The invoice to be created
     * @return The created Invoice.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Invoice createInvoice(String accessTokenId,
                                 @Optional @Default("#[payload]") Invoice invoice)
    {
        return client.create(getAccessTokenInformation(accessTokenId), invoice);
    }
    
    /**
     * Creates an Item.
     * The Item object represents any product or service that is sold or purchased. Inventory items 
     * are not currently supported.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Item">Item Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:create-item}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param item The item to be created
     * @return The created Item.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Item createItem(String accessTokenId,
                           @Optional @Default("#[payload]") Item item)
    {
        return client.create(getAccessTokenInformation(accessTokenId), item);
    }
    
    /**
     * Creates a Payment.
     * The Payment object  represents the financial transaction that signifies a payment from a customer 
     * for one or more sales transactions.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Payment">Payment Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:create-payment}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param payment The payment to be created
     * @return The created Payment.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Payment createPayment(String accessTokenId,
                                 @Optional @Default("#[payload]") Payment payment)
    {
        return client.create(getAccessTokenInformation(accessTokenId), payment);
    }
    
    /**
     * Creates a PaymentMethod.
     * PaymentMethod represents the method of payment for a transaction. It can be a credit card 
     * payment type or a non-credit card payment type.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/PaymentMethod">PaymentMethod Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:create-payment-method}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param paymentMethod The payment method to be created
     * @return The created PaymentMethod.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public PaymentMethod createPaymentMethod(String accessTokenId,
                                             @Optional @Default("#[payload]") PaymentMethod paymentMethod)
    {
        return client.create(getAccessTokenInformation(accessTokenId), paymentMethod);
    }
    
    /**
     * Creates a SalesReceipt.
     * The Payment object  represents the financial transaction that signifies a payment from a customer 
     * for one or more sales transactions.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/SalesReceipt">SalesReceipt Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:create-sales-receipt}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param salesReceipt The sales receipt to be created
     * @return The created SalesReceipt.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public SalesReceipt createSalesReceipt(String accessTokenId,
                                           @Optional @Default("#[payload]") SalesReceipt salesReceipt)
    {
        return client.create(getAccessTokenInformation(accessTokenId), salesReceipt);
    }
    
    /**
     * Creates a SalesTerm.
     * The SalesTerm object  represents the terms under which a sale is made. SalesTerm is typically 
     * expressed in the form of days due after the goods are received. There is an optional discount 
     * part of the sales term, where a discount of total amount can automatically be applied if 
     * payment is made within a few days of the stipulated time.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/SalesTerm">SalesTerm Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:create-sales-term}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param salesTerm The sales term to be created
     * @return The created SalesTerm.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public SalesTerm createSalesTerm(String accessTokenId,
                                     @Optional @Default("#[payload]") SalesTerm salesTerm)
    {
        
        return client.create(getAccessTokenInformation(accessTokenId), salesTerm);
    }
    
    /**
     * Creates a Vendor.
     * The Vendor object represents the buyer from whom you purchase any service or product 
     * for your organization.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Vendor">Vendor Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:create-vendor}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param vendor The vendor to be created
     * @return The created Vendor.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Vendor createVendor(String accessTokenId,
                               @Optional @Default("#[payload]") Vendor vendor)
    {
        return client.create(getAccessTokenInformation(accessTokenId), vendor);
    }
    
    /**
     * Retrieve objects by ID.
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:get-object}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param type EntityType of the object.
     * @param id Id which is assigned by Data Services when the object is created.
     * @return The object.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Object getObject(String accessTokenId,
                            OnlineEntityType type,
                            @Optional @Default("#[payload]") IdType id)
    {
        return client.getObject(getAccessTokenInformation(accessTokenId), type, id);
    }

    /**
     * Updates an Account
     * The Account object represents the accounts that you keep to track your business.
     * Account is a component of a chart of accounts, and is part of a ledger.
     * You can use Account to record the total monetary amount that is allocated for a specific use.
     * 
     * Specify all the parameters for the object, not just the new or changed elements.
     * If you omit an element, it is removed from the object by the update operation.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Account">Account Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:update-account}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param account The account to be updated
     * @return The updated Account.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Account updateAccount(String accessTokenId,
                                 @Optional @Default("#[payload]") Account account)
    {   
        return client.update(getAccessTokenInformation(accessTokenId), OnlineEntityType.ACCOUNT, account);
    }
    
    /**
     * Updates a Bill
     * The Bill object represents an expense to the business. 
     * 
     * Specify all the parameters for the object, not just the new or changed elements.
     * If you omit an element, it is removed from the object by the update operation.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Bill">Bill Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:update-bill}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param bill The bill to be updated
     * @return The updated Bill.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Bill updateBill(String accessTokenId,
                           @Optional @Default("#[payload]") Bill bill)
    {
        return client.update(getAccessTokenInformation(accessTokenId), OnlineEntityType.BILL, bill);
    }
    
    /**
     * Updates a BillPayment
     * BillPayment represents the financial transaction of payment of bills that the business owner receives
     * from a vendor for goods or services purchased from the vendor.
     * QBO supports bill payments through a credit card or a bank account.
     * 
     * Specify all the parameters for the object, not just the new or changed elements.
     * If you omit an element, it is removed from the object by the update operation.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/BillPayment">BillPayment Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:update-bill-payment}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param billPayment Bill payment to be updated
     * @return The updated BillPayment.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public BillPayment updateBillPayment(String accessTokenId,
                                         @Optional @Default("#[payload]") BillPayment billPayment)
    {    
        return client.update(getAccessTokenInformation(accessTokenId), OnlineEntityType.BILLPAYMENT, billPayment);
    }
    
    /**
     * Updates a CashPurchase.
     * CashPurchase represents an expense to the business as a cash transaction.
     * 
     * Specify all the parameters for the object, not just the new or changed elements.
     * If you omit an element, it is removed from the object by the update operation.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/CashPurchase">CashPurchase Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:update-cash-purchase}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param cashPurchase The cash purchase to be updated
     * @return The updated CashPurchase.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public CashPurchase updateCashPurchase(String accessTokenId,
                                           @Optional @Default("#[payload]") CashPurchase cashPurchase)
    {
        return client.update(getAccessTokenInformation(accessTokenId), OnlineEntityType.CASHPURCHASE, cashPurchase);
    }
    
    /**
     * Updates a Check.
     * The Check object represents an expense to the business paid as a check transaction.
     * 
     * Specify all the parameters for the object, not just the new or changed elements.
     * If you omit an element, it is removed from the object by the update operation.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Check">Check Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:update-check}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param check The check to be updated
     * @return The updated Check.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Check updateCheck(String accessTokenId,
                             @Optional @Default("#[payload]") Check check)
    {
        return client.update(getAccessTokenInformation(accessTokenId), OnlineEntityType.CHECK, check);
    }
    
    /**
     * Updates a CreditCardCharge.
     * The CreditCardCharge object represents an expense to the business as a credit card charge 
     * transaction. CreditCardCharge must have the total expense equal to the total expense of 
     * line items.
     * 
     * Specify all the parameters for the object, not just the new or changed elements.
     * If you omit an element, it is removed from the object by the update operation.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/CreditCardCharge">CreditCardCharge Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:update-credit-card-charge}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param creditCardCharge The credit card charge to be updated
     * @return The updated CreditCardCharge.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public CreditCardCharge updateCreditCardCharge(String accessTokenId,
                                                   @Optional @Default("#[payload]") CreditCardCharge creditCardCharge)
    {
        return client.update(getAccessTokenInformation(accessTokenId), OnlineEntityType.CREDITCARDCHARGE,
                creditCardCharge);
    }

    /**
     * Updates a Customer.
     * The Customer object represents the consumer of the service or the product that your business offers. 
     * QBO allows categorizing the customers in a way that is meaningful to the business.
     * 
     * Specify all the parameters for the object, not just the new or changed elements.
     * If you omit an element, it is removed from the object by the update operation.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Customer">Customer Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:update-customer}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param customer The customer to be updated
     * @return The updated Customer.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Customer updateCustomer(String accessTokenId,
                                   @Optional @Default("#[payload]") Customer customer)
    {
        if(customer.getPaymentMethodId() != null && (customer.getPaymentMethodId().getValue().isEmpty() 
                || StringUtils.isEmpty(customer.getPaymentMethodId().getValue())))
        {
            customer.setPaymentMethodId(new IdType());
        }

        return client.update(getAccessTokenInformation(accessTokenId), OnlineEntityType.CUSTOMER, customer);
    }
    
    /**
     * Updates an Estimate.
     * The Estimate object represents a proposal for a financial transaction from a business to a customer 
     * for goods or services proposed to be sold, including proposed pricing. It is also known as quote.
     * 
     * Specify all the parameters for the object, not just the new or changed elements.
     * If you omit an element, it is removed from the object by the update operation.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Estimate">Estimate Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:update-estimate}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param estimate The estimate to be updated
     * @return The updated Estimate.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Estimate updateEstimate(String accessTokenId,
                                   @Optional @Default("#[payload]") Estimate estimate)
    {
        return client.update(getAccessTokenInformation(accessTokenId), OnlineEntityType.ESTIMATE, estimate);
    }
    
    /**
     * Updates an Invoice.
     * The Invoice object represents an invoice to a customer. Invoice could be based on salesterm 
     * with invoice and due dates for payment. Invoice supports tax, but as of now shipping charges 
     * are not supported. Invoice can be printed and emailed to a customer.
     * 
     * Specify all the parameters for the object, not just the new or changed elements.
     * If you omit an element, it is removed from the object by the update operation.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Invoice">Invoice Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:update-invoice}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param invoice The invoice to be updated
     * @return The updated Invoice.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Invoice updateInvoice(String accessTokenId,
                                 @Optional @Default("#[payload]") Invoice invoice)
    {
        return client.update(getAccessTokenInformation(accessTokenId), OnlineEntityType.INVOICE, invoice);
    }
    
    /**
     * Updates an Item.
     * The Item object represents any product or service that is sold or purchased. Inventory items 
     * are not currently supported.
     * 
     * Specify all the parameters for the object, not just the new or changed elements.
     * If you omit an element, it is removed from the object by the update operation.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Item">Item Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:update-item}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param item The item to be updated
     * @return The updated Item.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Item updateItem(String accessTokenId,
                           @Optional @Default("#[payload]") Item item)
    {
        return client.update(getAccessTokenInformation(accessTokenId), OnlineEntityType.ITEM, item);
    }
    
    /**
     * Updates a Payment.
     * The Payment object  represents the financial transaction that signifies a payment from a customer 
     * for one or more sales transactions.
     * 
     * Specify all the parameters for the object, not just the new or changed elements.
     * If you omit an element, it is removed from the object by the update operation.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Payment">Payment Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:update-payment}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param payment The payment to be updated
     * @return The updated Payment.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Payment updatePayment(String accessTokenId,
                                 @Optional @Default("#[payload]") Payment payment)
    {
        return client.update(getAccessTokenInformation(accessTokenId), OnlineEntityType.PAYMENT, payment);
    }
    
    /**
     * Updates a PaymentMethod.
     * PaymentMethod represents the method of payment for a transaction. It can be a credit card 
     * payment type or a non-credit card payment type.
     * 
     * Specify all the parameters for the object, not just the new or changed elements.
     * If you omit an element, it is removed from the object by the update operation.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/PaymentMethod">PaymentMethod Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:update-payment-method}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param paymentMethod The payment method to be updated
     * @return The updated PaymentMethod.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public PaymentMethod updatePaymentMethod(String accessTokenId,
                                             @Optional @Default("#[payload]") PaymentMethod paymentMethod)
    {
        return client.update(getAccessTokenInformation(accessTokenId), OnlineEntityType.PAYMENTMETHOD, paymentMethod);
    }
    
    /**
     * Updates a SalesReceipt.
     * The Payment object  represents the financial transaction that signifies a payment from a customer 
     * for one or more sales transactions.
     * 
     * Specify all the parameters for the object, not just the new or changed elements.
     * If you omit an element, it is removed from the object by the update operation.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/SalesReceipt">SalesReceipt Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:update-sales-receipt}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param salesReceipt The sales receipt
     * @return The updated SalesReceipt.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public SalesReceipt updateSalesReceipt(String accessTokenId,
                                           @Optional @Default("#[payload]") SalesReceipt salesReceipt)
    {
        return client.update(getAccessTokenInformation(accessTokenId), OnlineEntityType.SALESRECEIPT, salesReceipt);
    }
    
    /**
     * Updates a SalesTerm.
     * The SalesTerm object  represents the terms under which a sale is made. SalesTerm is typically 
     * expressed in the form of days due after the goods are received. There is an optional discount 
     * part of the sales term, where a discount of total amount can automatically be applied if 
     * payment is made within a few days of the stipulated time.
     * 
     * Specify all the parameters for the object, not just the new or changed elements.
     * If you omit an element, it is removed from the object by the update operation.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/SalesTerm">SalesTerm Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:update-sales-term}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param salesTerm The sales term to be updated
     * @return The updated SalesTerm.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public SalesTerm updateSalesTerm(String accessTokenId,
                                     @Optional @Default("#[payload]") SalesTerm salesTerm)
    {
        
        return client.update(getAccessTokenInformation(accessTokenId), OnlineEntityType.SALESTERM, salesTerm);
    }
    
    /**
     * Updates a Vendor.
     * The Vendor object represents the buyer from whom you purchase any service or product 
     * for your organization.
     * 
     * Specify all the parameters for the object, not just the new or changed elements.
     * If you omit an element, it is removed from the object by the update operation.
     * 
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Vendor">Vendor Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:update-vendor}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param vendor The vendor to be updated
     * @return The updated Vendor.
     */
    @Processor
    public Vendor updateVendor(String accessTokenId,
                               @Optional @Default("#[payload]") Vendor vendor)
    {
        return client.update(getAccessTokenInformation(accessTokenId), OnlineEntityType.VENDOR, vendor);
    }
    
    /**
     * Deletes an object.
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:delete-object}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param type EntityType of the object.
     * @param id Id which is assigned by Data Services when the object is created.
     * @param syncToken Integer that indicates how many times the object has been updated.
     *                  Before performing the update, Data Services verifies that the SyncToken in the
     *                  request has the same value as the SyncToken in the Data Service's repository.
     */
    @Processor
    public void deleteObject(String accessTokenId,
                             OnlineEntityType type, 
                             @Optional @Default("#[payload]") IdType id, 
                             @Optional String syncToken)
    {
        client.deleteObject(getAccessTokenInformation(accessTokenId), type, id, syncToken);
    }

    /**
     * Lazily retrieves Objects
     *
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/Vendor">Vendor Specification</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:find-objects}
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:find-objects2}
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:find-objects3}
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:find-objects4}
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:find-objects5}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param type EntityType of the object.
     * @param queryFilter String with a filter format (see details). Each type of object to be 
     *                    retrieved, has a list of attributes for which it can be filtered (See this 
     *                    list following the link in the details of the documentation of the create
     *                    or update method of that object).
     * @param querySort String with a sort format (see details). Each type of object to be 
     *                    retrieved, has a list of attributes for which it can be sorted (See this 
     *                    list following the link in the details of the documentation of the create
     *                    or update method of that object).
     * @return Iterable of the objects to be retrieved.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Iterable findObjects(String accessTokenId,
                                OnlineEntityType type, 
                                @Optional String queryFilter,
                                @Optional String querySort)
    {
        return client.findObjects(getAccessTokenInformation(accessTokenId), type, queryFilter, querySort);
    }
    
    /**
     * Gets all of the transactions and objects that have been deleted on the Data Services server
     *
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/ChangeDataDeleted">ChangeDataDeleted</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:change-data-deleted}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @param queryFilter String with a filter format (see details). Each type of object to be 
     *                    retrieved, has a list of attributes for which it can be filtered (See this 
     *                    list following the link in the details of the documentation of the create
     *                    or update method of that object).
     * @param querySort String with a sort format (see details). Each type of object to be 
     *                    retrieved, has a list of attributes for which it can be sorted (See this 
     *                    list following the link in the details of the documentation of the create
     *                    or update method of that object).
     * @return Iterable of the objects to be retrieved.
     * 
     * @throws QuickBooksRuntimeException when there is a problem with the server. It has a code 
     *         and a message provided by quickbooks about the error.
     */
    @Processor
    public Iterable changeDataDeleted(String accessTokenId,
                                @Optional String queryFilter,
                                @Optional String querySort)
    {
        return client.findObjects(getAccessTokenInformation(accessTokenId), OnlineEntityType.CHANGEDATADELETED,
                queryFilter, querySort);
    }
    
    /**
     * Returns current user information such as first name, last name, and email address.
     *
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/
     * 0025_Intuit_Anywhere/0060_Reference/Current_User_API">CurrentUserAPI</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:get-current-user}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @return current user information
     * 
     */
    @Processor
    public UserInformation getCurrentUser(String accessTokenId)
    {
        return client.getCurrentUserInformation(getAccessTokenInformation(accessTokenId));
    }
    
    /**
     * Returns company metadata information.
     *
     * For details see: 
     * <a href="https://ipp.developer.intuit.com/0010_Intuit_Partner_Platform/0050_Data_Services/
     * 0400_QuickBooks_Online/CompanyMetaData">CompanyMetaData</a>
     * 
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:get-company-metadata}
     *
     * @param accessTokenId identifier for QuickBooks credentials.
     * @return company metadata
     * 
     */
    @Processor
    public Object getCompanyMetadata(String accessTokenId)
    {
        return client.get(getAccessTokenInformation(accessTokenId),
                OnlineEntityType.COMPANY_METADATA);
    }

    /**
     * Authorize an user using OAuth1.0a
     *
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:auth-user}
     *
     * @param requestTokenUrl requestTokenUrl
     * @param accessTokenUrl accessTokenUrl
     * @param authorizationUrl authorizationUrl
     * @param callbackUrl callbackUrl for OAuth service
     * @param requestTokenId Optional value for identifying the requestToken. If it is not passed the client will use a UUID
     * @param headers Outbound headers
     * @return Authorize Url
     * @throws ObjectStoreException from the Object Store instance
     * @throws oauth.signpost.exception.OAuthCommunicationException requesting to OAuth provider
     * @throws oauth.signpost.exception.OAuthExpectationFailedException requesting to OAuth provider
     * @throws oauth.signpost.exception.OAuthNotAuthorizedException requesting to OAuth provider
     * @throws oauth.signpost.exception.OAuthMessageSignerException requesting to OAuth provider
     */
    @Processor
    public String authUser(String requestTokenUrl, String accessTokenUrl, String authorizationUrl, String callbackUrl,
                           @Optional String requestTokenId, @OutboundHeaders Map<String, Object> headers)
            throws OAuthMessageSignerException, OAuthNotAuthorizedException,
            OAuthExpectationFailedException, OAuthCommunicationException, ObjectStoreException
    {
        String authUrl = new DefaultQuickbooksOAuthClient(getConsumerKey(), getConsumerSecret(), getObjectStore()).
                authorize(requestTokenUrl, accessTokenUrl, authorizationUrl,
                        callbackUrl, requestTokenId, new HmacSha1MessageSigner());

        headers.put("Location", authUrl);
        headers.put("http.status", "302");
        return authUrl;
    }

    /**
     * Extract accessToken
     *
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:get-access-token}
     *
     * @param apiUrl API URL
     * @param verifier OAuth verifier. It comes within the callback.
     * The default value is "#[header:inbound:oauth_verifier]"
     * @param requestTokenId id used for identifying the authorized request token. It comes within the callback.
     * By default the query string parameter is userId
     * @param userIdentifier id used for store the accessToken in the Object Store.
     *      By default the value is the realmId
     * @return credentials user credentials
     * @throws ObjectStoreException from the object store instance
     * @throws OAuthCommunicationException requesting to OAuth provider
     * @throws OAuthExpectationFailedException requesting to OAuth provider
     * @throws OAuthNotAuthorizedException requesting to OAuth provider
     * @throws OAuthMessageSignerException requesting to OAuth provider
     */
    @Processor
    public OAuthCredentials getAccessToken(@Optional String apiUrl,
                                           @Optional @Default("#[message.inboundProperties.oauth_verifier]") String verifier,
                                           @Optional @Default("#[message.inboundProperties.userId]") String requestTokenId,
                                           @Optional @Default("#[message.inboundProperties.realmId]") String userIdentifier)
            throws OAuthMessageSignerException, OAuthNotAuthorizedException,
            OAuthExpectationFailedException, OAuthCommunicationException, ObjectStoreException
    {
        OAuthCredentials credentials = new DefaultQuickbooksOAuthClient(getConsumerKey(), getConsumerSecret(),
                getObjectStore()).getAccessToken(verifier, requestTokenId, new HmacSha1MessageSigner());
        credentials.setUserId(userIdentifier);
        credentials.setRealmId(userIdentifier);

        if (StringUtils.isNotBlank(apiUrl)) {
            credentials.setBaseUri(apiUrl);
        }
        else {
            credentials.setBaseUri(client.getCompanyBaseUri(credentials));
        }

        //Use the prefix if it is defined in the config
        if (StringUtils.isNotEmpty(getAccessTokenIdentifierPrefix())) {
            userIdentifier = getAccessTokenIdentifierPrefix() + userIdentifier;
        }

        getObjectStoreHelper().store(userIdentifier, credentials, true);

        return credentials;
    }

    /**
     * Initializes OpenID process
     *
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:open-id-initialize}
     *
     * @param providerUrl OpenID provider url
     * @param callbackUrl OpenID callbackUrl. It has to point to an endpoint callback to process the response
     * @param headers openId response headers
     *
     * @return url to redirect the user
     * @throws ObjectStoreException if the operation cannot store the OpenIDManagers
     *
     */
    @Processor
    public String openIdInitialize(@Optional @Default("https://openid.intuit.com/OpenId/Provider") String providerUrl,
                                   String callbackUrl,
                                   @OutboundHeaders Map<String, Object> headers) throws ObjectStoreException {
        String url = new DefaultOpenIDClient(getObjectStoreHelper()).initialize(
                providerUrl, callbackUrl, getVerifyOpenIdResponse());

        headers.put("Location", url);
        headers.put("http.status", "302");
        return url;
    }

    /**
     * Verify response from Intuit
     *
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:verify-open-id}
     *
     * @param muleMessage injected MuleMessage
     * @param receivingUrl url from OpenID provider.
     *                     <p>If it is not provided the processor will extract it from the
     *                     Mule message</p>
     * @param responseParameters response parameters from Intuit. It process a map<string, string> with all the OpenID
     *                           attributes sent from Intuit.
     *
     * @return OpenID credentials for the authenticated user
     * @throws MessageException from OpenID manager
     * @throws ObjectStoreException if the operation cannot retrieve OpenID managers from ObjectStore
     *
     */
    @Processor
    @Inject
    public OpenIDCredentials verifyOpenId(MuleMessage muleMessage, @Optional String receivingUrl,
            @Optional @Default("#[header:INBOUND:http.query.params]") Map<String, String> responseParameters)
            throws MessageException, ObjectStoreException {

        //Build receivingUrl
        if (StringUtils.isEmpty(receivingUrl)) {
            receivingUrl = String.format("%s%s%s", muleMessage.getInboundProperty("http.context.uri"), "?",
                    muleMessage.getInboundProperty("http.query.string"));
        }

        return new DefaultOpenIDClient(getObjectStoreHelper()).verifyOpenIDFromIntuit(
                receivingUrl, responseParameters, getVerifyOpenIdResponse());
    }

    /**
     * Gets blueDot menu information from Intuit
     *
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:get-blue-dot-information}
     *
     *
     *
     * @param accessTokenId credentials identifier for the user information to be requested
     * @param regex Regex for extracting the information
     *              <p>The regex has to extract the information in this way:</p>
     *              <p>match[0]: "appId,appName,contextArea"</p>
     *              <p>match[1]: "logoImageUrl"</p>
     *              <p>The method will split the application information to generate the @link{AppMenuInformation} object</p>
     *
     * @return list with AppMenu information from Intuit
     *
     */
    @Processor
    public List<AppMenuInformation> getBlueDotInformation(String accessTokenId,
                                                          @Optional @Default("intuitPlatformOpenOtherApp\\((.+?)\\)\" style=\'background-image: url\\((.+?)\\)") String regex)
    {
        return client.getBlueDotInformation(getAccessTokenInformation(accessTokenId), regex);
    }

    /**
     * Invalidates the OAuth access token in the request, thereby disconnecting the user from QuickBooks for this app.
     *
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:disconnect}
     *
     *
     * @param accessTokenId credentials identifier for the user information to be requested
     *
     * @return true if the user was disconnect from QBO
     *
     */
    @Processor
    public boolean disconnect(String accessTokenId)
    {
        return client.disconnect(getAccessTokenInformation(accessTokenId));
    }

    /**
     * Generates a new OAuth access token and invalidates the OAuth access token used in the request,
     * thereby extending the life span by six months. Because accessing QuickBooks data requires a valid access token,
     * when the OAuth access token is renewed, your app can continue to access the user's QuickBooks company data.
     *
     * {@sample.xml ../../../doc/mule-module-quick-books-online.xml.sample quickbooks:reconnect}
     *
     *
     * @param accessTokenId credentials identifier for the user information to be requested
     * @throws ObjectStoreException if the credentials store failed
     *
     */
    @Processor
    public void reconnect(String accessTokenId) throws ObjectStoreException {
        OAuthCredentials creds = client.reconnect(getAccessTokenInformation(accessTokenId));

        //Stores new credentials
        getObjectStoreHelper().store(creds.getUserId(), creds, true);
    }
    
    /**
     * This method retrieves the accessTokenInformation from the object store instance
     * @return OAuthCredentials AuthToken and AuthTokenSecret
     */
    private OAuthCredentials getAccessTokenInformation(String accessTokenIdentifier) {

        //Check if there is a prefix in the config
        if(StringUtils.isNotEmpty(getAccessTokenIdentifierPrefix())) {
            accessTokenIdentifier = getAccessTokenIdentifierPrefix() + accessTokenIdentifier;
        }

        try {
            return (OAuthCredentials) objectStoreHelper.retrieve(accessTokenIdentifier);
        } catch (ObjectDoesNotExistException e) {
            throw new QuickBooksRuntimeException(String.format("The user token could not be retrieved from the " +
                    "Object Store using the key %s. It seems the user is not authenticated, " +
                    "please start OAuth dance again", accessTokenIdentifier));
        } catch (ObjectStoreException e) {
            throw new QuickBooksRuntimeException("The user token could not be retrieved: " + e.getMessage());
        }
    }

    /**
     * Create OAuthCredentials object
     * @param accessToken user accessToken
     * @param accessTokenSecret user accessTokenSecret
     * @return credentials
     */
    private OAuthCredentials createCredentials(String accessToken, String accessTokenSecret) {
        return new OAuthCredentials(accessToken, accessTokenSecret);
    }

    /**
     *  Generates the QuickbooksOnlineClient
     */
    @PostConstruct
    public void init()
    {
        if (client == null )
        {
            //TODO: is it necessary the apiKey?
            client = new DefaultQuickBooksOnlineClient(baseUri, consumerKey, consumerSecret, "");
            setObjectStoreHelper(new ObjectStoreHelper(objectStore));
        }
    }
    
    public void setBaseUri(String baseUri)
    {
        this.baseUri = baseUri;
    }
    
    public String getBaseUri()
    {
        return baseUri;
    }

    public QuickBooksOnlineClient getClient() {
        return client;
    }

    public void setClient(QuickBooksOnlineClient client) {
        this.client = client;
    }

    public String getConsumerSecret() {
        return consumerSecret;
    }

    public void setConsumerSecret(String consumerSecret) {
        this.consumerSecret = consumerSecret;
    }

    public String getConsumerKey() {
        return consumerKey;
    }

    public void setConsumerKey(String consumerKey) {
        this.consumerKey = consumerKey;
    }


    public ObjectStore getObjectStore() {
        return objectStore;
    }

    public void setObjectStore(ObjectStore objectStore) {
        this.objectStore = objectStore;
    }

    public ObjectStoreHelper getObjectStoreHelper() {
        return objectStoreHelper;
    }

    public void setObjectStoreHelper(ObjectStoreHelper objectStoreHelper) {
        this.objectStoreHelper = objectStoreHelper;
    }

    public String getAccessTokenIdentifierPrefix() {
        return accessTokenIdentifierPrefix;
    }

    public void setAccessTokenIdentifierPrefix(String accessTokenIdentifierPrefix) {
        this.accessTokenIdentifierPrefix = accessTokenIdentifierPrefix;
    }

    public boolean getVerifyOpenIdResponse() {
        return verifyOpenIdResponse;
    }

    public void setVerifyOpenIdResponse(boolean verifyOpenIdResponse) {
        this.verifyOpenIdResponse = verifyOpenIdResponse;
    }
}
