/**
 * (c) 2003-2012 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */

package org.mule.module.netsuite.api;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.validation.constraints.NotNull;

import org.apache.commons.lang.Validate;
import org.mule.module.netsuite.api.model.entity.RecordId;
import org.mule.module.netsuite.api.model.entity.RecordReference;
import org.mule.module.netsuite.api.model.expression.date.DateExpression;
import org.mule.module.netsuite.api.model.expression.filter.parser.FilterExpressionParser;
import org.mule.module.netsuite.api.paging.AsyncRecordSearchIterable;
import org.mule.module.netsuite.api.paging.RecordSearchIterable;
import org.mule.module.netsuite.api.paging.SavedRecordSearchIterable;
import org.mule.modules.utils.date.XmlGregorianCalendars;
import org.mule.modules.utils.mom.JaxbMapObjectMappers;

import com.netsuite.webservices.platform.core_2010_2.AsyncStatusResult;
import com.netsuite.webservices.platform.core_2010_2.AttachBasicReference;
import com.netsuite.webservices.platform.core_2010_2.AttachContactReference;
import com.netsuite.webservices.platform.core_2010_2.BaseRef;
import com.netsuite.webservices.platform.core_2010_2.BudgetExchangeRateFilter;
import com.netsuite.webservices.platform.core_2010_2.ConsolidatedExchangeRateFilter;
import com.netsuite.webservices.platform.core_2010_2.CustomizationType;
import com.netsuite.webservices.platform.core_2010_2.DetachBasicReference;
import com.netsuite.webservices.platform.core_2010_2.GetAllRecord;
import com.netsuite.webservices.platform.core_2010_2.GetDeletedFilter;
import com.netsuite.webservices.platform.core_2010_2.GetSavedSearchRecord;
import com.netsuite.webservices.platform.core_2010_2.InitializeRecord;
import com.netsuite.webservices.platform.core_2010_2.ItemAvailabilityFilter;
import com.netsuite.webservices.platform.core_2010_2.Record;
import com.netsuite.webservices.platform.core_2010_2.RecordRef;
import com.netsuite.webservices.platform.core_2010_2.RecordRefList;
import com.netsuite.webservices.platform.core_2010_2.SearchEnumMultiSelectField;
import com.netsuite.webservices.platform.core_2010_2.UpdateInviteeStatusReference;
import com.netsuite.webservices.platform.core_2010_2.types.CalendarEventAttendeeResponse;
import com.netsuite.webservices.platform.core_2010_2.types.GetAllRecordType;
import com.netsuite.webservices.platform.core_2010_2.types.GetCustomizationType;
import com.netsuite.webservices.platform.core_2010_2.types.InitializeType;
import com.netsuite.webservices.platform.core_2010_2.types.RecordType;
import com.netsuite.webservices.platform.core_2010_2.types.SearchEnumMultiSelectFieldOperator;
import com.netsuite.webservices.platform.core_2010_2.types.SearchRecordType;
import com.netsuite.webservices.platform.messages_2010_2.AddListRequest;
import com.netsuite.webservices.platform.messages_2010_2.AddRequest;
import com.netsuite.webservices.platform.messages_2010_2.AsyncSearchRequest;
import com.netsuite.webservices.platform.messages_2010_2.AttachRequest;
import com.netsuite.webservices.platform.messages_2010_2.CheckAsyncStatusRequest;
import com.netsuite.webservices.platform.messages_2010_2.DeleteListRequest;
import com.netsuite.webservices.platform.messages_2010_2.DeleteRequest;
import com.netsuite.webservices.platform.messages_2010_2.DetachRequest;
import com.netsuite.webservices.platform.messages_2010_2.GetAllRequest;
import com.netsuite.webservices.platform.messages_2010_2.GetBudgetExchangeRateRequest;
import com.netsuite.webservices.platform.messages_2010_2.GetConsolidatedExchangeRateRequest;
import com.netsuite.webservices.platform.messages_2010_2.GetCustomizationIdRequest;
import com.netsuite.webservices.platform.messages_2010_2.GetDeletedRequest;
import com.netsuite.webservices.platform.messages_2010_2.GetItemAvailabilityRequest;
import com.netsuite.webservices.platform.messages_2010_2.GetRequest;
import com.netsuite.webservices.platform.messages_2010_2.GetSavedSearchRequest;
import com.netsuite.webservices.platform.messages_2010_2.GetServerTimeRequest;
import com.netsuite.webservices.platform.messages_2010_2.InitializeRequest;
import com.netsuite.webservices.platform.messages_2010_2.UpdateInviteeStatusRequest;
import com.netsuite.webservices.platform.messages_2010_2.UpdateListRequest;
import com.netsuite.webservices.platform.messages_2010_2.UpdateRequest;
import com.netsuite.webservices.platform.messages_2010_2.WriteResponseList;
import com.netsuite.webservices.platform_2010_2.NetSuitePortType;
import com.zauberlabs.commons.mom.MapObjectMapper;

/**
 * Implementation of the {@link SoapNetSuiteClient} that uses CXF generated-based
 * interface
 */
public class CxfNetSuiteClient implements SoapNetSuiteClient, CxfPortProvider
{

    private final CxfPortProvider portProvider;
    private final MapObjectMapper mom = JaxbMapObjectMappers.defaultWithPackage("com.netsuite.webservices")
        .build();

    static
    {
        System.setProperty("com.sun.xml.bind.v2.runtime.JAXBContextImpl.fastBoot", "true");
    }

    public CxfNetSuiteClient(@NotNull CxfPortProvider portProvider)
    {
        Validate.notNull(portProvider);
        this.portProvider = portProvider;
    }

    public Object updateRecord(RecordReference recordReference, Map<String, Object> recordAttributes)
        throws Exception
    {
        Validate.notNull(recordReference);
        Validate.notEmpty(recordAttributes);
        return getAuthenticatedPort().update(
            new UpdateRequest(createRecord(recordReference.getType(),
                recordReference.getId().populate(new HashMap<String, Object>(recordAttributes)))));
    }

    public WriteResponseList addList(RecordType recordType, List<Map<String, Object>> recordsAttributes)
        throws Exception
    {
        Validate.notNull(recordType);
        Validate.notEmpty(recordsAttributes);
        return getAuthenticatedPort().addList(
            new AddListRequest(createRecords(recordType, recordsAttributes))).getWriteResponseList();
    }

    public WriteResponseList updateList(RecordType recordType, List<Map<String, Object>> recordsAttributes)
        throws Exception
    {
        Validate.notNull(recordType);
        Validate.notEmpty(recordsAttributes);
        return getAuthenticatedPort().updateList(
            new UpdateListRequest(createRecords(recordType, recordsAttributes))).getWriteResponseList();
    }

    public WriteResponseList deleteList(List<Map<String, Object>> recordsToDelete)
        throws Exception
    {
        Validate.notEmpty(recordsToDelete);
        return getAuthenticatedPort().deleteList(new DeleteListRequest(createRecordsRef(recordsToDelete)))
            .getWriteResponseList();
    }

    private List<BaseRef> createRecordsRef(List<Map<String, Object>> recordsToDelete) throws Exception
    {
        List<BaseRef> list = new ArrayList<BaseRef>();
        for (Map<String, Object> record : recordsToDelete)
        {
            list.add(createRecordRef(record));
        }
        return list;
    }

    private List<Record> createRecords(RecordType recordType, List<Map<String, Object>> recordsAttributes)
        throws Exception
    {
        List<Record> list = new ArrayList<Record>();
        for (Map<String, Object> record : recordsAttributes)
        {
            list.add(createRecord(recordType, record));
        }
        return list;
    }

    public Object addRecord(RecordType recordType, Map<String, Object> recordAttributes) throws Exception
    {
        Validate.notNull(recordType);
        Validate.notEmpty(recordAttributes);
        return getAuthenticatedPort().add(new AddRequest(createRecord(recordType, recordAttributes)));
    }

    public Iterable<Record> findRecords(@NotNull final SearchRecordType recordType,
                                        @NotNull final String expression) throws Exception
    {
        Validate.notNull(recordType);
        return new RecordSearchIterable(this, recordType, expression);
    }

    public Iterable<Record> savedFindRecords(@NotNull final SearchRecordType recordType,
                                             @NotNull String searchId) throws Exception
    {
        Validate.notEmpty(searchId);
        return new SavedRecordSearchIterable(this, recordType, searchId);
    }

    public AsyncStatusResult asyncFindRecord(@NotNull SearchRecordType recordType, @NotNull String expression)
        throws Exception
    {
        Validate.notNull(recordType);
        Validate.notEmpty(expression);
        return getAuthenticatedPort().asyncSearch(
            new AsyncSearchRequest(FilterExpressionParser.parse(recordType, expression)))
            .getAsyncStatusResult();
    }

    private Record createRecord(RecordType recordType, Map<String, Object> recordAttributes) throws Exception
    {
        return (Record) mom.unmap(recordAttributes, recordType.getRecordClass());
    }

    private BaseRef createRecordRef(Map<String, Object> recordAttributes) throws Exception
    {
        return (BaseRef) mom.unmap(recordAttributes, RecordRef.class);
    }

    public Object attachRecord(@NotNull RecordReference sourceRecord,
                               @NotNull RecordReference destinationRecord,
                               RecordReference contactEntity) throws Exception
    {
        Validate.notNull(sourceRecord);
        Validate.notNull(destinationRecord);

        if (contactEntity == null)
        {
            return getAuthenticatedPort().attach(
                new AttachRequest(new AttachBasicReference(destinationRecord.createRef(),
                    sourceRecord.createRef())));
        }
        else
        {
            return getAuthenticatedPort().attach(
                new AttachRequest(new AttachContactReference(destinationRecord.createRef(),
                    sourceRecord.createRef(), contactEntity.createRef())));
        }
    }

    public Object deleteRecord(RecordReference record) throws Exception
    {
        Validate.notNull(record);
        return getAuthenticatedPort().delete(new DeleteRequest(record.createRef()));
    }

    public Object detachRecord(@NotNull RecordReference sourceEntity,
                               @NotNull RecordReference destinationEntity) throws Exception
    {
        return getAuthenticatedPort().detach(
            new DetachRequest(new DetachBasicReference(sourceEntity.createRef(),
                destinationEntity.createRef())));
    }

    public Object getDeletedRecords(RecordType type, DateExpression expression) throws Exception
    {
        GetDeletedFilter filter = new GetDeletedFilter();
        filter.setDeletedDate(expression.createSearchDateField());
        filter.setType(new SearchEnumMultiSelectField(Arrays.asList(type.value()),
            SearchEnumMultiSelectFieldOperator.ANY_OF));
        return getAuthenticatedPort().getDeleted(new GetDeletedRequest(filter));
    }

    public Object getRecord(RecordReference record) throws Exception
    {
        Validate.notNull(record);
        return getAuthenticatedPort().get(new GetRequest(record.createRef()));
    }

    public Object getRecords(RecordType type) throws Exception
    {
        Validate.notNull(type);
        return getAuthenticatedPort().getAll(
            new GetAllRequest(new GetAllRecord(GetAllRecordType.fromValue(type.value()))));
    }

    public Object getBudgetExchangeRates(@NotNull RecordId period,
                                         @NotNull RecordId fromSubsidiary,
                                         RecordId toSubsidiary) throws Exception
    {
        Validate.notNull(period);
        Validate.notNull(fromSubsidiary);
        return getAuthenticatedPort().getBudgetExchangeRate(
            new GetBudgetExchangeRateRequest(new BudgetExchangeRateFilter(period.createRef(),
                fromSubsidiary.createRef(), createRefNullSafe(toSubsidiary))));
    }

    public Object getConsolidatedExchangeRates(@NotNull RecordId period,
                                               @NotNull RecordId fromSubsidiary,
                                               RecordId toSubsidiary) throws Exception
    {
        Validate.notNull(period);
        Validate.notNull(fromSubsidiary);
        return getAuthenticatedPort().getConsolidatedExchangeRate(
            new GetConsolidatedExchangeRateRequest(new ConsolidatedExchangeRateFilter(period.createRef(),
                fromSubsidiary.createRef(), createRefNullSafe(toSubsidiary))));
    }

    public Object getCustomizationIds(@NotNull GetCustomizationType type, boolean includeInactives)
        throws Exception
    {
        Validate.notNull(type);
        return getAuthenticatedPort().getCustomizationId(
            new GetCustomizationIdRequest(new CustomizationType(type), includeInactives));
    }

    public Object getItemAvailabilities(@NotNull RecordReference recordReference, Date ifModifiedSince)
        throws Exception
    {
        Validate.notNull(recordReference);
        return getAuthenticatedPort().getItemAvailability(//
            new GetItemAvailabilityRequest(//
                new ItemAvailabilityFilter(singletonRecordRefList(recordReference),
                    XmlGregorianCalendars.nullSafeFrom(ifModifiedSince))));
    }

    public Object getSavedSearch(@NotNull RecordType type) throws Exception
    {
        Validate.notNull(type);
        return getAuthenticatedPort().getSavedSearch(
            new GetSavedSearchRequest(new GetSavedSearchRecord(SearchRecordType.fromValue(type.value()))));
    }

    public Object getServerTime() throws Exception
    {
        return getAuthenticatedPort().getServerTime(new GetServerTimeRequest());
    }

    public Object updateInviteeStatus(@NotNull RecordId eventId, @NotNull CalendarEventAttendeeResponse status)
        throws Exception
    {
        Validate.notNull(eventId);
        Validate.notNull(status);
        return getAuthenticatedPort().updateInviteeStatus(
            new UpdateInviteeStatusRequest(new UpdateInviteeStatusReference(eventId.createRef(), status)));
    }

    public AsyncStatusResult checkAsyncStatus(String jobId) throws Exception
    {
        Validate.notEmpty(jobId);
        return getAuthenticatedPort().checkAsyncStatus(new CheckAsyncStatusRequest(jobId))
            .getAsyncStatusResult();
    }

    public Iterable<Record> getAsyncFindResult(String jobId) throws Exception
    {
        Validate.notEmpty(jobId);
        return new AsyncRecordSearchIterable(this, jobId);
    }

    public Object initialize(InitializeType type, RecordReference recordReference) throws Exception
    {
        Validate.notNull(type);
        Validate.notNull(recordReference);
        return getAuthenticatedPort().initialize(
            new InitializeRequest(new InitializeRecord(type, recordReference.createInitializeRef(), null)));
    }

    public NetSuitePortType getAuthenticatedPort() throws Exception
    {
        return portProvider.getAuthenticatedPort();
    }
    
    public void login() throws Exception
    {
        portProvider.login();
    }

    private RecordRef createRefNullSafe(RecordId toSubsidiary)
    {
        return toSubsidiary != null ? toSubsidiary.createRef() : null;
    }

    private RecordRefList singletonRecordRefList(RecordReference recordReference)
    {
        return new RecordRefList(Collections.singletonList(recordReference.createRef()));
    }

}
