/**
 * Copyright (c) MuleSoft, Inc. All rights reserved. http://www.mulesoft.com
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.md file.
 */

/**
 * This file was automatically generated by the Mule Cloud Connector Development Kit
 */

package org.mule.module.jira;

import java.util.*;
import java.util.Map.Entry;

import com.atlassian.jira.rpc.soap.beans.*;

import org.apache.commons.collections.CollectionUtils;
import org.mule.api.ConnectionException;
import org.mule.api.annotations.*;
import org.mule.api.annotations.display.Password;
import org.mule.api.annotations.display.Placement;
import org.mule.api.annotations.param.ConnectionKey;
import org.mule.api.annotations.param.Default;
import org.mule.api.annotations.param.Optional;
import org.mule.module.jira.api.JiraClient;

/**
 * JIRA is a proprietary issue tracking product, developed by Atlassian, commonly used for bug tracking, issue
 * tracking, and project management.
 *
 * @author MuleSoft, Inc.
 */
@Connector(name = "jira", schemaVersion = "2.0", friendlyName = "Jira")
public class JiraConnector {

    private JiraClient<List<Object>> client;
    private String token;
    private String connectionUser;
    private String connectionAddress;
    /**
     * if external custom fields names (instead of internal ids) are used when modifying custom fields, the provided
     * user must be a Jira administrator in order to be able to use this feature
     */
    @Configurable
    @Optional
    @Default("false")
    private Boolean useCustomFieldsExternalName;
    private static final String MULTIVALUED_FIELD_SEPARATOR = "\\|";
    private Map<String, String> customFieldsNamesToIdsMapping = null;
    private Calendar customFieldsCacheExpiration = null;

    private Map<String, String> getCustomFieldsNamesToIdsMapping() {
        if (customFieldsNamesToIdsMapping == null || new GregorianCalendar().after(customFieldsCacheExpiration)) {
            customFieldsNamesToIdsMapping = new HashMap<String, String>();

            List<Object> customFields = client.getCustomFields(token);
            for (Object customField : customFields) {
                RemoteField remoteField = (RemoteField) customField;
                customFieldsNamesToIdsMapping.put(remoteField.getName(), remoteField.getId());
            }

            // cached fields expire after 1 hour
            customFieldsCacheExpiration = new GregorianCalendar();
            customFieldsCacheExpiration.add(Calendar.HOUR_OF_DAY, 1);
        }

        return customFieldsNamesToIdsMapping;
    }

    private Map<String, List<String>> convertFieldsToMultivaluedAndMapNamesToIds(Map<String, String> fields) {
        Map<String, List<String>> multivaluedFields = new HashMap<String, List<String>>();
        if (fields != null) {
            for (Entry<String, String> field : fields.entrySet()) {
                String fieldKey = field.getKey();
                if (useCustomFieldsExternalName) {
                    Map<String, String> customFields = getCustomFieldsNamesToIdsMapping();
                    String id = customFields.get(fieldKey);
                    if (id != null) {
                        fieldKey = id;
                    }
                }
                String value = field.getValue();
                if (value == null) {
                    value = "";
                }
                multivaluedFields.put(fieldKey, Arrays.asList(value.split(MULTIVALUED_FIELD_SEPARATOR)));
            }
        }
        return multivaluedFields;
    }

    /**
     * If configured using useCustomFieldsExternalName, tries to map custom fields by name to its id
     * If the mapping is not found, it's left as-is (and expected to fail while trying to set the value in Jira)
     * @param issue that will get its custom fields mapped
     */
    private void mapCustomFieldsFromNamesToIds(RemoteIssue issue) {
        if (issue.getCustomFieldValues() != null && useCustomFieldsExternalName) {
            Map<String, String> customFields = getCustomFieldsNamesToIdsMapping();
            for (RemoteCustomFieldValue field: issue.getCustomFieldValues()) {
                if (!field.getCustomfieldId().startsWith("customfield_")) {
                    if (customFields.containsKey(field.getCustomfieldId())) {
                        field.setCustomfieldId(customFields.get(field.getCustomfieldId()));
                    }
                }
            }
        }
        
    }
    
    /**
     * Finds a comment.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-comment}
     *
     * @param commentId the commentId of the comment
     * @return the RemoteComment
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteComment getComment(Long commentId) {
        return client.getComment(token, commentId);
    }

    /**
     * Returns information about the current configuration of JIRA.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-configuration}
     *
     * @return a RemoteConfiguration object which contains information about the current configuration of JIRA.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteConfiguration getConfiguration() {
        return client.getConfiguration(token);
    }

    /**
     * Creates a group with the given name optionally adding the given user to it.
     * <p/>
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:create-group}
     *
     * @param groupName the name of the group to create.
     * @param userName  the user to add to the group (if null, no user will be added).
     * @return the RemoteGroup created
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteGroup createGroup(String groupName, @Optional String userName) {
        return client.createGroup(token, groupName, userName);
    }

    /**
     * Returns information about the server JIRA is running on including build number and base URL.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-server-info}
     *
     * @return information about the server JIRA is running on including build number and base URL.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteServerInfo getServerInfo() {
        return client.getServerInfo(token);
    }

    /**
     * Find the group with the specified name in JIRA.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-group}
     *
     * @param groupName the name of the group to find
     * @return a RemoteGroup object for the found group or null if it cant be found.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteGroup getGroup(String groupName) {
        return client.getGroup(token, groupName);
    }

    /**
     * Creates a user in JIRA with the specified user details
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:create-user}
     *
     * @param username the user name to create
     * @param password the password for the new user
     * @param fullName the full name of the new user
     * @param email    the email of the new user
     * @return the newly created RemoteUser
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteUser createUser(String username, String password, String fullName, String email) {
        return client.createUser(token, username, password, fullName, email);
    }

    /**
     * Adds a new comment to the issue.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:add-comment}
     *
     * @param issueKey          the key of the issue
     * @param commentAuthor     the author of the comment
     * @param commentBody       the body of the comment
     * @param commentGroupLevel the group level of the comment
     * @param commentRoleLevel  the role level of the comment
     * @return Added comment
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteComment addComment(String issueKey, String commentAuthor, String commentBody,
                                    @Optional String commentGroupLevel, @Optional String commentRoleLevel) {
        return client.addComment(token, issueKey, commentAuthor, commentBody, commentGroupLevel, commentRoleLevel);
    }

    /**
     * Returns an array of all the components for the specified project key.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-components}
     *
     * @param projectKey the key of the requested project
     * @return an array of RemoteComponent objects
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getComponents(String projectKey) {
        return client.getComponents(token, projectKey);
    }

    /**
     * Returns information about a user defined to JIRA.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-user}
     *
     * @param username the user name to look up
     * @return a RemoteUser or null if it cant be found
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteUser getUser(String username) {
        return client.getUser(token, username);
    }

    /**
     * Updates the given group name with the provided users.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:update-group}
     *
     * @param groupName the group name to update
     * @param usernames the updated usernames
     * @return the updated group
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteGroup updateGroup(String groupName, List<String> usernames) {
        return client.updateGroup(token, groupName, usernames);
    }

    /**
     * Adds a user to the given group name.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:add-user-to-group}
     *
     * @param groupName the group name
     * @param userName  the user name
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void addUserToGroup(String groupName, String userName) {
        client.addUserToGroup(token, groupName, userName);
    }

    /**
     * Removes a user from the group name.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:remove-user-from-group}
     *
     * @param groupName the group name for which to remove the user
     * @param userName  the username to remove
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void removeUserFromGroup(String groupName, String userName) {
        client.removeUserFromGroup(token, groupName, userName);
    }

    /**
     * Finds an issue by key.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-issue}
     *
     * @param issueKey the key of the issue to find.
     * @return the issue matching the given key.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteIssue getIssue(String issueKey) {
        return client.getIssue(token, issueKey);
    }

    /**
     * Creates an issue.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:create-issue}
     *
     * @param assignee     the assignee of the new issue
     * @param summary      the summary of the new issue
     * @param description  the description of the new issue
     * @param dueDate      the due date of the new issue using the format MM-dd-yyy'T'HH:mm:ss
     * @param environment  the environment of the new issue
     * @param priority     the priority of the new issue
     * @param project      the project of the new issue
     * @param reporter     the reporter of the new issue
     * @param type         the type of the new issue
     * @param votes        the votes of the new issue
     * @param customFields the custom fields of the new issue, the keys of the map are the field ids, the values should be separated by a "|" if it is multivalued
     * @param componentName the component name
     * @param componentId   the componentId
     * @return the new created issue
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteIssue createIssue(@Optional String assignee,
                                   @Placement(group = "Basic", order = 3) String summary,
                                   @Placement(group = "Basic", order = 4) @Optional String description,
                                   @Optional String dueDate,
                                   @Optional String environment,
                                   @Optional String priority,
                                   @Placement(group = "Basic", order = 1) String project,
                                   @Optional String reporter,
                                   @Placement(group = "Basic", order = 2) String type,
                                   @Optional Long votes,
                                   @Placement(group = "Custom Fields") @Optional Map<String, String> customFields,
                                   @Optional String componentName,
                                   @Optional String componentId) {
        Map<String, List<String>> multivaluedFields = convertFieldsToMultivaluedAndMapNamesToIds(customFields);
        return client.createIssue(token, assignee, summary, description, dueDate, environment, priority, project, reporter, type, votes, multivaluedFields, componentName, componentId);
    }

    /**
     * Creates an issue.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:create-issue-using-object}
     *
     * @param issue JIRA issue to be created
     * @return the new created issue
     */
    @Processor(name="create-issue-using-object")
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteIssue createIssueUsingObject(@Optional @Default("#[payload]") RemoteIssue issue) {
        mapCustomFieldsFromNamesToIds(issue);
        return client.createIssue(token, issue);
    }

    /**
     * Creates an issue using the the security level denoted by the given id.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:create-issue-with-security-level}
     *
     * @param assignee        the assignee of the new issue
     * @param summary         the summary of the new issue
     * @param description     the description of the new issue
     * @param dueDate         the due date of the new issue using the format MM-dd-yyy'T'HH:mm:ss
     * @param environment     the environment of the new issue
     * @param priority        the priority of the new issue
     * @param project         the project of the new issue
     * @param reporter        the reporter of the new issue
     * @param type            the type of the new issue
     * @param votes           the votes of the new issue
     * @param customFields    the custom fields of the new issue, the keys of the map are the field ids, the values should be separated by a "|" if it is multivalued
     * @param securityLevelId the id of the security level to use
     * @param componentName the component name
     * @param componentId   the componentId
     * @return the new created issue
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteIssue createIssueWithSecurityLevel(@Optional String assignee,
                                                    String summary,
                                                    @Optional String description,
                                                    @Optional String dueDate,
                                                    @Optional String environment,
                                                    @Optional String priority,
                                                    String project,
                                                    @Optional String reporter,
                                                    String type,
                                                    @Optional Long votes,
                                                    @Optional Map<String, String> customFields,
                                                    Long securityLevelId,
                                                    @Optional String componentName,
                                                    @Optional String componentId) {
        Map<String, List<String>> multivaluedFields = convertFieldsToMultivaluedAndMapNamesToIds(customFields);
        return client.createIssueWithSecurityLevel(token, assignee, summary, description, dueDate, environment, priority, project, reporter, type, votes, multivaluedFields, securityLevelId, componentName, componentId);
    }

    /**
     * This will update an issue with new values.
     * NOTE : You cannot update the 'status' field of the issue via this method.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:update-issue}
     *
     * @param issueKey The issue to update.
     * @param fields   The fields to be updated, the key of the map is the field id and the value is a list of values for that field, the values should be separated by a "|" if it is multivalued
     * 
     * @return the updated RemoteIssue
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteIssue updateIssue(String issueKey, Map<String, String> fields) {
        Map<String, List<String>> multivaluedFields = convertFieldsToMultivaluedAndMapNamesToIds(fields);
        return client.updateIssue(token, issueKey, multivaluedFields);
    }
    
    /**
     * This will update the set of issues that result from the jql search
     * NOTE : You cannot update the 'status' field of the issues via this method.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:update-issues-by-jql}
     *
     * @param fields   The fields to be updated, the key of the map is the field id and the value is a list of values for that field, the values should be separated by a "|" if it is multivalued
     * @param jql      The jql to search the issues that will be updated if issueKey is not set.
     * @param maxRecordsToUpdate The number of issues you expect the jql search will find. If set,
     * the update will only take place if the number of issues found is the same as this
     * 
     * @return A {@link List} with the updated {@link RemoteIssue}s
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<RemoteIssue> updateIssuesByJql(String jql, Map<String, String> fields, @Optional Integer maxRecordsToUpdate) {
        List<Object> issuesToUpdate = client.getIssuesFromJqlSearch(token, jql, 10000);
        if (issuesToUpdate != null && maxRecordsToUpdate != null)
        {
            if (issuesToUpdate.size() > maxRecordsToUpdate)
            {
                throw new JiraConnectorException("Couldn't execute update-issues-by-jql. "
                    + "The number of issues found by the jql query (" + issuesToUpdate.size()
                    + ") was greater than the maxRecordsToUpdate given (" + maxRecordsToUpdate + ")");
            }
        }
        if (CollectionUtils.isNotEmpty(issuesToUpdate))
        {
            List<RemoteIssue> result = new ArrayList<RemoteIssue>();
            Map<String, List<String>> multivaluedFields = convertFieldsToMultivaluedAndMapNamesToIds(fields);
            for (Object issue : issuesToUpdate)
            {
                RemoteIssue updatedIssue = client.updateIssue(token, ((RemoteIssue)issue).getKey(), multivaluedFields);
                result.add(updatedIssue);
            }
            return result;
        } else {
            throw new JiraConnectorException("Can't execute update. The jql search returned no issues");
        }
    }

    /**
     * Deletes the issue with the given key
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:delete-issue}
     *
     * @param issueKey the key of the issue to delete
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void deleteIssue(String issueKey) {
        client.deleteIssue(token, issueKey);
    }

    /**
     * Finds the available actions for the given issue key
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-available-actions}
     *
     * @param issueKey the key of the issue
     * @return the available actions for the given issue key
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getAvailableActions(String issueKey) {
        return client.getAvailableActions(token, issueKey);
    }

    /**
     * Returns an array of all the sub task issue types in JIRA.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-sub-task-issue-types}
     *
     * @return an array of RemoteIssueType objects
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getSubTaskIssueTypes() {
        return client.getSubTaskIssueTypes(token);
    }

    /**
     * Creates a new project
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:create-project}
     *
     * @param key                    the key for the new project
     * @param projectName            the name for the new project
     * @param description            the description for the new project
     * @param url                    the url for the new project
     * @param lead                   the lead of the new project
     * @param permissionSchemeName   the name of the permission scheme for the new project
     * @param notificationSchemeName the name of the notification scheme for the new project
     * @param securityShemeName      the name of the security scheme  for the new project
     * @return the new project
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteProject createProject(String key,
                                       String projectName,
                                       String description,
                                       @Optional String url,
                                       String lead,
                                       @Optional String permissionSchemeName,
                                       @Optional String notificationSchemeName,
                                       @Optional String securityShemeName) {
        return client.createProject(token, key, projectName, description, url, lead, permissionSchemeName, notificationSchemeName, securityShemeName);
    }

    /**
     * Updates the project denoted by the given key.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:update-project}
     *
     * @param key                    the key of the project to update
     * @param description            the new description
     * @param url                    the new url
     * @param lead                   the new lead
     * @param permissionSchemeName   the new permission scheme name
     * @param notificationSchemeName the new notification scheme name
     * @param securityShemeName      the new security scheme name
     * @return the updated project
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteProject updateProject(String key,
                                       String description,
                                       @Optional String url,
                                       String lead,
                                       @Optional String permissionSchemeName,
                                       @Optional String notificationSchemeName,
                                       @Optional String securityShemeName) {
        return client.updateProject(token, key, description, url, lead, permissionSchemeName, notificationSchemeName, securityShemeName);
    }

    /**
     * Returns the Project with the matching key (if the user has permission to browse it).
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-project-by-key}
     *
     * @param projectKey the key of the requested projec
     * @return the RemoteProject object specified by the key, if it exists and the user has the BROWSE permission for it
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteProject getProjectByKey(String projectKey) {
        return client.getProjectByKey(token, projectKey);
    }

    /**
     * Removes all role actors for the given project key.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:remove-all-role-actors-by-project}
     *
     * @param projectKey the project key for which to remove all role actors
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void removeAllRoleActorsByProject(String projectKey) {
        client.removeAllRoleActorsByProject(token, projectKey);
    }

    /**
     * Returns an array of all the issue statuses in JIRA.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-priorities}
     *
     * @return an array of RemoteStatus objects
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getPriorities() {
        return client.getPriorities(token);
    }

    /**
     * Returns an array of all the issue resolutions in JIRA.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-resolutions}
     *
     * @return an array of RemoteResolution objects
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getResolutions() {
        return client.getResolutions(token);
    }

    /**
     * Returns an array of all the issue types for all projects in JIRA.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-issue-types}
     *
     * @return an array of RemoteIssueType objects
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getIssueTypes() {
        return client.getIssueTypes(token);
    }


    /**
     * Returns an array of all the issue statuses in JIRA.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-statuses}
     *
     * @return an array of RemoteStatus objects
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getStatuses() {
        return client.getStatuses(token);
    }


    /**
     * Returns an array of all the (non-sub task) issue types for the specified project id.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-issue-types-for-project}
     *
     * @param projectId id of the project
     * @return an array of RemoteIssueType objects
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getIssueTypesForProject(String projectId) {
        return client.getIssueTypesForProject(token, projectId);
    }

    /**
     * Returns the project roles.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-project-roles}
     *
     * @return the project roles.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getProjectRoles() {
        return client.getProjectRoles(token);
    }

    /**
     * Returns the project role by projectRoleId.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-project-role}
     *
     * @param projectRoleId the projectRoleId of the project role
     * @return the project role by projectRoleId.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteProjectRole getProjectRole(Long projectRoleId) {
        return client.getProjectRole(token, projectRoleId);
    }

    /**
     * Returns the project role actors for the given project
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-project-role-actors}
     *
     * @param projectRoleId the project role id to use
     * @param projectKey    the project key to use
     * @return the project role actors for the given project
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteProjectRoleActors getProjectRoleActors(Long projectRoleId, String projectKey) {
        return client.getProjectRoleActors(token, projectRoleId, projectKey);
    }

    /**
     * Returns the default role actors for the given project role id.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-default-role-actors}
     *
     * @param projectRoleId the id of the project role
     * @return the default role actors for the given project role id.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteRoleActors getDefaultRoleActors(Long projectRoleId) {
        return client.getDefaultRoleActors(token, projectRoleId);
    }

    /**
     * Removes all role actors using the given name and type.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:remove-all-role-actors-by-name-and-type}
     *
     * @param roleName the name to delete
     * @param type the type to delete
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void removeAllRoleActorsByNameAndType(String roleName, String type) {
        client.removeAllRoleActorsByNameAndType(token, roleName, type);
    }

    /**
     * Deletes the project role denoted by the given project role id.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:delete-project-role}
     *
     * @param projectRoleId the id of the project role to delete
     * @param confirm       whether confirm
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void deleteProjectRole(Long projectRoleId, Boolean confirm) {
        client.deleteProjectRole(token, projectRoleId, confirm);
    }

    /**
     * Updates the project role with the given id.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:update-project-role}
     *
     * @param projectRoleId          the id of the project role to update
     * @param projectRoleName        the new project role name
     * @param projectRoleDescription the new project role description
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void updateProjectRole(Long projectRoleId, @Optional String projectRoleName, @Optional String projectRoleDescription) {
        client.updateProjectRole(token, projectRoleId, projectRoleName, projectRoleDescription);
    }

    /**
     * Creates a new project role.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:create-project-role}
     *
     * @param projectRoleName        the name of the new project role
     * @param projectRoleDescription the description of the new project role
     * @return the created project role
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteProjectRole createProjectRole(String projectRoleName, String projectRoleDescription) {
        return client.createProjectRole(token, projectRoleName, projectRoleDescription);
    }

    /**
     * Checks if the given project role name is unique.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:is-project-role-name-unique}
     *
     * @param roleName the project role name to check for uniqueness
     * @return true if the given project role name is unique, false otherwise.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public boolean isProjectRoleNameUnique(String roleName) {
        return client.isProjectRoleNameUnique(token, roleName);
    }

    /**
     * Releases the version denoted by the given name
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:release-version}
     *
     * @param projectKey  the project key to use
     * @param versionName the version name to release
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void releaseVersion(String projectKey, String versionName) {
        client.releaseVersion(token, projectKey, versionName);
    }

    /**
     * Adds the given actors to the project role.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:add-default-actors-to-project-role}
     *
     * @param actors        the actors to add
     * @param projectRoleId the id of the project role to use
     * @param projectKey    the key of project to use
     * @param actorType     the actor type to use
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void addActorsToProjectRole(List<String> actors, Long projectRoleId, String projectKey, @Optional String actorType) {
        client.addActorsToProjectRole(token, actors, projectRoleId, projectKey, actorType);
    }

    /**
     * Removes the given actors from the project role
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:remove-default-actors-from-project-role}
     *
     * @param actors        the actors to remove
     * @param projectRoleId the id of the project role to use
     * @param projectKey    the key of project to use
     * @param actorType     the actor type to use
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void removeActorsFromProjectRole(List<String> actors, Long projectRoleId, String projectKey, @Optional String actorType) {
        client.removeActorsFromProjectRole(token, actors, projectRoleId, projectKey, actorType);
    }

    /**
     * Adds the default actors to the project role denoted by this id.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:add-default-actors-to-project-role}
     *
     * @param actors        the actors to add
     * @param projectRoleId the id of the project role
     * @param type          the type
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void addDefaultActorsToProjectRole(List<String> actors, Long projectRoleId, @Optional String type) {
        client.addDefaultActorsToProjectRole(token, actors, projectRoleId, type);
    }

    /**
     * Removes the default actors from the project role denoted by this id.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:remove-default-actors-from-project-role}
     *
     * @param actors              the actors to remove
     * @param remoteProjectRoleId the id of the project role
     * @param type                the type
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void removeDefaultActorsFromProjectRole(List<String> actors, Long remoteProjectRoleId, @Optional String type) {
        client.removeDefaultActorsFromProjectRole(token, actors, remoteProjectRoleId, type);
    }

    /**
     * Returns the associated notification schemes for the given project role.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-associated-notification-schemes}
     *
     * @param projectRoleId the project role to search
     * @return the associated notification schemes for the given project role.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getAssociatedNotificationSchemes(Long projectRoleId) {
        return client.getAssociatedNotificationSchemes(token, projectRoleId);
    }

    /**
     * Returns the associated permission schemas for the given project role.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-associated-permission-schemes}
     *
     * @param projectRoleId the project role to search
     * @return the associated permission schemas for the given project role.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getAssociatedPermissionSchemes(Long projectRoleId) {
        return client.getAssociatedPermissionSchemes(token, projectRoleId);
    }

    /**
     * Deletes the project represented by the given project key.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:delete-project}
     *
     * @param projectKey the key of the project to delete
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void deleteProject(String projectKey) {
        client.deleteProject(token, projectKey);
    }

    /**
     * Returns the Project with the matching id (if the user has permission to browse it).
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-project-by-id}
     *
     * @param projectId the id of the requested project
     * @return the RemoteProject object specified by the key, if it exists and the user has the BROWSE permission for it
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteProject getProjectById(Long projectId) {
        return client.getProjectById(token, projectId);
    }

    /**
     * Returns an array of all the versions for the specified project key.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-versions}
     *
     * @param projectKey the key of the requested project
     * @return an array of RemoteVersion objects
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getVersions(String projectKey) {
        return client.getVersions(token, projectKey);
    }

    /**
     * Returns the comments for the issue denoted by the given key.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-comments}
     *
     * @param issueKey the key of the issue to get the comments for
     * @return the comments for the issue denoted by the given key.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getComments(
            String issueKey) {
        return client.getComments(token, issueKey);
    }

    /**
     * This retreives a list of the currently logged in user's favourite fitlers.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-favourite-filters}
     *
     * @return a list of the currently logged in user's favourite fitlers.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getFavouriteFilters() {
        return client.getFavouriteFilters(token);
    }

    /**
     * Archieves the given version/
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:archive-version}
     *
     * @param projectKey  the project key to use
     * @param versionName the version name to use
     * @param archive     whether it should be archived
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void archiveVersion(String projectKey, String versionName, Boolean archive) {
        client.archiveVersion(token, projectKey, versionName, archive);
    }

    /**
     * Returns the fields for edit for the given issue key
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-fields-for-edit}
     *
     * @param issueKey the issue key to get the fields for
     * @return the fields for edit
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getFieldsForEdit(String issueKey) {
        return client.getFieldsForEdit(token, issueKey);
    }

    /**
     * Returns an array of all the sub task issue types for the specified project id.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-sub-task-issue-types-for-project}
     *
     * @param projectId id of the project
     * @return an array of RemoteIssueType objects
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getSubTaskIssueTypesForProject(String projectId) {
        return client.getSubTaskIssueTypesForProject(token, projectId);
    }

    /**
     * Log in using the given credentials, it returns the authentication token.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:login}
     *
     * @param username the username to use
     * @param password the password to use
     * @return the authentication token
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public String login(String username, String password) {
        return client.login(username, password);
    }

    /**
     * Returns the current security level for given issue
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-security-level}
     *
     * @param issueKey the issue key
     * @return issue security level
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteSecurityLevel getSecurityLevel(String issueKey) {
        return client.getSecurityLevel(token, issueKey);
    }

    /**
     * Returns the custom fields for the current user
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-custom-fields}
     *
     * @return the custom fields for the current user
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getCustomFields() {
        return client.getCustomFields(token);
    }

    /**
     * Cleans up an authentication token that was previously created with a call to login
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:logout}
     *
     * @param token the token to invalidate
     * @return true if the logout succeeded
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public boolean logout(String token) {
        return client.logout(token);
    }

    /**
     * Returns the Project with the matching id (if the user has permission to browse it) with notification, issue security and permission schemes attached.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-project-with-schemes-by-id}
     *
     * @param projectId the id of the requested project
     * @return the RemoteProject object specified by the key, if it exists and the user has the BROWSE permission for it
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteProject getProjectWithSchemesById(Long projectId) {
        return client.getProjectWithSchemesById(token, projectId);
    }

    /**
     * Returns an array of all security levels for a given project.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-security-levels}
     *
     * @param projectKey the key for the project
     * @return array of RemoteSecurityLevels for the project
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getSecurityLevels(String projectKey) {
        return client.getSecurityLevels(token, projectKey);
    }

    /**
     * Retrieves avatars for the given project. If the includeSystemAvatars parameter is true, this will include both
     * system (built-in) avatars as well as custom (user-supplied) avatars for that project, otherwise it will include
     * only the custom avatars. Project browse permission is required.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-project-avatars}
     *
     * @param projectKey           the key for the project.
     * @param includeSystemAvatars if false, only custom avatars will be included in the returned array.
     * @return the avatars for the project, possibly empty.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getProjectAvatars(String projectKey, Boolean includeSystemAvatars) {
        return client.getProjectAvatars(token, projectKey, includeSystemAvatars);
    }

    /**
     * Sets the current avatar for the given project to that with the given id. Project administration permission is required.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:set-project-avatar}
     *
     * @param projectKey the key for the project.
     * @param avatarId   the id of an existing avatar to use for the project or null for the default avatar.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void setProjectAvatar(String projectKey, Long avatarId) {
        client.setProjectAvatar(token, projectKey, avatarId);
    }

    /**
     * Retrieves the current avatar for the given project. Project browse permission is required.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-project-avatar}
     *
     * @param projectKey the key for the project.
     * @return the current avatar for the project.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteAvatar getProjectAvatar(String projectKey) {
        return client.getProjectAvatar(token, projectKey);
    }

    /**
     * Deletes the given custom Avatar from the system. System avatars cannot be deleted. Project administration permission is required.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:delete-project-avatar}
     *
     * @param avatarId id of the custom avatar to delete.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void deleteProjectAvatar(Long avatarId) {
        client.deleteProjectAvatar(token, avatarId);
    }

    /**
     * Returns notification schemes.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-notification-schemes}
     *
     * @return the notification schemes.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getNotificationSchemes() {
        return client.getNotificationSchemes(token);
    }

    /**
     * Returns the permission schemes.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-permission-schemes}
     *
     * @return the permission schemes.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getPermissionSchemes() {
        return client.getPermissionSchemes(token);
    }

    /**
     * Returns all the permissions.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-all-permissions}
     *
     * @return all the permissions
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getAllPermissions() {
        return client.getAllPermissions(token);
    }

    /**
     * Creates a new permission schema using the given name and description.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:create-permission-scheme}
     *
     * @param permissionName        the name of the new permission scheme
     * @param description the description of the new permission scheme
     * @return the created permission scheme
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemotePermissionScheme createPermissionScheme(String permissionName, String description) {
        return client.createPermissionScheme(token, permissionName, description);
    }

    /**
     * Adds the permission to the given entity name (username or group name)
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:add-permission-to}
     *
     * @param permissionSchemeName the name of the permission scheme to use
     * @param permissionCode       the permission code to use
     * @param entityName           the entity name, username or group name
     * @return the modified permission scheme
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemotePermissionScheme addPermissionTo(String permissionSchemeName, Long permissionCode, String entityName) {
        return client.addPermissionTo(token, permissionSchemeName, permissionCode, entityName);
    }

    /**
     * Removes the permission to the given entity name (username or group name)
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:delete-permission-from}
     *
     * @param permissionSchemeName the name of the permission scheme to use
     * @param permissionCode       the permission code to use
     * @param entityName           the entity name, username or group name
     * @return the modified permission scheme
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemotePermissionScheme deletePermissionFrom(String permissionSchemeName, Long permissionCode, String entityName) {
        return client.deletePermissionFrom(token, permissionSchemeName, permissionCode, entityName);
    }

    /**
     * Deletes the permission scheme denoted by the given name
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:delete-permission-scheme}
     *
     * @param permissionSchemeName the name of the permission scheme to delete
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void deletePermissionScheme(String permissionSchemeName) {
        client.deletePermissionScheme(token, permissionSchemeName);
    }

    /**
     * Returns the attachments for the issue denoted by the given key.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-attachments-from-issue}
     *
     * @param issueKey the issue key to use
     * @return the attachments for the issue denoted by the given key.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getAttachmentsFromIssue(String issueKey) {
        return client.getAttachmentsFromIssue(token, issueKey);
    }

    /**
     * Returns whether the current user has permissions to edit the comment denoted by the given id.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:has-permission-to-edit-comment}
     *
     * @param commentId the comment id to use
     * @return whether the current user has permissions to edit the comment denoted by the given id.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public boolean hasPermissionToEditComment(Long commentId) {
        return client.hasPermissionToEditComment(token, commentId);
    }

    /**
     * Edits the comment denoted by the given id
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:edit-comment}
     *
     * @param commentId    the id of the comment to edit
     * @param body         the updated body comment
     * @param updateAuthor the update author
     * @return the edited comment
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteComment editComment(Long commentId, @Optional String body, @Optional String updateAuthor) {
        return client.editComment(token, commentId, body, updateAuthor);
    }

    /**
     * Returns the fields for the given action.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-fields-for-action}
     *
     * @param issueKey       the issue key to use
     * @param actionIdString the action id to use
     * @return the fields for the given action
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getFieldsForAction(String issueKey, String actionIdString) {
        return client.getFieldsForAction(token, issueKey, actionIdString);
    }

    /**
     * Returns the issue for the given issue id.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-issue-by-id}
     *
     * @param issueId the issue id to use
     * @return the issue for the given issue id.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteIssue getIssueById(String issueId) {
        return client.getIssueById(token, issueId);
    }

    /**
     * Deletes the worklog with the given id and sets the remaining estimate field on the isssue to the given value. The time spent field of the issue is reduced by the time spent amount on the worklog being deleted.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:delete-worklog-with-new-remaining-estimate}
     * <p/>
     * the SOAP auth token.
     *
     * @param workLogId            the id of the worklog to delete.
     * @param newRemainingEstimate the new value for the issue's remaining estimate as a duration string, eg 1d 2h.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void deleteWorklogWithNewRemainingEstimate(String workLogId, String newRemainingEstimate) {
        client.deleteWorklogWithNewRemainingEstimate(token, workLogId, newRemainingEstimate);
    }

    /**
     * Deletes the worklog with the given id and updates the remaining estimate field on the isssue by increasing it
     * by the time spent amount on the worklog being deleted. The time spent field of the issue is reduced by the
     * time spent amount on the worklog being deleted.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:delete-worklog-and-auto-adjust-remaining-estimate}
     * <p/>
     * the SOAP auth token.
     *
     * @param worklogId the id of the worklog to delete.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void deleteWorklogAndAutoAdjustRemainingEstimate(String worklogId) {
        client.deleteWorklogAndAutoAdjustRemainingEstimate(token, worklogId);
    }

    /**
     * Deletes the worklog with the given id but leaves the remaining estimate field on the isssue unchanged. The time
     * spent field of the issue is reduced by the time spent amount on the worklog being deleted.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:delete-worklog-and-retain-remaining-estimate}
     * <p/>
     * the SOAP auth token.
     *
     * @param worklogId the id of the worklog to delete.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void deleteWorklogAndRetainRemainingEstimate(String worklogId) {
        client.deleteWorklogAndRetainRemainingEstimate(token, worklogId);
    }

    /**
     * Returns all worklogs for the given issue.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-worklogs}
     * <p/>
     * the SOAP auth token.
     *
     * @param issueKey the key of the issue.
     * @return all the worklogs of the issue.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getWorklogs(String issueKey) {
        return client.getWorklogs(token, issueKey);
    }

    /**
     * Determines if the user has the permission to add worklogs to the specified issue, that timetracking is enabled
     * in JIRA and that the specified issue is in an editable workflow state.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:has-permission-to-create-worklog}
     * <p/>
     * the SOAP auth token.
     *
     * @param issueKey the key of the issue.
     * @return true if the user has permission to create a worklog on the specified issue, false otherwise
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public boolean hasPermissionToCreateWorklog(String issueKey) {
        return client.hasPermissionToCreateWorklog(token, issueKey);
    }

    /**
     * Determine whether the current user has the permission to delete the supplied worklog, that timetracking is enabled in JIRA and that the associated issue is in an editable workflow state.
     * This method will return true if the user is a member of the worklog's group/role level (if specified) AND
     * The user has the WORKLOG_DELETE_ALL permission; OR
     * The user is the worklog author and has the WORKLOG_DELETE_OWN permission
     * and false otherwise.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:has-permission-to-delete-worklog}
     * <p/>
     * the SOAP auth token.
     *
     * @param worklogId the id of the worklog wishes to delete.
     * @return true if the user has permission to delete the supplied worklog, false otherwise
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public boolean hasPermissionToDeleteWorklog(String worklogId) {
        return client.hasPermissionToDeleteWorklog(token, worklogId);
    }

    /**
     * Determine whether the current user has the permission to update the supplied worklog, that timetracking is enabled in JIRA and that the associated issue is in an editable workflow state.
     * This method will return true if the user is a member of the worklog's group/role level (if specified) AND
     * The user has the WORKLOG_EDIT_ALL permission; OR
     * The user is the worklog author and has the WORKLOG_EDIT_OWN permission
     * and false otherwise.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:has-permission-to-update-worklog}
     *
     * @param worklogId the ide of the worklog wishes to update.
     * @return true if the user has permission to update the supplied worklog, false otherwise
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public boolean hasPermissionToUpdateWorklog(String worklogId) {
        return client.hasPermissionToUpdateWorklog(token, worklogId);
    }

    /**
     * Modifies the worklog with the id of the given worklog, updating its fields to match the given worklog and sets
     * the remaining estimate field on the relevant issue to the given value. The time spent field of the issue is
     * changed by subtracting the previous value of the worklog's time spent amount and adding the new value in the
     * given worklog.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:update-worklog-with-new-remaining-estimate}
     *
     * @param issueKey             the issue key to use
     * @param worklogId            the worklog id to use
     * @param newRemainingEstimate the new value for the issue's remaining estimate as a duration string, eg 1d 2h.
     * @param comment              the new comment
     * @param groupLevel           the new group level
     * @param roleLevelId          the new role level id
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void updateWorklogWithNewRemainingEstimate(String issueKey, String worklogId, String newRemainingEstimate, @Optional String comment, @Optional String groupLevel, @Optional String roleLevelId) {
        client.updateWorklogWithNewRemainingEstimate(token, issueKey, worklogId, comment, groupLevel, roleLevelId, newRemainingEstimate);
    }

    /**
     * Adds a new version
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:add-version}
     *
     * @param projectKey  the project key to use
     * @param versionName the version name to use
     * @param archived    whether is archived
     * @param released    whether is released
     * @param releaseDate the release date to use in the format MM-dd-yyy'T'HH:mm:ss
     * @return the new version
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteVersion addVersion(String projectKey, String versionName, Boolean archived, Boolean released, String releaseDate) {
        return client.addVersion(token, projectKey, versionName, archived, released, releaseDate);
    }

    /**
     * Given an issue key, this method returns the resolution date for this issue. If the issue hasn't been resolved
     * yet, this method will return null.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-resolution-date-by-key}
     *
     * @param issueKey the key of the issue
     * @return The resolution date of the issue. May be null
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public Calendar getResolutionDateByKey(String issueKey) {
        return client.getResolutionDateByKey(token, issueKey);
    }

    /**
     * Given an issue id, this method returns the resolution date for this issue. If the issue hasn't been resolved yet, this method will return null.
     * If the no issue with the given id exists a RemoteException will be thrown.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-resolution-date-by-id}
     *
     * @param issueId the id of the issue
     * @return The resolution date of the issue. May be null
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public Calendar getResolutionDateById(Long issueId) {
        return client.getResolutionDateById(token, issueId);
    }

    /**
     * Returns the issue count for the filter denoted by this id.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-issue-count-for-filter}
     *
     * @param filterId the fiter id to use
     * @return the issue count for the filter denoted by this id.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public long getIssueCountForFilter(String filterId) {
        return client.getIssueCountForFilter(token, filterId);
    }

    /**
     * Returns issues containing searchTerms that are within the specified projects.
     * Note: this is a fuzzy search, returned in order of 'relevance', so the results are only generally useful for
     * human consumption.
     * <p/>
     * This method will return no more than the maxNumResults.
     * <p/>
     * This method also respects the jira.search.views.max.limit and jira.search.views.max.unlimited.group JIRA
     * properties which will override the max number of results returned.
     * <p/>
     * If the jira.search.views.max.limit property is set and you are not in a group specified by
     * jira.search.views.max.unlimited.group then the number of results returned will be constrained by the value of
     * jira.search.views.max.limit if it is less than the specified maxNumResults.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-issues-from-text-search-with-project}
     *
     * @param projectKeys   the project keys to use
     * @param searchTerms   earch terms
     * @param maxNumResults the maximum number of results that this method will return.
     * @return issues matching the search terms
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getIssuesFromTextSearchWithProject(List<String> projectKeys, String searchTerms, Integer maxNumResults) {
        return client.getIssuesFromTextSearchWithProject(token, projectKeys, searchTerms, maxNumResults);
    }

    /**
     * Execute a specified JQL query and return the resulting issues.
     * This method also respects the jira.search.views.max.limit and jira.search.views.max.unlimited.group JIRA properties
     * which will override the max number of results returned.
     * <p/>
     * If the jira.search.views.max.limit property is set and you are not in a group specified by
     * jira.search.views.max.unlimited.group then the number of results returned will be constrained by the value of
     * jira.search.views.max.limit if it is less than the specified maxNumResults.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-issues-from-jql-search}
     *
     * @param jqlSearch     JQL query string to execute
     * @param maxNumResults the maximum number of results that this method will return
     * @return issues matching the JQL query
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getIssuesFromJqlSearch(String jqlSearch, Integer maxNumResults) {
        return client.getIssuesFromJqlSearch(token, jqlSearch, maxNumResults);
    }

    /**
     * Deletes a user in JIRA with the specified username.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:delete-user}
     *
     * @param username the user name to delete
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void deleteUser(
            String username) {
        client.deleteUser(token, username);
    }

    /**
     * Deletes the group denoted by the given group name.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:delete-group}
     *
     * @param groupName     the group name to use
     * @param swapGroupName the swap group name to use
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void deleteGroup(
            String groupName,
            @Optional String swapGroupName) {
        client.deleteGroup(token, groupName, swapGroupName);
    }

    /**
     * Refreshs custom fields for the current user.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:refresh-custom-fields}
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void refreshCustomFields() {
        client.refreshCustomFields(token);
    }

    /**
     * An alternative mechanism for adding attachments to an issue. This method accepts the data of the attachments as
     * Base64 encoded strings instead of byte arrays. This is to combat the XML message bloat created by Axis when
     * SOAP-ifying byte arrays.
     * For more information, please see JRA-11693.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:add-base64-encoded-attachments-to-issue}
     *
     * @param issueKey                    the issue to attach to
     * @param fileNames                   an array of filenames; each element names an attachment to be uploaded
     * @param base64EncodedAttachmentData an array of Base 64 encoded Strings; each element contains the data of the attachment to be uploaded
     * @return true if attachments were successfully added; if the operation was not successful, an exception would be thrown
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public boolean addBase64EncodedAttachmentsToIssue(String issueKey, List<String> fileNames, List<String> base64EncodedAttachmentData) {
        return client.addBase64EncodedAttachmentsToIssue(token, issueKey, fileNames, base64EncodedAttachmentData);
    }

    /**
     * Returns issues that match the saved filter specified by the filterId.
     * This method will return no more than the maxNumResults.
     * <p/>
     * It will start the result set at the provided off set.
     * <p/>
     * This method also respects the jira.search.views.max.limit and jira.search.views.max.unlimited.group JIRA properties
     * which will override the max number of results returned.
     * <p/>
     * If the jira.search.views.max.limit property is set and you are not in a group specified by
     * jira.search.views.max.unlimited.group then the number of results returned will be constrained by the value of
     * jira.search.views.max.limit if it is less than the specified maxNumResults.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-issues-from-filter-with-limit}
     *
     * @param filterId      identifies the saved filter to use for the search.
     * @param offset        the place in the result set to use as the first result returned
     * @param maxNumResults the maximum number of results that this method will return.
     * @return issues matching the saved filter
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getIssuesFromFilterWithLimit(String filterId, Integer offset, Integer maxNumResults) {
        return client.getIssuesFromFilterWithLimit(token, filterId, offset, maxNumResults);
    }

    /**
     * Returns issues containing searchTerms.
     * Note: this is a fuzzy search, returned in order of 'relevance', so the results are only generally useful for human
     * consumption.
     * <p/>
     * This method will return no more than the maxNumResults.
     * <p/>
     * It will start the result set at the provided off set.
     * <p/>
     * This method also respects the jira.search.views.max.limit and jira.search.views.max.unlimited.group JIRA properties
     * which will override the max number of results returned.
     * <p/>
     * If the jira.search.views.max.limit property is set and you are not in a group specified by
     * jira.search.views.max.unlimited.group then the number of results returned will be constrained by the value of
     * jira.search.views.max.limit if it is less than the specified maxNumResults.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-issues-from-text-search-with-limit}
     *
     * @param searchTerms   search terms
     * @param offset        the place in the result set to use as the first result returned
     * @param maxNumResults the maximum number of results that this method will return.
     * @return issues matching the search terms
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getIssuesFromTextSearchWithLimit(String searchTerms, Integer offset, Integer maxNumResults) {
        return client.getIssuesFromTextSearchWithLimit(token, searchTerms, offset, maxNumResults);
    }

    /**
     * Returns an array of all the Projects defined in JIRA.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-projects-no-schemes}
     *
     * @return an array of RemoteProject objects.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getProjectsNoSchemes() {
        return client.getProjectsNoSchemes(token);
    }

    /**
     * Creates a new custom avatar for the given project and sets it to be current for the project. The image data
     * must be provided as base64 encoded data and should be 48 pixels square. If the image is larger, the top left
     * 48 pixels are taken, if it is smaller it is upscaled to 48 pixels. The small version of the avatar image (16
     * pixels) is generated automatically. Project administration permission is required.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:set-new-project-avatar}
     *
     * @param projectKey      the key for the project.
     * @param contentType     the MIME type of the image provided, e.g. image/gif, image/jpeg, image/png.
     * @param base64ImageData a base 64 encoded image, 48 pixels square.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public void setNewProjectAvatar(String projectKey, String contentType, String base64ImageData) {
        client.setNewProjectAvatar(token, projectKey, contentType, base64ImageData);
    }

    /**
     * This will progress an issue through a workflow.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:progress-workflow-action }
     *
     * @param issueKey       the issue to update.
     * @param actionIdString the workflow action to progress to
     * @param fields         the fields to be updated, the key of the map is the field id and the value is a list of values for that field, the values should be separated by a "|" if it is multivalued
     * @return the updated RemoteIssue
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteIssue progressWorkflowAction(String issueKey, String actionIdString, @Optional Map<String, String> fields) {
        Map<String, List<String>> multivaluedFields = convertFieldsToMultivaluedAndMapNamesToIds(fields);
        return client.progressWorkflowAction(token, issueKey, actionIdString, multivaluedFields);
    }

    /**
     * Adds a worklog to the given issue.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:add-worklog-and-auto-adjust-remaining-estimate}
     * 
     * @param issueKey    the key of the issue.
     * @param timeSpent   specifies a time duration in JIRA duration format, representing the time spent working on the worklog, eg 1d 2h.
     * @param startDate   the start date of the worklog using the format MM-dd-yyy'T'HH:mm:ss
     * @param comment     add a comment to the worklog.
     * @param groupLevel  the new group level.
     * @param roleLevelId the new role level id.
     * @return Created worklog with the id set or null if no worklog was created.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteWorklog addWorklogAndAutoAdjustRemainingEstimate(String issueKey, String timeSpent, String startDate, 
                                                                  @Optional String comment, @Optional String groupLevel, @Optional String roleLevelId) {
        return client.addWorklogAndAutoAdjustRemainingEstimate(token, issueKey, timeSpent, startDate, comment, groupLevel, roleLevelId);
    }

    /**
     * Adds a worklog to the given issue and sets the issue's remaining estimate field to the given value.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:add-worklog-with-new-remaining-estimate}
     * 
     * @param issueKey              the key of the issue.
     * @param timeSpent             specifies a time duration in JIRA duration format, representing the time spent working on the worklog, eg 1d 2h.
     * @param startDate             the start date of the worklog using the format MM-dd-yyy'T'HH:mm:ss
     * @param newRemainingEstimate  specifies the issue's remaining estimate as a duration string, eg 1d 2h.
     * @param comment               add a comment to the worklog.
     * @param groupLevel            the new group level.
     * @param roleLevelId           the new role level id.
     * @return Created worklog with the id set or null if no worklog was created.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteWorklog addWorklogWithNewRemainingEstimate(String issueKey, String timeSpent, String startDate, 
                                                            String newRemainingEstimate, @Optional String comment, @Optional String groupLevel, 
                                                            @Optional String roleLevelId) {
        return client.addWorklogWithNewRemainingEstimate(token, issueKey, timeSpent, startDate, newRemainingEstimate, comment, groupLevel, roleLevelId);
    }

    /**
     * Adds a worklog to the given issue but leaves the issue's remaining estimate field unchanged.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:add-worklog-and-retain-remaining-estimate}
     * 
     * @param issueKey    the key of the issue.
     * @param timeSpent   specifies a time duration in JIRA duration format, representing the time spent working on the worklog, eg 1d 2h.
     * @param startDate   the start date of the worklog using the format MM-dd-yyy'T'HH:mm:ss
     * @param comment     add a comment to the worklog.
     * @param groupLevel  the new group level.
     * @param roleLevelId the new role level id.
     * @return Created worklog with the id set or null if no worklog was created.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public RemoteWorklog addWorklogAndRetainRemainingEstimate(String issueKey, String timeSpent, String startDate, 
                                                              @Optional String comment, @Optional String groupLevel, @Optional String roleLevelId) {
        return client.addWorklogAndRetainRemainingEstimate(token, issueKey, timeSpent, startDate, comment, groupLevel, roleLevelId);
    }

    /**
     * Returns the security schemes.
     * <p/>
     * {@sample.xml ../../../doc/mule-module-jira.xml.sample jira:get-security-schemes}
     *
     * @return the security schemes.
     */
    @Processor
    @InvalidateConnectionOn(exception = JiraConnectorException.class)
    public List<Object> getSecuritySchemes() {
        return client.getSecuritySchemes(token);
    }

    public void setClient(JiraClient<?> client) {
        this.client = JiraClientAdaptor.adapt(client);
    }

    public String getConnectionUser() {
        return connectionUser;
    }

    public void setConnectionUser(String connectionUser) {
        this.connectionUser = connectionUser;
    }

    public String getConnectionAddress() {
        return connectionAddress;
    }

    public void setConnectionAddress(String connectionAddress) {
        this.connectionAddress = connectionAddress;
    }

    public Boolean getUseCustomFieldsExternalName() {
        return useCustomFieldsExternalName;
    }

    public void setUseCustomFieldsExternalName(Boolean useCustomFieldsExternalName) {
        this.useCustomFieldsExternalName = useCustomFieldsExternalName;
    }

    /**
     * Creates a connection to Jira by making a login call with the given credentials to the specified address.
     * The login call, if successfull, returns a token which will be used in the subsequent calls to Jira.
     *
     * @param connectionUser     the user login user
     * @param connectionPassword the user login pass
     * @param connectionAddress  the JIRA Server Soap address. It usually looks like https://&lt;jira server hostname&gt;/rpc/soap/jirasoapservice-v2 or http://&lt;jira server hostname&gt;/rpc/soap/jirasoapservice-v2
     */
    @Connect
    public void connect(@ConnectionKey String connectionUser, @Password String connectionPassword, String connectionAddress) throws ConnectionException {
        this.connectionUser = connectionUser;
        this.connectionAddress = connectionAddress;
        setClient(JiraClientFactory.getClient(connectionAddress));
        token = login(connectionUser, connectionPassword);
    }

    /**
     * Performs a logout call to Jira.
     */
    @Disconnect
    public void disconnect() {
        if (token != null) {
            String oldToken = token;
            token = null;
            logout(oldToken);
            client = null;
        }
    }

    /**
     * Returns whether the current user is authenticated. It does not mean tell anything whether the current session
     * has expired
     */
    @ValidateConnection
    public boolean validateConnection() {
        return token != null;
    }

    /**
     * Returns a connection identifier.
     */
    @Override
    @ConnectionIdentifier
    public String toString() {
        return "username='" + connectionUser + '\'' +
                ", address='" + connectionAddress + '\'' +
                '}';
    }
}
