/**
 * (c) 2003-2015 MuleSoft, Inc. The software in this package is published under the terms of the CPAL v1.0 license,
 * a copy of which has been included with this distribution in the LICENSE.md file.
 */

package org.mule.modules.jirarest.client;

import java.net.URI;
import java.util.List;
import java.util.Map;

import javax.ws.rs.ProcessingException;
import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.Invocation;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.GenericType;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriBuilder;

import org.apache.log4j.Logger;
import org.glassfish.jersey.client.ClientConfig;
import org.glassfish.jersey.client.authentication.HttpAuthenticationFeature;
import org.glassfish.jersey.client.filter.EncodingFilter;
import org.glassfish.jersey.jackson1.Jackson1Feature;
import org.glassfish.jersey.message.DeflateEncoder;
import org.glassfish.jersey.message.GZipEncoder;
import org.mule.module.jira.api.rest.JiraClientAuthenticationException;
import org.mule.modules.jirarest.ConnectorConnectionStrategy;

public class JiraClient {

    private Client client;
    private String baseUrl;
    private RestIssueClient issueClient;
    private static final Logger log = Logger.getLogger(JiraClient.class);

    public enum HttpMethod {
        GET, POST, PUT, DELETE
    }

    public JiraClient(ConnectorConnectionStrategy strategy) {
        HttpAuthenticationFeature httpAuthenticationFeature = HttpAuthenticationFeature.basic(strategy.getUsername(),
                strategy.getPassword());

        ClientConfig clientConfig = new ClientConfig();
        clientConfig
        .register(Jackson1Feature.class)
        .register(EncodingFilter.class)
        .register(GZipEncoder.class)
        .register(DeflateEncoder.class)
        .register(RedirectFilter.class)
        .register(httpAuthenticationFeature);
        client = ClientBuilder.newClient(clientConfig);

        this.baseUrl = strategy.getServerUrl();
        this.issueClient = new RestIssueClient(this);
    }

    public Client getClient() {
        return client;
    }

    public void setClient(Client client) {
        this.client = client;
    }

    public RestIssueClient getIssueClient() {
        return issueClient;
    }

    public void setIssueClient(RestIssueClient issueClient) {
        this.issueClient = issueClient;
    }

    public String getBaseUrl() {
        return baseUrl;
    }

    public void setBaseUrl(String baseUrl) {
        this.baseUrl = baseUrl;
    }

    public WebTarget getTarget(URI uri) {
        return client.target(baseUrl + uri);
    }

    private <T> T genericHttpRequest(HttpMethod httpMethod, String url, Object objectRequest, GenericType<T> type) {

        final URI uri = UriBuilder.fromUri(url).build();
        Invocation.Builder builder = getTarget(uri)
                .request(MediaType.APPLICATION_JSON_TYPE);

        Response response = null;
        switch (httpMethod) {
        case POST:
            response = builder.post(Entity.entity(objectRequest, MediaType.APPLICATION_JSON_TYPE));
            break;
        case PUT:
            response = builder.put(Entity.entity(objectRequest, MediaType.APPLICATION_JSON_TYPE));
            break;
        case DELETE:
            response = builder.delete();
            break;
        default:
            response = builder.get();
            break;
        }

        return handleResponse(response, type);
    }

    public Map<String, Object> genericHttpRequestMap(HttpMethod httpMethod, String url, Object objectRequest) {
        return genericHttpRequest(httpMethod, url, objectRequest, new GenericType<Map<String, Object>>() {
        });
    }

    public List<Map<String, Object>> genericHttpRequestList(HttpMethod httpMethod, String url, Object objectRequest) {
        return genericHttpRequest(httpMethod, url, objectRequest, new GenericType<List<Map<String, Object>>>() {
        });
    }

    public Object genericHttpRequestObject(HttpMethod httpMethod, String url, Object objectRequest) {
        return genericHttpRequest(httpMethod, url, objectRequest, new GenericType<Object>() {
        });
    }

    public <T> T get(WebTarget target, Class<T> returnClass) {
        Response response = target.request(MediaType.APPLICATION_JSON).get();
        return handleResponse(response, returnClass);
    }

    public <T> T get(WebTarget target, GenericType<T> returnGenericType) {
        Response response = target.request(MediaType.APPLICATION_JSON).get();
        return handleResponse(response, returnGenericType);
    }

    public <T> T post(WebTarget target, Class<T> returnClass, Object requestObject) {
        Response response = target.request(MediaType.APPLICATION_JSON).post(
                Entity.entity(requestObject, MediaType.APPLICATION_JSON));
        return handleResponse(response, returnClass);
    }

    public <T> T put(WebTarget target, Class<T> returnClass, Object requestObject) {
        Response response = target.request(MediaType.APPLICATION_JSON).put(
                Entity.entity(requestObject, MediaType.APPLICATION_JSON));
        return handleResponse(response, returnClass);
    }

    protected void delete(WebTarget target) {
        Response response = target.request().delete();
        handleResponse(response);
    }

    private <T> T handleResponse(Response response, Class<T> returnClass) {
        if (response.getStatus() == 200) {
            return response.readEntity(returnClass);
        }

        handleResponse(response);
        return null;
    }

    private <T> T handleResponse(Response response, GenericType<T> returnGenericType) {
        if (response.getStatus() == 204) {
            return null;
        }
        if (response.getStatus() >= 200 && response.getStatus() < 300) {
            try {
                return response.readEntity(returnGenericType);
            } catch (ProcessingException | IllegalStateException e) {
                log.warn("No content returned");
            }
            return null;
        }
        handleResponse(response);
        return null;
    }

    private void handleResponse(Response response) {
        if (response.getStatus() >= 200 && response.getStatus() < 300) {
            return;
        } else if (response.getStatus() == 400 || response.getStatus() == 404) {
            log.warn("STATUS: " + response.getStatus());
            log.warn("MESSAGE: " + response.readEntity(String.class));
            return;
        } else if (response.getStatus() == 401 || response.getStatus() == 403) {
            log.error("STATUS: " + response.getStatus());
            log.error("MESSAGE: " + response.readEntity(String.class));
            throw new JiraClientAuthenticationException();
        } else if (response.getStatus() == 500) {
            log.error("STATUS: " + response.getStatus());
            log.error("MESSAGE: " + response.readEntity(String.class));
            throw new JiraClientException(response.getStatus(), response.readEntity(String.class));
        }
    }
}
