/**
 * (c) 2003-2015 MuleSoft, Inc. The software in this package is published under the terms of the CPAL v1.0 license,
 * a copy of which has been included with this distribution in the LICENSE.md file.
 */

package org.mule.modules.jirarest;

import java.io.InputStream;
import java.net.URI;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ws.rs.core.UriBuilder;

import org.mule.api.annotations.Configurable;
import org.mule.api.annotations.ConnectionStrategy;
import org.mule.api.annotations.Connector;
import org.mule.api.annotations.MetaDataScope;
import org.mule.api.annotations.Processor;
import org.mule.api.annotations.ReconnectOn;
import org.mule.api.annotations.param.Default;
import org.mule.api.annotations.param.MetaDataKeyParam;
import org.mule.api.annotations.param.Optional;
import org.mule.module.jira.api.datasense.category.AssigneeCategory;
import org.mule.module.jira.api.datasense.category.CommentCategory;
import org.mule.module.jira.api.datasense.category.CommentsCategory;
import org.mule.module.jira.api.datasense.category.ComponentCategory;
import org.mule.module.jira.api.datasense.category.ComponentPostCategory;
import org.mule.module.jira.api.datasense.category.ComponentsCategory;
import org.mule.module.jira.api.datasense.category.CreateMetaCategory;
import org.mule.module.jira.api.datasense.category.EditMetaCategory;
import org.mule.module.jira.api.datasense.category.GroupCategory;
import org.mule.module.jira.api.datasense.category.IssueCategory;
import org.mule.module.jira.api.datasense.category.IssueLinkCategory;
import org.mule.module.jira.api.datasense.category.IssueLinkGetCategory;
import org.mule.module.jira.api.datasense.category.IssueLinkTypeCategory;
import org.mule.module.jira.api.datasense.category.IssueLinkTypesCategory;
import org.mule.module.jira.api.datasense.category.IssueTypeCategory;
import org.mule.module.jira.api.datasense.category.IssueTypesCategory;
import org.mule.module.jira.api.datasense.category.IssueUpdateCategory;
import org.mule.module.jira.api.datasense.category.NotifyCategory;
import org.mule.module.jira.api.datasense.category.PrioritiesCategory;
import org.mule.module.jira.api.datasense.category.PriorityCategory;
import org.mule.module.jira.api.datasense.category.ProjectCategoriesCategory;
import org.mule.module.jira.api.datasense.category.ProjectCategory;
import org.mule.module.jira.api.datasense.category.ProjectCategoryCategory;
import org.mule.module.jira.api.datasense.category.ProjectStatusesCategory;
import org.mule.module.jira.api.datasense.category.ProjectVersionsCategory;
import org.mule.module.jira.api.datasense.category.ProjectsCategory;
import org.mule.module.jira.api.datasense.category.RemoteLinkCategory;
import org.mule.module.jira.api.datasense.category.RemoteLinksCategory;
import org.mule.module.jira.api.datasense.category.SearchCategory;
import org.mule.module.jira.api.datasense.category.StatusCategory;
import org.mule.module.jira.api.datasense.category.StatusesCategory;
import org.mule.module.jira.api.datasense.category.TransitionCategory;
import org.mule.module.jira.api.datasense.category.TransitionsCategory;
import org.mule.module.jira.api.datasense.category.UserCategory;
import org.mule.module.jira.api.datasense.category.UserGetCategory;
import org.mule.module.jira.api.datasense.category.UsersCategory;
import org.mule.module.jira.api.datasense.category.VersionCategory;
import org.mule.module.jira.api.datasense.category.VotesCategory;
import org.mule.module.jira.api.datasense.category.WatchersCategory;
import org.mule.module.jira.api.datasense.category.WorklogCategory;
import org.mule.module.jira.api.datasense.category.WorklogsCategory;
import org.mule.module.jira.api.rest.JiraClientAuthenticationException;
import org.mule.module.jira.api.rest.JiraRestClientCallWrapper;
import org.mule.module.jira.api.rest.JiraRestClientInvocation;
import org.mule.modules.jirarest.client.JiraClient;
import org.mule.modules.jirarest.model.Attachment;

/**
 * Anypoint Connector
 *
 * @author MuleSoft, Inc.
 */
@Connector(name = "jirarest", schemaVersion = "1.0", friendlyName = "JiraRest")
@ReconnectOn(exceptions = { JiraClientAuthenticationException.class })
public class JiraRestConnector {

    @ConnectionStrategy
    private ConnectorConnectionStrategy strategy;

    public ConnectorConnectionStrategy getStrategy() {
        return strategy;
    }

    public void setStrategy(ConnectorConnectionStrategy strategy) {
        this.strategy = strategy;
    }

    /**
     * Key of a sample Issue for which metadata will be retrieved, if Datasense is enabled. When omitted, default Issue structure will be used
     */
    @Optional
    @Configurable
    private String issueKey;

    public void setIssueKey(String issueKey) {
        this.issueKey = issueKey;
    }

    public String getIssueKey() {
        return issueKey;
    }

    /**
     * Generic rest request
     *
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:a-generic-rest-request}
     *
     * @param httpMethod
     *            The Http method which needs to be performed
     * @param url
     *            relative url endpoint of the rest servise
     * @param body
     *            input body of the method
     * @return REST response body of type Map<String, Object> or List<Map<String, Object>>
     */
    @Processor(friendlyName = "Generic REST request")
    public Object aGenericRestRequest(final JiraClient.HttpMethod httpMethod, final String url, @Optional
    final Map<String, Object> body) {
        return strategy.getJiraClient().genericHttpRequestObject(httpMethod, url, body);
    }

    /**
     * Get info about currently logged user
     *
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:myself}
     *
     * @return map containing currently logged user
     */
    @Processor(friendlyName = "MYSELF - get logged user")
    public Map<String, Object> myself() {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/myself");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    // Attachments

    /**
     * Adds attachment to the issue
     *
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:attachments-add-attachment}
     *
     * @param issueKeyOrId
     *            The id or key of the issue to add the attachment to
     * @param in
     *            stream to upload
     * @param filename
     *            file name of the uploaded attachment
     */
    @Processor(friendlyName = "ATTACHMENTS - Add attachment")
    public void attachmentsAddAttachment(final String issueKeyOrId, @Default("#[payload]")
    final InputStream in, final String filename) {
        final URI attachmentsUri = UriBuilder.fromUri(strategy.getServerUrl())
                .path("/rest/api/2/issue/{issueKeyOrId}/attachments").build(issueKeyOrId);
        JiraRestClientCallWrapper.wrap(new JiraRestClientInvocation<Void>() {
            @Override
            public Void invoke() {
                strategy.getJiraRestClient().getIssueClient().addAttachment(attachmentsUri, in, filename).claim();
                return null;
            }
        });
    }

    /**
     * Retrieves the content of given attachment.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:attachments-get-attachment}
     *
     * @param attachmentLink
     *            attachment link
     * @return stream of the attachment, caller is responsible for closing the stream
     */
    @Processor(friendlyName = "ATTACHMENTS - Get attachment")
    public InputStream attachmentsGetAttachment(final String attachmentLink) {
        final URI attachmentUri = UriBuilder.fromUri(attachmentLink).build();
        return JiraRestClientCallWrapper.wrap(new JiraRestClientInvocation<InputStream>() {
            @Override
            public InputStream invoke() {
                return strategy.getJiraRestClient().getIssueClient().getAttachment(attachmentUri).claim();
            }
        });
    }

    /**
     * Retrieves attachments from given issue.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:attachments-get-attachments-from-issue}
     *
     * @param issueKeyOrId
     *            The id or key of the issue
     * @return list of attachments from the given issue
     */
    @Processor(friendlyName = "ATTACHMENTS - Get attachments from issue")
    public List<Attachment> attachmentsGetAttachmentsFromIssue(final String issueKeyOrId) {
        return strategy.getJiraClient().getIssueClient().getAttachments(issueKeyOrId);
    }

    // Components

    /**
     * Creates a new component
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:components-create-component}
     *
     * @param entityType
     *            type of the entity
     * @param payload
     *            component input map
     * @return Map with full representation of a JIRA component
     */
    @MetaDataScope(ComponentPostCategory.class)
    @Processor(friendlyName = "COMPONENTS - create component ")
    public Map<String, Object> componentsCreateComponent(@MetaDataKeyParam @Default("Components") String entityType,
            @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/component");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.POST, url, payload);
    }

    /**
     * Updates an existing component
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:components-update-component}
     *
     * @param entityType
     *            type of the entity
     * @param id
     *            of the component
     * @param payload
     *            component input map
     * @return Map with full representation of a JIRA component
     */
    @MetaDataScope(ComponentPostCategory.class)
    @Processor(friendlyName = "COMPONENTS - update component ")
    public Map<String, Object> componentsUpdateComponent(@MetaDataKeyParam @Default("ComponentPost") String entityType,
            String id, @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/component").path(id);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.PUT, url, payload);
    }

    /**
     * Gets a component by Id
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:components-get-component}
     *
     * @param entityType
     *            type of the entity
     * @param id
     *            of the component
     * @return Map with full representation of a JIRA component
     */
    @MetaDataScope(ComponentCategory.class)
    @Processor(friendlyName = "COMPONENTS - get component ")
    public Map<String, Object> componentsGetComponent(@MetaDataKeyParam @Default("Component") String entityType,
            String id) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/component").path(id);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Deletes a component by Id
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:components-delete-component}
     *
     * @param id
     *            of the component
     */
    @Processor(friendlyName = "COMPONENTS - delete component ")
    public void componentsDeleteComponent(String id) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/component").path(id);
        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.DELETE, url, null);
    }

    // Groups

    /**
     * Gets a group by group name
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:groups-get-group}
     *
     * @param entityType
     *            type of the entity
     * @param groupName
     *            name of the group
     * @param expand
     *            List of fields to expand. Currently only available expand is "users".
     * @return Map with full representation of a JIRA group
     */
    @MetaDataScope(GroupCategory.class)
    @Processor(friendlyName = "GROUPS - get group ")
    public Map<String, Object> groupsGetGroup(@MetaDataKeyParam @Default("Group") String entityType, String groupName,
            @Optional String expand) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/group").queryParam("groupname", groupName);
        if (expand != null)
            builder = builder.queryParam("expand", expand);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Deletes a group by group name
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:groups-delete-group}
     *
     * @param groupName
     *            a group to delete
     * @param swapGroup
     *            a group to transfer visibility restrictions of the group that is being deleted
     */
    @Processor(friendlyName = "GROUPS - delete group ")
    public void groupsDeleteGroup(String groupName, @Optional String swapGroup) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/group").queryParam("groupname", groupName);
        if (swapGroup != null)
            builder = builder.queryParam("swapGroup", swapGroup);
        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.DELETE, url, null);
    }

    /**
     * Creates a new group
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:groups-create-group}
     *
     * @param entityType
     *            type of the entity
     * @param payload
     *            group input map
     * @return Map with full representation of a JIRA group
     */
    @MetaDataScope(GroupCategory.class)
    @Processor(friendlyName = "GROUPS - create group ")
    public Map<String, Object> groupsCreateGroup(@MetaDataKeyParam @Default("Group") String entityType,
            @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/group");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.POST, url, payload);
    }

    /**
     * Adds given user to a group.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:groups-add-user-to-group}
     *
     * @param groupName
     *            name of the requested group
     * @param user
     *            key or name of the user
     */
    @Processor(friendlyName = "GROUPS - add user to group ")
    public void groupsAddUserToGroup(String groupName, String user) {
        Map<String, Object> payload = new HashMap<String, Object>();
        payload.put("name", user);

        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/group/user").queryParam("groupname", groupName);
        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.POST, url, payload);
    }

    /**
     * Removes given user from a group.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:groups-delete-user-from-group}
     *
     * @param groupName
     *            name of the requested group
     * @param user
     *            key or name of the user
     */
    @Processor(friendlyName = "GROUPS - delete user from group ")
    public void groupsDeleteUserFromGroup(String groupName, String user) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/group/user").queryParam("groupname", groupName)
                .queryParam("username", user);
        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.DELETE, url, null);
    }

    // Issues

    /**
     * Creates an issue or a sub-task from a input Map
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-issue-create}
     *
     * @param entityType
     *            type of the entity
     * @param payload
     *            issue input map
     * @return Map with representation of a JIRA Issue
     */
    @MetaDataScope(IssueCategory.class)
    @Processor(friendlyName = "ISSUES - create issue ")
    public Map<String, Object> issuesIssueCreate(@MetaDataKeyParam @Default("Issue") String entityType,
            @Default("#[payload]") Map<String, Object> payload) {
        final String url = "/rest/api/2/issue";
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.POST, url, payload);
    }

    /**
     * Updates an issue with input. The issue can either be updated by setting explicit the field value(s) or by using an operation to change the
     * field value.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-issue-get-update}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or issue id
     * @param payload
     *            issue update input Map
     * @return Map with representation of a JIRA Issue
     */
    @MetaDataScope(IssueUpdateCategory.class)
    @Processor(friendlyName = "ISSUES - update Issue ")
    public Map<String, Object> issuesIssueGetUpdate(@MetaDataKeyParam @Default("IssueUpdate") String entityType,
            String issueKeyOrId, @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.PUT, url, payload);
    }

    /**
     * Returns a full representation of the issue for the given issue key
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-issue-get}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @param fields
     *            param (which can be specified multiple times) gives a comma-separated list of fields to include in the response. This can be used to
     *            retrieve a subset of fields. A particular field can be excluded by prefixing it with a minus. By default, all (*all) fields are
     *            returned in this get-issue resource. Note: the default is different when doing a jql search -- the default there is just navigable
     *            fields (*navigable).
     * @param expand
     *            comma separated list of entities that you want expanded. (worklog, comments, attachments, etc)
     * @return Map with representation of a JIRA Issue
     */
    @MetaDataScope(IssueCategory.class)
    @Processor(friendlyName = "ISSUES - get Issue ")
    public Map<String, Object> issuesIssueGet(@MetaDataKeyParam @Default("Issue") String entityType,
            String issueKeyOrId, @Optional String fields, @Optional String expand) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId);
        if (fields != null)
            builder = builder.queryParam("fields", fields);
        if (expand != null)
            builder = builder.queryParam("expand", expand);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Deletes an issue by issue key or id
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-issue-delete}
     *
     * @param issueKeyOrId
     *            issue key or id
     * @param deleteSubtasks
     *            Boolean indicating that any subtasks should also be deleted. If the issue has no subtasks this parameter is ignored. If the issue
     *            has subtasks and this parameter is missing or false, then the issue will not be deleted and an error will be returned.
     */
    @Processor(friendlyName = "ISSUES - delete Issue ")
    public void issuesIssueDelete(String issueKeyOrId, @Optional Boolean deleteSubtasks) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId);
        if (deleteSubtasks != null && deleteSubtasks)
            builder = builder.queryParam("deleteSubtasks", "true");
        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.DELETE, url, null);
    }

    // Assignee

    /**
     * Sets assignee for requested issue
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-assignee}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @param payload
     *            assignee request input map
     * @return
     */
    @MetaDataScope(AssigneeCategory.class)
    @Processor(friendlyName = "ASSIGNEE - set  assignee on issue ")
    public void issuesAssignee(@MetaDataKeyParam @Default("Assignee") String entityType, String issueKeyOrId,
            @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("assignee");
        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.PUT, url, payload);
    }

    // Comments

    /**
     * Gets comments for requested issue
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-comments-get}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @param expand
     *            optional flags: renderedBody (provides body rendered in HTML)
     * @return Map with representation of a JIRA Comments
     */
    @MetaDataScope(CommentsCategory.class)
    @Processor(friendlyName = "COMMENTS - get comments for Issue ")
    public Map<String, Object> issuesCommentsGet(@MetaDataKeyParam @Default("Comments") String entityType,
            String issueKeyOrId, @Optional String expand) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("comment");
        if (expand != null)
            builder = builder.queryParam("expand", expand);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Gets a comment by id
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-comment-get}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @param commentId
     *            comment id
     * @param expand
     *            optional flags: renderedBody (provides body rendered in HTML)
     * @return Map with representation of a JIRA Comment
     */
    @MetaDataScope(CommentCategory.class)
    @Processor(friendlyName = "COMMENTS - get single comment of Issue ")
    public Map<String, Object> issuesCommentGet(@MetaDataKeyParam @Default("Comment") String entityType,
            String issueKeyOrId, String commentId, @Optional String expand) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("comment")
                .path(commentId);
        if (expand != null)
            builder = builder.queryParam("expand", expand);

        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Adds new comment to an issue
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-comment-add}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @param payload
     *            comment create input map
     * @param expand
     *            optional flags: renderedBody (provides body rendered in HTML)
     * @return Map with representation of a JIRA Comment
     */
    @MetaDataScope(CommentCategory.class)
    @Processor(friendlyName = "COMMENTS - add comment to Issue ")
    public Map<String, Object> issuesCommentAdd(@MetaDataKeyParam @Default("Comment") String entityType,
            String issueKeyOrId, @Default("#[payload]") Map<String, Object> payload, @Optional String expand) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("comment");
        if (expand != null)
            builder = builder.queryParam("expand", expand);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.POST, url, payload);
    }

    /**
     * Updates existing comment
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-comment-update}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @param commentId
     *            id of the comment
     * @param payload
     *            comment update input map
     * @param expand
     *            optional flags: renderedBody (provides body rendered in HTML)
     * @return Map with representation of a JIRA Comment
     */
    @MetaDataScope(CommentCategory.class)
    @Processor(friendlyName = "COMMENTS - update comment of Issue ")
    public Map<String, Object> issuesCommentUpdate(@MetaDataKeyParam @Default("Comment") String entityType,
            String issueKeyOrId, String commentId, @Default("#[payload]") Map<String, Object> payload,
            @Optional String expand) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("comment")
                .path(commentId);
        if (expand != null)
            builder = builder.queryParam("expand", expand);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.PUT, url, payload);
    }

    /**
     * Deletes comment by id
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-comment-delete}
     *
     * @param issueKeyOrId
     *            issue key or id
     * @param commentId
     *            id of the comment
     */
    @Processor(friendlyName = "COMMENTS - delete comment of Issue ")
    public void issuesCommentDelete(String issueKeyOrId, String commentId) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("comment")
                .path(commentId);
        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.DELETE, url, null);
    }

    // Edit meta

    /**
     * Returns the meta data for editing an issue. The fields in the editmeta correspond to the fields in the edit screen for the issue. Fields not in
     * the screen will not be in the editemeta.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-issue-metadata-get}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @return Map with representation of a JIRA Issue metadata
     */
    @MetaDataScope(EditMetaCategory.class)
    @Processor(friendlyName = "META - get metadata for editing issue ")
    public Map<String, Object> issuesIssueMetadataGet(@MetaDataKeyParam @Default("EditMeta") String entityType,
            String issueKeyOrId) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("editmeta");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    // Notify

    /**
     * Sends a notification (email) to the list or recipients defined in the request.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-notify}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @param payload
     *            notification input map
     */
    @MetaDataScope(NotifyCategory.class)
    @Processor(friendlyName = "NOTIFY - sends a notification (email) to the list or recipients")
    public void issuesNotify(@MetaDataKeyParam @Default("Notify") String entityType, String issueKeyOrId,
            @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("notify");
        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.POST, url, payload);
    }

    // Transition

    /**
     * Get a list of the transitions possible for this issue by the current user, along with fields that are required and their types.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-transitions-get}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @param transitionId
     *            transition id
     * @return Map with representation of a JIRA Issue transition
     */
    @MetaDataScope(TransitionsCategory.class)
    @Processor(friendlyName = "TRANSITIONS - get transitions of an issue ")
    public Map<String, Object> issuesTransitionsGet(@MetaDataKeyParam @Default("Transitions") String entityType,
            String issueKeyOrId, @Optional String transitionId) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("transitions");
        if (transitionId != null)
            builder = builder.queryParam("transitionId", transitionId);

        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Perform a transition on an issue. When performing the transition you can update or set other issue fields.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-transition-perform}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @param payload
     *            transition update input map
     */
    @MetaDataScope(TransitionCategory.class)
    @Processor(friendlyName = "TRANSITIONS - perform transition on issue ")
    public void issuesTransitionPerform(@MetaDataKeyParam @Default("Transition") String entityType,
            String issueKeyOrId, @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("transitions");

        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.POST, url, payload);
    }

    // Remote link

    /**
     * A sub-resource representing the remote issue links on the issue.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-remote-links-get}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @param globalId
     *            The id of the remote issue link to be returned. If not provided, all remote links for the issue are returned.
     * @return List with representation of a JIRA Issue remote links
     */
    @MetaDataScope(RemoteLinksCategory.class)
    @Processor(friendlyName = "REMOTE LINK - get remote links of an issue ")
    public List<Map<String, Object>> issuesRemoteLinksGet(@MetaDataKeyParam @Default("RemoteLinks") String entityType,
            String issueKeyOrId, @Optional String globalId) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("remotelink");
        if (globalId != null)
            builder = builder.queryParam("globalId", globalId);

        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestList(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Creates or updates a remote issue link from a input.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-remote-link-create}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @param payload
     *            remote link input map
     * @return Map with representation of a JIRA Issue remote link
     */
    @MetaDataScope(RemoteLinkCategory.class)
    @Processor(friendlyName = "REMOTE LINK - create or update remote link of an issue ")
    public Map<String, Object> issuesRemoteLinkCreate(@MetaDataKeyParam @Default("RemoteLink") String entityType,
            String issueKeyOrId, @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("remotelink");

        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.POST, url, payload);
    }

    /**
     * Gets a remote link of an Issue by id
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-remote-link-get}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @param linkId
     *            remote link id
     * @return Map with representation of a JIRA Issue remote link
     */
    @MetaDataScope(RemoteLinkCategory.class)
    @Processor(friendlyName = "REMOTE LINK - get remote link of an issue by linkId ")
    public Map<String, Object> issuesRemoteLinkGet(@MetaDataKeyParam @Default("RemoteLink") String entityType,
            String issueKeyOrId, String linkId) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("remotelink")
                .path(linkId);

        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Updates an existing remote link
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-remote-link-update}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @param linkId
     *            remote link id
     * @param payload
     *            remote link input map
     */
    @MetaDataScope(RemoteLinkCategory.class)
    @Processor(friendlyName = "REMOTE LINK - update remote link of an issue by linkId ")
    public void issuesRemoteLinkUpdate(@MetaDataKeyParam @Default("RemoteLink") String entityType, String issueKeyOrId,
            String linkId, @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("remotelink")
                .path(linkId);

        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.PUT, url, payload);
    }

    /**
     * Deletes a remote link
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-remote-link-delete}
     *
     * @param issueKeyOrId
     *            issue key or id
     * @param linkId
     *            link id
     */
    @Processor(friendlyName = "REMOTE LINK - delete remote link of an issue by linkId ")
    public void issuesRemoteLinkDelete(String issueKeyOrId, String linkId) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("remotelink")
                .path(linkId);

        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.DELETE, url, null);
    }

    // Votes

    /**
     * Get voters of an issue
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-votes-get}
     *
     * @param entityType
     *            type of an entity
     * @param issueKeyOrId
     *            issue key or id
     * @return Map with representation of voters
     */
    @MetaDataScope(VotesCategory.class)
    @Processor(friendlyName = "VOTES - get voters for issue ")
    public Map<String, Object> issuesVotesGet(@MetaDataKeyParam @Default("Votes") String entityType, String issueKeyOrId) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("votes");

        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Cast your vote in favour of an issue.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-vote-add}
     *
     * @param issueKeyOrId
     *            issue key or id
     * @return Map with representation of a JIRA vote
     */
    @Processor(friendlyName = "VOTES - add vote on issue ")
    public Map<String, Object> issuesVoteAdd(String issueKeyOrId) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("votes");

        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.POST, url, null);
    }

    /**
     * Remove your vote from an issue. (i.e. "unvote")
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-vote-delete}
     *
     * @param issueKeyOrId
     *            issue key or id
     */
    @Processor(friendlyName = "VOTES - remove your vote on issue ")
    public void issuesVoteDelete(String issueKeyOrId) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("votes");

        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.DELETE, url, null);
    }

    // Watchers

    /**
     * Returns the list of watchers for the issue with the given key.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-watchers-get}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @return List with representation of watchers
     */
    @MetaDataScope(WatchersCategory.class)
    @Processor(friendlyName = "WATCHERS - get watchers for issue ")
    public Map<String, Object> issuesWatchersGet(@MetaDataKeyParam @Default("Watchers") String entityType,
            String issueKeyOrId) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("watchers");

        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Adds a user to an issue's watcher list.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-watcher-create}
     *
     * @param issueKeyOrId
     *            issue key or id
     * @param user
     *            name of the user
     * @return
     */
    @Processor(friendlyName = "WATCHERS - add user to issue's watchlist ")
    public void issuesWatcherCreate(String issueKeyOrId, String user) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("watchers");

        final String url = builder.build().toString();
        String userParam = user;
        if (!user.contains("\""))
            userParam = "\"" + user + "\"";
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.POST, url, userParam);
    }

    /**
     * Removes a user from an issue's watcher list.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-watcher-delete}
     *
     * @param issueKeyOrId
     *            issue key or id
     * @param user
     *            name of the user
     */
    @Processor(friendlyName = "WATCHERS - remove a watcher from issue's watchlist ")
    public void issuesWatcherDelete(String issueKeyOrId, String user) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("watchers");
        builder = builder.queryParam("username", user);
        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.DELETE, url, null);
    }

    // Worklog

    /**
     * Returns all work logs for an issue.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-worklogs-get}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @return Map with representation of a JIRA Issue worklogs
     */
    @MetaDataScope(WorklogsCategory.class)
    @Processor(friendlyName = "WORKLOG - get worklogs for issue ")
    public Map<String, Object> issuesWorklogsGet(@MetaDataKeyParam @Default("Worklogs") String entityType,
            String issueKeyOrId) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("worklog");

        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Adds a new worklog entry to an issue.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-worklog-create}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @param payload
     *            worklog input map
     * @return Map with representation of a JIRA Issue worklog
     */
    @MetaDataScope(WorklogCategory.class)
    @Processor(friendlyName = "WORKLOG - add worklog for issue ")
    public Map<String, Object> issuesWorklogCreate(@MetaDataKeyParam @Default("Worklog") String entityType,
            String issueKeyOrId, @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("worklog");

        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.POST, url, payload);
    }

    /**
     * Returns a specific worklog.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-worklog-create}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @param id
     *            worklog id
     * @return Map with representation of a JIRA Issue worklog
     */
    @MetaDataScope(WorklogCategory.class)
    @Processor(friendlyName = "WORKLOG - get worklog for issue by id ")
    public Map<String, Object> issuesWorklogGet(@MetaDataKeyParam @Default("Worklog") String entityType,
            String issueKeyOrId, String id) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("worklog").path(id);

        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Updates an existing worklog entry
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-worklog-update}
     *
     * @param entityType
     *            type of the entity
     * @param issueKeyOrId
     *            issue key or id
     * @param id
     *            worklog id
     * @param payload
     *            worklog update map
     * @param adjustEstimate
     *            (optional) allows you to provide specific instructions to update the remaining time estimate of the issue. Valid values are "new" -
     *            sets the estimate to a specific value "leave"- leaves the estimate as is "auto"- Default option. Will automatically adjust the value
     *            based on the new timeSpent specified on the worklog
     * @param newEstimate
     *            (required when "new" is selected for adjustEstimate) the new value for the remaining estimate field.
     * @return Map with representation of a JIRA Issue worklog
     */
    @MetaDataScope(WorklogCategory.class)
    @Processor(friendlyName = "WORKLOG - update worklog for issue ")
    public Map<String, Object> issuesWorklogUpdate(@MetaDataKeyParam @Default("Worklog") String entityType,
            String issueKeyOrId, String id, @Default("#[payload]") Map<String, Object> payload,
            @Optional String adjustEstimate, @Optional String newEstimate) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("worklog").path(id);
        if (adjustEstimate != null)
            builder = builder.queryParam("adjustEstimate", adjustEstimate);
        if (newEstimate != null)
            builder = builder.queryParam("newEstimate", newEstimate);

        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.PUT, url, payload);
    }

    /**
     * Deletes specific worklog entry
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-worklog-delete}
     *
     * @param issueKeyOrId
     *            issue key or id
     * @param id
     *            worklog id
     * @param adjustEstimate
     *            (optional) allows you to provide specific instructions to update the remaining time estimate of the issue. Valid values are "new" -
     *            sets the estimate to a specific value "leave"- leaves the estimate as is "manual" - specify a specific amount to increase remaining
     *            estimate by "auto"- Default option. Will automatically adjust the value based on the new timeSpent specified on the worklog
     * @param newEstimate
     *            (required when "new" is selected for adjustEstimate) the new value for the remaining estimate field.
     * @param increaseBy
     *            (required when "manual" is selected for adjustEstimate) the amount to increase the remaining estimate by e.g. "2d"
     * @return
     */
    @Processor(friendlyName = "WORKLOG - delete worklog for issue by id ")
    public void issuesWorklogDelete(String issueKeyOrId, String id, @Optional String adjustEstimate,
            @Optional String newEstimate, @Optional String increaseBy) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path(issueKeyOrId).path("worklog").path(id);
        if (adjustEstimate != null)
            builder = builder.queryParam("adjustEstimate", adjustEstimate);
        if (newEstimate != null)
            builder = builder.queryParam("newEstimate", newEstimate);
        if (increaseBy != null)
            builder = builder.queryParam("increaseBy", increaseBy);

        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.DELETE, url, null);
    }

    // Create meta

    /**
     * Returns the meta data for creating issues. This includes the available projects, issue types and fields, including field types and whether or
     * not those fields are required. Projects will not be returned if the user does not have permission to create issues in that project.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-create-meta-get}
     *
     * @param entityType
     *            type of the entity
     * @param projectIds
     *            combined with the projectKeys param, lists the projects with which to filter the results. If absent, all projects are returned. This
     *            parameter can be specified multiple times, and/or be a comma-separated list. Specifiying a project that does not exist (or that you
     *            cannot create issues in) is not an error, but it will not be in the results.
     * @param projectKeys
     *            combined with the projectIds param, lists the projects with which to filter the results. If null, all projects are returned. This
     *            parameter can be specified multiple times, and/or be a comma-separated list. Specifiying a project that does not exist (or that you
     *            cannot create issues in) is not an error, but it will not be in the results.
     * @param issueTypeIds
     *            combinded with issuetypeNames, lists the issue types with which to filter the results. If null, all issue types are returned. This
     *            parameter can be specified multiple times, and/or be a comma-separated list. Specifiying an issue type that does not exist is not an
     *            error.
     * @param issueTypeNames
     *            combinded with issuetypeIds, lists the issue types with which to filter the results. If null, all issue types are returned. This
     *            parameter can be specified multiple times, but is NOT interpreted as a comma-separated list. Specifiying an issue type that does not
     *            exist is not an error.
     * @return Map with representation of a JIRA create meta object
     */
    @MetaDataScope(CreateMetaCategory.class)
    @Processor(friendlyName = "CREATEMETA - returns the meta data for creating issues ")
    public Map<String, Object> issuesCreateMetaGet(@MetaDataKeyParam @Default("CreateMeta") String entityType,
            @Optional String projectIds, @Optional String projectKeys, @Optional String issueTypeIds,
            @Optional String issueTypeNames) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issue").path("createmeta");
        if (projectIds != null)
            builder = builder.queryParam("projectIds", projectIds);
        if (projectKeys != null)
            builder = builder.queryParam("projectKeys", projectKeys);
        if (issueTypeIds != null)
            builder = builder.queryParam("issuetypeIds", issueTypeIds);
        if (issueTypeNames != null)
            builder = builder.queryParam("issuetypeNames", issueTypeNames);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    // Issue link

    /**
     * Creates an issue link between two issues. The user requires the link issue permission for the issue which will be linked to another issue. The
     * specified link type in the request is used to create the link and will create a link from the first issue to the second issue using the outward
     * description. It also create a link from the second issue to the first issue using the inward description of the issue link type. It will add
     * the supplied comment to the first issue. The comment can have a restriction who can view it. If group is specified, only users of this group
     * can view this comment, if roleLevel is specified only users who have the specified role can view this comment. The user who creates the issue
     * link needs to belong to the specified group or have the specified role.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-issue-link-create}
     *
     * @param entityType
     *            type of the entity
     * @param payload
     *            issue link create map
     * @return Map with representation of a JIRA issue link
     */
    @MetaDataScope(IssueLinkCategory.class)
    @Processor(friendlyName = "ISSUELINK - create link between issues ")
    public Map<String, Object> issuesIssueLinkCreate(@MetaDataKeyParam @Default("IssueLink") String entityType,
            @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issueLink");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.POST, url, payload);
    }

    /**
     * Returns an issue link with the specified id.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-issue-link-get}
     *
     * @param entityType
     *            type of the entity
     * @param linkId
     *            id of the link
     * @return Map with representation of a JIRA issue link
     */
    @MetaDataScope(IssueLinkGetCategory.class)
    @Processor(friendlyName = "ISSUELINK - get link between issues ")
    public Map<String, Object> issuesIssueLinkGet(@MetaDataKeyParam @Default("IssueLink") String entityType,
            String linkId) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issueLink").path(linkId);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Deletes an issue link with the specified id. To be able to delete an issue link you must be able to view both issues and must have the link
     * issue permission for at least one of the issues.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-issue-link-delete}
     *
     * @param linkId
     *            id of the link
     */
    @Processor(friendlyName = "ISSUELINK - delete link between issues ")
    public void issuesIssueLinkDelete(String linkId) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issueLink").path(linkId);
        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.DELETE, url, null);
    }

    // Issue link type

    /**
     * Returns a list of available issue link types, if issue linking is enabled. Each issue link type has an id, a name and a label for the outward
     * and inward link relationship.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-issue-link-types-get}
     *
     * @param entityType
     *            type of the entity
     * @return Map with representation of a JIRA issue link types
     */
    @MetaDataScope(IssueLinkTypesCategory.class)
    @Processor(friendlyName = "ISSUELINKTYPE - get issue link types ")
    public Map<String, Object> issuesIssueLinkTypesGet(@MetaDataKeyParam @Default("IssueLinkTypes") String entityType) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issueLinkType");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Returns for a given issue link type id all information about this issue link type.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-issue-link-type-get}
     *
     * @param entityType
     *            type of the entity
     * @param issueLinkTypeId
     *            issue link type id
     * @return Map with representation of a JIRA issue link type
     */
    @MetaDataScope(IssueLinkTypeCategory.class)
    @Processor(friendlyName = "ISSUELINKTYPE - get issue link type by id ")
    public Map<String, Object> issuesIssueLinkTypeGet(@MetaDataKeyParam @Default("IssueLinkType") String entityType,
            String issueLinkTypeId) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issueLinkType").path(issueLinkTypeId);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Create a new issue link type.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-issue-link-type-create}
     *
     * @param entityType
     *            type of the entity
     * @param payload
     *            issue link type input map
     * @return Map with representation of a JIRA issue link type
     */
    @MetaDataScope(IssueLinkTypeCategory.class)
    @Processor(friendlyName = "ISSUELINKTYPE - create issue link type ")
    public Map<String, Object> issuesIssueLinkTypeCreate(@MetaDataKeyParam @Default("IssueLinkType") String entityType,
            @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issueLinkType");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.POST, url, payload);
    }

    /**
     * Update the specified issue link type.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-issue-link-type-update}
     *
     * @param entityType
     *            type of the entity
     * @param issueLinkTypeId
     *            issue link type id
     * @param payload
     *            issue link type input map
     * @return Map with representation of a JIRA issue link type
     */
    @MetaDataScope(IssueLinkTypeCategory.class)
    @Processor(friendlyName = "ISSUELINKTYPE - update issue link type ")
    public Map<String, Object> issuesIssueLinkTypeUpdate(@MetaDataKeyParam @Default("IssueLinkType") String entityType,
            String issueLinkTypeId, @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issueLinkType").path(issueLinkTypeId);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.PUT, url, payload);
    }

    /**
     * Delete the specified issue link type.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-issue-link-type-delete}
     *
     * @param issueLinkTypeId
     *            issue link type id
     */
    @Processor(friendlyName = "ISSUELINKTYPE - delete issue link type ")
    public void issuesIssueLinkTypeDelete(String issueLinkTypeId) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issueLinkType").path(issueLinkTypeId);
        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.DELETE, url, null);
    }

    // Issue type

    /**
     * Returns a list of all issue types visible to the user
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-issue-types-get}
     *
     * @param entityType
     *            type of the entity
     * @return List with representation of a JIRA issue types
     */
    @MetaDataScope(IssueTypesCategory.class)
    @Processor(friendlyName = "ISSUETYPE - get issue types ")
    public List<Map<String, Object>> issuesIssueTypesGet(@MetaDataKeyParam @Default("IssueTypes") String entityType) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issuetype");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestList(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Returns a full representation of the issue type that has the given id.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-issue-type-get}
     *
     * @param entityType
     *            type of the entity
     * @param id
     *            issue type id
     * @return Map with representation of a JIRA issue type
     */
    @MetaDataScope(IssueTypeCategory.class)
    @Processor(friendlyName = "ISSUETYPE - get issue type with by id Datasense")
    public Map<String, Object> issuesIssueTypeGet(@MetaDataKeyParam @Default("IssueType") String entityType, String id) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/issuetype").path(id);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    // Issue priority

    /**
     * Returns a list of all issue priorities.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-priorities-get}
     *
     * @param entityType
     *            type of the entity
     * @return List with representation of a JIRA issue priorities
     */
    @MetaDataScope(PrioritiesCategory.class)
    @Processor(friendlyName = "PRIORITY - get priorities ")
    public List<Map<String, Object>> issuesPrioritiesGet(@MetaDataKeyParam @Default("Priorities") String entityType) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/priority");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestList(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Returns an issue priority.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-priority-get}
     *
     * @param entityType
     *            type of the entity
     * @param id
     *            the id of the priority
     * @return Map with representation of a JIRA issue priority
     */
    @MetaDataScope(PriorityCategory.class)
    @Processor(friendlyName = "PRIORITY - get priority by id ")
    public Map<String, Object> issuesPriorityGet(@MetaDataKeyParam @Default("Priority") String entityType, String id) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/priority").path(id);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    // Issue status

    /**
     * Returns a list of all issue statuses.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-statuses-get}
     *
     * @param entityType
     *            type of the entity
     * @return List with representation of a JIRA issue statuses
     */
    @MetaDataScope(StatusesCategory.class)
    @Processor(friendlyName = "STATUS - get statuses ")
    public List<Map<String, Object>> issuesStatusesGet(@MetaDataKeyParam @Default("Statuses") String entityType) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/status");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestList(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Returns an issue status.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:issues-status-get}
     *
     * @param entityType
     *            type of the entity
     * @param idOrName
     *            the id of the status
     * @return Map with representation of a JIRA issue status
     */
    @MetaDataScope(StatusCategory.class)
    @Processor(friendlyName = "STATUS - get status by id ")
    public Map<String, Object> issuesStatusGet(@MetaDataKeyParam @Default("Status") String entityType, String idOrName) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/status").path(idOrName);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    // Project

    /**
     * Returns all projects which are visible for the currently logged in user. If no user is logged in, it returns the list of projects that are
     * visible when using anonymous access.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:projects-projects-get}
     *
     * @param entityType
     *            type of the entity
     * @return List with representation of a JIRA projects
     */
    @MetaDataScope(ProjectsCategory.class)
    @Processor(friendlyName = "PROJECT - get projects ")
    public List<Map<String, Object>> projectsProjectsGet(@MetaDataKeyParam @Default("Projects") String entityType) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/project");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestList(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Gets a full representation of requested project
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:projects-project-get}
     *
     * @param entityType
     *            type of the entity
     * @param projectIdOrKey
     *            project id or key
     * @return Map with representation of a JIRA project
     */
    @MetaDataScope(ProjectCategory.class)
    @Processor(friendlyName = "PROJECT - get project ")
    public Map<String, Object> projectsProjectGet(@MetaDataKeyParam @Default("Project") String entityType,
            String projectIdOrKey) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/project").path(projectIdOrKey);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Gets a full representation of the specified project's components.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:projects-components-get}
     *
     * @param entityType
     *            type of the entity
     * @param projectIdOrKey
     *            project id or key
     * @return List with representation of a JIRA project components
     */
    @MetaDataScope(ComponentsCategory.class)
    @Processor(friendlyName = "PROJECT - get project components ")
    public List<Map<String, Object>> projectsComponentsGet(@MetaDataKeyParam @Default("Components") String entityType,
            String projectIdOrKey) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/project").path(projectIdOrKey).path("components");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestList(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Gets a full representation of the specified project's statuses.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:projects-statuses-get}
     *
     * @param entityType
     *            type of the entity
     * @param projectIdOrKey
     *            project id or key
     * @return List with representation of a JIRA project statuses
     */
    @MetaDataScope(ProjectStatusesCategory.class)
    @Processor(friendlyName = "PROJECT - get project statuses ")
    public List<Map<String, Object>> projectsStatusesGet(
            @MetaDataKeyParam @Default("ProjectStatuses") String entityType, String projectIdOrKey) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/project").path(projectIdOrKey).path("statuses");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestList(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Gets a full representation of the specified project's versions.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:projects-versions-get}
     *
     * @param entityType
     *            type of the entity
     * @param projectIdOrKey
     *            project id or key
     * @return List with representation of a JIRA project versions
     */
    @MetaDataScope(ProjectVersionsCategory.class)
    @Processor(friendlyName = "PROJECT - get project versions ")
    public List<Map<String, Object>> projectsVersionsGet(
            @MetaDataKeyParam @Default("ProjectVersions") String entityType, String projectIdOrKey) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/project").path(projectIdOrKey).path("versions");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestList(JiraClient.HttpMethod.GET, url, null);
    }

    // Project category

    /**
     * Gets a full representation of the project categories.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:project-categories-get}
     *
     * @param entityType
     *            type of the entity
     * @return List with representation of a JIRA project categories
     */
    @MetaDataScope(ProjectCategoriesCategory.class)
    @Processor(friendlyName = "PROJECT CATEGORY - get project categories ")
    public List<Map<String, Object>> projectCategoriesGet(
            @MetaDataKeyParam @Default("ProjectCategories") String entityType) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/projectCategory");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestList(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Create a project category
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:project-category-create}
     *
     * @param entityType
     *            type of the entity
     * @param payload
     *            project category input map
     * @return Map with representation of JIRA project category
     */
    @MetaDataScope(ProjectCategoryCategory.class)
    @Processor(friendlyName = "PROJECT CATEGORY - create project category ")
    public Map<String, Object> projectCategoryCreate(@MetaDataKeyParam @Default("ProjectCategory") String entityType,
            @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/projectCategory");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.POST, url, payload);
    }

    /**
     * Gets a specific project category
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:project-category-get}
     *
     * @param entityType
     *            type of the entity
     * @param categoryId
     *            id of the category
     * @return Map with representation of JIRA project category
     */
    @MetaDataScope(ProjectCategoryCategory.class)
    @Processor(friendlyName = "PROJECT CATEGORY - get project category ")
    public Map<String, Object> projectCategoryGet(@MetaDataKeyParam @Default("ProjectCategory") String entityType,
            String categoryId) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/projectCategory").path(categoryId);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Updates a project category
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:project-category-update}
     *
     * @param entityType
     *            type of the entity
     * @param categoryId
     *            id of the category
     * @param payload
     *            project category input map
     * @return Map with representation of JIRA project category
     */
    @MetaDataScope(ProjectCategoryCategory.class)
    @Processor(friendlyName = "PROJECT CATEGORY - update project category ")
    public Map<String, Object> projectCategoryUpdate(@MetaDataKeyParam @Default("ProjectCategory") String entityType,
            String categoryId, @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/projectCategory").path(categoryId);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.PUT, url, payload);
    }

    /**
     * Deletes a specific project category
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:project-category-delete}
     *
     * @param categoryId
     *            id of the category
     */
    @Processor(friendlyName = "PROJECT CATEGORY - delete project category ")
    public void projectCategoryDelete(String categoryId) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/projectCategory").path(categoryId);
        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.DELETE, url, null);
    }

    // Search

    /**
     * Searches for issues using JQL.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:search}
     *
     * @param entityType
     *            type of the entity
     * @param jql
     *            a JQL query string
     * @param startAt
     *            the index of the first issue to return (0-based)
     * @param maxResults
     *            the maximum number of issues to return (defaults to 50). The maximum allowable value is dictated by the JIRA property
     *            'jira.search.views.default.max'. If you specify a value that is higher than this number, your search results will be truncated.
     * @param validateQuery
     *            whether to validate the JQL query
     * @param fields
     *            the list of fields to return for each issue. By default, all navigable fields are returned.
     * @param expand
     *            A comma-separated list of the parameters to expand.
     * @return Map with representation of JIRA issues
     */
    @MetaDataScope(SearchCategory.class)
    @Processor(friendlyName = "SEARCH - searches for issues using JQL. ")
    public Map<String, Object> search(@MetaDataKeyParam @Default("Search") String entityType, String jql,
            @Optional String startAt, @Optional String maxResults, @Optional Boolean validateQuery,
            @Optional String fields, @Optional String expand) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/search").queryParam("jql", jql);
        if (startAt != null)
            builder = builder.queryParam("startAt", startAt);
        if (maxResults != null)
            builder = builder.queryParam("maxResults", maxResults);
        if (validateQuery != null && validateQuery)
            builder = builder.queryParam("validateQuery", "true");
        if (fields != null)
            builder = builder.queryParam("fields", fields);
        if (expand != null)
            builder = builder.queryParam("expand", expand);

        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    // User
    /**
     * Gets requested user
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:user-get}
     *
     * @param entityType
     *            type of the entity
     * @param user
     *            the user name (specify only one)
     * @param key
     *            the user key
     * @return Map with representation of JIRA user
     */
    @MetaDataScope(UserGetCategory.class)
    @Processor(friendlyName = "USER - get user ")
    public Map<String, Object> userGet(@MetaDataKeyParam @Default("UserGet") String entityType, @Optional String user,
            @Optional String key) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/user");
        if (user != null)
            builder = builder.queryParam("username", user);
        if (key != null)
            builder = builder.queryParam("key", key);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Updates requested user.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:user-update}
     *
     * @param entityType
     *            type of the entity
     * @param payload
     *            user input map
     * @param username
     *            name of the JIRA user, provide username or key, but not both
     * @param key
     *            key of the JIRA user, provide username or key, but not both
     * @return Map with representation of JIRA user
     */
    @MetaDataScope(UserCategory.class)
    @Processor(friendlyName = "USER - update user ")
    public Map<String, Object> userUpdate(@MetaDataKeyParam @Default("User") String entityType,
            @Default("#[payload]") Map<String, Object> payload, @Optional String username, @Optional String key) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/user");
        if (username != null)
            builder = builder.queryParam("username", username);
        if (key != null)
            builder = builder.queryParam("key", key);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.PUT, url, payload);
    }

    /**
     * Create user. By default created user will not be notified with email. If password field is not set then password will be randomly generated.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:user-create}
     *
     * @param entityType
     *            type of the entity
     * @param payload
     *            user input map
     * @return Map with representation of JIRA user
     */
    @MetaDataScope(UserCategory.class)
    @Processor(friendlyName = "USER - create user ")
    public Map<String, Object> userCreate(@MetaDataKeyParam @Default("User") String entityType,
            @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/user");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.POST, url, payload);
    }

    /**
     * Deletes requested user
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:user-delete}
     *
     * @param user
     *            the user name (specify only one)
     * @param key
     *            the user key
     */
    @MetaDataScope(UserCategory.class)
    @Processor(friendlyName = "USER - delete user ")
    public void userDelete(@Optional String user, @Optional String key) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/user");
        if (user != null)
            builder = builder.queryParam("username", user);
        if (key != null)
            builder = builder.queryParam("key", key);
        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.DELETE, url, null);
    }

    /**
     * Returns a list of users that match the search string. This resource cannot be accessed anonymously. Please note that this resource should be
     * called with an issue key when a list of assignable users is retrieved for editing. For create only a project key should be supplied. The list
     * of assignable users may be incorrect if it's called with the project key for editing.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:user-get-assignable}
     *
     * @param entityType
     *            type of the entity
     * @param user
     *            user name
     * @param project
     *            the key of the project we are finding assignable users for
     * @param issueKey
     *            the issue key for the issue being edited we need to find assignable users for.
     * @param startAt
     *            the index of the first user to return (0-based)
     * @param maxResults
     *            the maximum number of users to return (defaults to 50). The maximum allowed value is 1000. If you specify a value that is higher
     *            than this number, your search results will be truncated.
     * @return List with representation of JIRA users
     */
    @MetaDataScope(UsersCategory.class)
    @Processor(friendlyName = "USER - get assignable users ")
    public List<Map<String, Object>> userGetAssignable(@MetaDataKeyParam @Default("Users") String entityType,
            @Optional String user, @Optional String project, @Optional String issueKey, @Optional String startAt,
            @Optional String maxResults) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/user/assignable/search");
        if (user != null)
            builder = builder.queryParam("username", user);
        if (project != null)
            builder = builder.queryParam("project", project);
        if (issueKey != null)
            builder = builder.queryParam("issueKey", issueKey);
        if (startAt != null)
            builder = builder.queryParam("startAt", startAt);
        if (maxResults != null)
            builder = builder.queryParam("maxResults", maxResults);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestList(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Returns a list of users that match the search string and can be assigned issues for all the given projects. This resource cannot be accessed
     * anonymously.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:user-get-assignable-project}
     *
     * @param entityType
     *            type of the entity
     * @param user
     *            user name
     * @param projectKeys
     *            the keys of the projects we are finding assignable users for, comma-separated
     * @param startAt
     *            the index of the first user to return (0-based)
     * @param maxResults
     *            the maximum number of users to return (defaults to 50). The maximum allowed value is 1000. If you specify a value that is higher
     *            than this number, your search results will be truncated.
     * @return List with representation of JIRA users
     */
    @MetaDataScope(UsersCategory.class)
    @Processor(friendlyName = "USER - get assignable users multiproject search ")
    public List<Map<String, Object>> userGetAssignableProject(@MetaDataKeyParam @Default("Users") String entityType,
            @Optional String user, @Optional String projectKeys, @Optional String startAt, @Optional String maxResults) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/user/assignable/multiProjectSearch");
        if (user != null)
            builder = builder.queryParam("username", user);
        if (projectKeys != null)
            builder = builder.queryParam("projectKeys", projectKeys);
        if (startAt != null)
            builder = builder.queryParam("startAt", startAt);
        if (maxResults != null)
            builder = builder.queryParam("maxResults", maxResults);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestList(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Returns a list of users that match the search string. This resource cannot be accessed anonymously.
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:user-search}
     *
     * @param entityType
     *            type of the entity
     * @param user
     *            user name
     * @param startAt
     *            the index of the first user to return (0-based)
     * @param maxResults
     *            the maximum number of users to return (defaults to 50). The maximum allowed value is 1000. If you specify a value that is higher
     *            than this number, your search results will be truncated.
     * @param includeActive
     *            If true, then active users are included in the results (default true)
     * @param includeInactive
     *            If true, then inactive users are included in the results (default false)
     * @return List with representation of JIRA users
     */
    @MetaDataScope(UsersCategory.class)
    @Processor(friendlyName = "USER - search users ")
    public List<Map<String, Object>> userSearch(@MetaDataKeyParam @Default("Users") String entityType,
            @Optional String user, @Optional String startAt, @Optional String maxResults,
            @Optional Boolean includeActive, @Optional Boolean includeInactive) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/user/search");
        if (user != null)
            builder = builder.queryParam("username", user);
        if (startAt != null)
            builder = builder.queryParam("startAt", startAt);
        if (maxResults != null)
            builder = builder.queryParam("maxResults", maxResults);
        if (includeActive != null)
            builder = builder.queryParam("includeActive", "true");
        if (includeInactive != null)
            builder = builder.queryParam("includeInactive", "true");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestList(JiraClient.HttpMethod.GET, url, null);
    }

    // Version

    /**
     * Create a version
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:version-create}
     *
     * @param entityType
     *            type of the entity
     * @param payload
     *            version input map
     * @return Map with representation of JIRA version
     */
    @MetaDataScope(VersionCategory.class)
    @Processor(friendlyName = "VERSIONS - create version ")
    public Map<String, Object> versionCreate(@MetaDataKeyParam @Default("Version") String entityType,
            @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/version");
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.POST, url, payload);
    }

    /**
     * Gets a specific version
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:version-get}
     *
     * @param entityType
     *            type of the entity
     * @param id
     *            the id of the version
     * @return Map with representation of JIRA version
     */
    @MetaDataScope(VersionCategory.class)
    @Processor(friendlyName = "VERSIONS - get version ")
    public Map<String, Object> versionGet(@MetaDataKeyParam @Default("Version") String entityType, String id) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/version").path(id);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Updates a version
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:version-update}
     *
     * @param entityType
     *            type of the entity
     * @param id
     *            the id of the version
     * @param payload
     *            version input map
     * @return Map with representation of JIRA version
     */
    @MetaDataScope(VersionCategory.class)
    @Processor(friendlyName = "VERSIONS - update version ")
    public Map<String, Object> versionUpdate(@MetaDataKeyParam @Default("Version") String entityType, String id,
            @Default("#[payload]") Map<String, Object> payload) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/version").path(id);
        final String url = builder.build().toString();
        return strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.PUT, url, payload);
    }

    /**
     * Deletes a version
     * <p/>
     * {@sample.xml ../../../doc/jirarest-connector.xml.sample jirarest:version-delete}
     *
     * @param id
     *            the id of the version
     */
    @Processor(friendlyName = "VERSIONS - delete version ")
    public void versionDelete(String id) {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/version").path(id);
        final String url = builder.build().toString();
        strategy.getJiraClient().genericHttpRequestMap(JiraClient.HttpMethod.DELETE, url, null);
    }
}
