/**
 * (c) 2003-2015 MuleSoft, Inc. The software in this package is published under the terms of the CPAL v1.0 license,
 * a copy of which has been included with this distribution in the LICENSE.md file.
 */

package org.mule.modules.jirarest;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;

import javax.ws.rs.ProcessingException;
import javax.ws.rs.core.UriBuilder;

import org.apache.log4j.Logger;
import org.mule.api.ConnectionException;
import org.mule.api.ConnectionExceptionCode;
import org.mule.api.annotations.Connect;
import org.mule.api.annotations.ConnectionIdentifier;
import org.mule.api.annotations.Disconnect;
import org.mule.api.annotations.TestConnectivity;
import org.mule.api.annotations.ValidateConnection;
import org.mule.api.annotations.components.ConnectionManagement;
import org.mule.api.annotations.display.Password;
import org.mule.api.annotations.param.ConnectionKey;
import org.mule.module.jira.api.rest.JiraClientAuthenticationException;
import org.mule.modules.jirarest.client.JiraClient;

import com.atlassian.jira.rest.client.api.JiraRestClient;
import com.atlassian.jira.rest.client.api.JiraRestClientFactory;
import com.atlassian.jira.rest.client.internal.async.AsynchronousJiraRestClientFactory;

@ConnectionManagement(configElementName = "config-type", friendlyName = "Basic Authentification")
public class ConnectorConnectionStrategy {

    private static final Logger log = Logger.getLogger(JiraClient.class);
    private JiraRestClient jiraRestClient;
    private JiraClient jiraClient;

    private String username;
    private String password;
    private String serverUrl;

    /**
     * Connect
     *
     * @param username
     *            A username
     * @param password
     *            A password
     * @param serverUrl
     *            Url of a server
     * @throws ConnectionException
     */
    @Connect
    @TestConnectivity
    public void connect(@ConnectionKey String username, @Password String password, String serverUrl)
            throws ConnectionException {

        this.username = username;
        this.password = password;
        this.serverUrl = serverUrl;

        if (serverUrl != null) {
            final JiraRestClientFactory factory = new AsynchronousJiraRestClientFactory();
            try {
                final URI jiraServerUri = new URI(serverUrl);
                this.jiraRestClient = factory.createWithBasicHttpAuthentication(jiraServerUri, username, password);
                this.jiraClient = new JiraClient(this);

                final Map<String, Object> response = myself();
                if (response.get("key") != null) {
                    return;
                } else {
                    throw new JiraClientAuthenticationException();
                }

            } catch (JiraClientAuthenticationException e) {
                log.error(e.getMessage(), e);
                throw new ConnectionException(ConnectionExceptionCode.INCORRECT_CREDENTIALS, "", "Invalid credentials");
            } catch (ProcessingException e) {
                log.error(e.getMessage(), e);
                throw new ConnectionException(ConnectionExceptionCode.INCORRECT_CREDENTIALS, "",
                        "Invalid server url provided");
            } catch (URISyntaxException e) {
                log.error(e.getMessage(), e);
                throw new ConnectionException(ConnectionExceptionCode.INCORRECT_CREDENTIALS, "",
                        "Invalid server url provided");
            }
        }
    }

    private Map<String, Object> myself() {
        UriBuilder builder = UriBuilder.fromPath("/rest/api/2/myself/");
        final String url = builder.build().toString();
        return jiraClient.genericHttpRequestMap(JiraClient.HttpMethod.GET, url, null);
    }

    /**
     * Disconnect
     */
    @Disconnect
    public void disconnect() throws RuntimeException {
        this.username = null;
        this.password = null;
        this.serverUrl = null;
        this.jiraClient = null;
        if (jiraRestClient != null) {
            try {
                jiraRestClient.close();
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
            jiraRestClient = null;
        }
    }

    /**
     * Are we connected
     */
    @ValidateConnection
    public boolean isConnected() {
        return jiraClient != null;
    }

    /**
     * Are we connected
     */
    @ConnectionIdentifier
    public String connectionId() {
        return "{username = '" + username + "', baseUrl = '" + serverUrl + "'}";
    }

    public String getUsername() {
        return username;
    }

    public void setUsername(String username) {
        this.username = username;
    }

    public String getPassword() {
        return password;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    public String getServerUrl() {
        return serverUrl;
    }

    public void setServerUrl(String serverUrl) {
        this.serverUrl = serverUrl;
    }

    public JiraClient getJiraClient() {
        return jiraClient;
    }

    public JiraRestClient getJiraRestClient() {
        return jiraRestClient;
    }
}
