/**
 * (c) 2003-2015 MuleSoft, Inc. The software in this package is published under the terms of the CPAL v1.0 license,
 * a copy of which has been included with this distribution in the LICENSE.md file.
 */

package org.mule.module.jira.api.datasense;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.Iterator;

import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.UriBuilder;

import org.apache.commons.io.IOUtils;
import org.codehaus.jettison.json.JSONArray;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;
import org.codehaus.jettison.json.JSONTokener;
import org.mule.common.metadata.MetaDataModel;
import org.mule.common.metadata.builder.DefaultMetaDataBuilder;
import org.mule.common.metadata.builder.DynamicObjectBuilder;
import org.mule.common.metadata.builder.ListMetaDataBuilder;
import org.mule.common.metadata.datatype.DataType;
import org.mule.module.jira.api.rest.JiraRestConnectorException;
import org.mule.modules.jirarest.JiraRestConnector;

public class JiraMetaDataBuilder {

    private JiraRestConnector connector;

    public JiraMetaDataBuilder(JiraRestConnector connector) {
        this.connector = connector;
    }

    public MetaDataModel build(String entityId) throws IOException, JSONException, JiraRestConnectorException {
        String jsonString = retrieveDynamicContentForEntity(entityId);
        if (jsonString == null) {
            final InputStream in = this.getClass().getClassLoader().getResourceAsStream(entityId + ".json");
            jsonString = IOUtils.toString(in);
        }

        Object json = new JSONTokener(jsonString).nextValue();
        if (json instanceof JSONObject) {
            final JSONObject jo = (JSONObject) json;
            final DynamicObjectBuilder<?> builder = new DefaultMetaDataBuilder().createDynamicObject(entityId);
            visit(jo, builder, entityId);
            return builder.build();
        } else if (json instanceof JSONArray) {
            final JSONArray ja = (JSONArray) json;
            final ListMetaDataBuilder<?> builder = new DefaultMetaDataBuilder().createList();
            visit(ja, builder, entityId);
            return builder.build();
        }
        return null;
    }

    private void visit(JSONObject jo, DynamicObjectBuilder<?> builder, String prefix) {
        @SuppressWarnings("unchecked")
        final Iterator<String> keys = jo.keys();

        while (keys.hasNext()) {
            final String key = keys.next();
            final String keyNext = prefix + " > " + key;

            final JSONArray jsonArray = jo.optJSONArray(key);
            final JSONObject jsonObject = jo.optJSONObject(key);
            final String jsonString = jo.optString(key);

            if (jsonArray != null) {
                visit(jsonArray, builder.addList(key), keyNext);
            } else if (jsonObject != null) {
                visit(jsonObject, builder.addDynamicObjectField(key), keyNext);
            } else if (jsonString != null) {
                builder.addSimpleField(key, DataType.STRING).setDescription(jsonString).setExample(jsonString)
                .setLabel(key);
            }
        }
    }

    private Object visit(JSONArray ja, ListMetaDataBuilder<?> listBuilder, String prefix) {
        if (ja.length() > 0) {
            // process only first item
            final JSONObject jsonObject = ja.optJSONObject(0);
            final JSONArray jsonArray = ja.optJSONArray(0);
            final String jsonString = ja.optString(0);

            if (jsonObject != null) {
                DynamicObjectBuilder<?> builder = listBuilder.ofDynamicObject(prefix);
                visit(jsonObject, builder, prefix);
                builder.endDynamicObject();
            } else if (jsonArray != null) {
                ListMetaDataBuilder<?> ofList = listBuilder.ofList();
                visit(jsonArray, ofList, prefix);
                ofList.endList();
            } else if (jsonString != null) {
                listBuilder.ofSimpleField(DataType.STRING);
            }
        }
        return listBuilder.endList();
    }

    private String retrieveDynamicContentForEntity(String entityId) {
        String jsonString = null;
        String objectKey = null;
        if ("Issue".equals(entityId) && connector.getIssueKey() != null) {
            objectKey = connector.getIssueKey();
            jsonString = getContent(objectKey, UriBuilder.fromUri("/rest/api/2/issue/{issueKey}").build(objectKey));
        }
        return jsonString;
    }

    private String getContent(String objectKey, URI uri) {
        final WebTarget webTarget = connector.getStrategy().getJiraClient().getTarget(uri);
        final String response = connector.getStrategy().getJiraClient().get(webTarget, String.class);
        if (response == null) {
            throw new JiraRestConnectorException(String.format(
                    "Metadata for sample object %s could not be retrieved. ", objectKey));
        }
        return response;
    }
}
