/**
 * (c) 2003-2014 MuleSoft, Inc. The software in this package is published under the terms of the CPAL v1.0 license, a
 * copy of which has been included with this distribution in the LICENSE.md file.
 */

package org.mule.modules.edi.edifact;

import com.anypoint.df.edi.lexical.EdiConstants;
import com.anypoint.df.edi.lexical.EdifactConstants;
import com.anypoint.df.edi.lexical.EdifactConstants.SyntaxIdentifier;
import com.anypoint.df.edi.lexical.EdifactConstants.SyntaxVersion;
import com.anypoint.df.edi.schema.*;
import com.anypoint.df.edi.schema.EdiSchema.EdiForm;
import com.anypoint.df.edi.schema.EdiSchema.Structure;
import com.mulesoft.api.b2b.B2BProviderException;
import com.mulesoft.api.b2b.Direction;
import com.mulesoft.api.b2b.config.Config;
import com.mulesoft.api.b2b.config.ConfigType;
import com.mulesoft.api.b2b.config.QueryObject;
import com.mulesoft.api.b2b.transmission.Transmission;
import com.mulesoft.api.b2b.transmission.TransmissionSession;
import com.mulesoft.api.b2b.transmission.TransmissionType;
import org.mule.api.MuleEvent;
import org.mule.api.annotations.Configurable;
import org.mule.api.annotations.Connector;
import org.mule.api.annotations.MetaDataScope;
import org.mule.api.annotations.Processor;
import org.mule.api.annotations.display.FriendlyName;
import org.mule.api.annotations.display.Placement;
import org.mule.api.annotations.licensing.RequiresEnterpriseLicense;
import org.mule.api.annotations.lifecycle.Start;
import org.mule.api.annotations.param.Default;
import org.mule.api.annotations.param.MetaDataStaticKey;
import org.mule.api.annotations.param.Optional;
import org.mule.api.registry.MuleRegistry;
import org.mule.api.transport.OutputHandler;
import org.mule.modules.edi.BaseEdiModule;
import org.mule.modules.edi.TapOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import scala.util.Try;

import javax.inject.Inject;
import java.io.*;
import java.nio.charset.Charset;
import java.util.*;

/**
 * The EDIFACT EDI Module lets you send and receive EDIFACT EDI messages. Received EDI messages are parsed and converted
 * to map structures, while send messages are generated from matching map structures. {@sample.config
 * ../../../doc/edi-module.xml.sample edi:config}
 *
 * @author MuleSoft, Inc.
 */
@Connector(name = "edifact-edi", schemaVersion = "1.0", friendlyName = "EDIFACT EDI")
@MetaDataScope(EdifactEdiMetaData.class)
@RequiresEnterpriseLicense(allowEval=true)
public class EdifactEdiModule extends BaseEdiModule
{
    private final static Logger logger = LoggerFactory.getLogger(EdifactEdiModule.class);

    /**
     * Character encodings supported for EDIFACT.
     */
    public enum EdifactCharacterEncoding {
        ASCII_A("ASCII Level A", EdifactConstants.LEVELA),
        ASCII_B("ASCII Level B", EdifactConstants.LEVELB),
        ISO8859_1("ISO-8859-1", EdifactConstants.LEVELC),
        ISO8859_2("ISO-8859-2", EdifactConstants.EDIFACT_CHARSETS.get("UNOD")),
        ISO8859_3("ISO-8859-3", EdifactConstants.EDIFACT_CHARSETS.get("UNOG")),
        ISO8859_4("ISO-8859-4", EdifactConstants.EDIFACT_CHARSETS.get("UNOH")),
        ISO8859_5("ISO-8859-5", EdifactConstants.EDIFACT_CHARSETS.get("UNOE")),
        ISO8859_6("ISO-8859-6", EdifactConstants.EDIFACT_CHARSETS.get("UNOI")),
        ISO8859_7("ISO-8859-7", EdifactConstants.EDIFACT_CHARSETS.get("UNOF")),
        ISO8859_8("ISO-8859-8", EdifactConstants.EDIFACT_CHARSETS.get("UNOJ")),
        ISO8859_9("ISO-8859-9", EdifactConstants.EDIFACT_CHARSETS.get("UNOK")),
        UTF8("UTF-8", EdifactConstants.LEVELY);

        public final Charset characterSet;
        public final SyntaxIdentifier syntaxIdentifier;

        private EdifactCharacterEncoding(String name, SyntaxIdentifier syntax) {
            if (name.startsWith("ASCII")) {
                characterSet = EdiConstants.ASCII_CHARSET;
            } else {
                Charset chset = null;
                try {
                    chset = Charset.forName(name);
                } catch (Throwable e) {
                    e.printStackTrace();
                }
                characterSet = chset;
            }
            syntaxIdentifier = syntax;
        }
    }

    /**
     * Delimiter set usage choices.
     */
    public enum EdifactDelimiterUsage {
        USE_DEFAULTS("Use default delimiters from standard"),
        USE_SPECIFIED_FOR_WRITES("Use specified delimiters for writes"),
        USE_SPECIFIED_FOR_ALL("Use specified delimiters for reads and writes");

        public final String text;
        private EdifactDelimiterUsage(String msg) {
            text = msg;
        }

        public String toString() {
            return text;
        }
    }

    private EdifactConfig edifactConfig;

    @Configurable
    @Optional
    @FriendlyName("Use B2B Provider")
    private Boolean useB2BProvider = false;

    /**
     * List of paths for schemas to be used by module. The paths may be either for file system or classpath.
     */
    @Configurable
    private List<String> schemas;

    /**
     * Identification for Mule in exchange (UNB segment). This value is used in combination with the self identification
     * interchange identifier qualifier value, and if one is set the other must also be set. If this value is set,
     * it effects both send and receive message handling. On the receive side, the UNB interchange recipient
     * identification must match this value. On the send side, this value is used as the UNB interchange
     * sender identification unless overridden in either the send configuration or the message parameters.
     */
    @Configurable
    @Optional
    @Placement(group = "Self identification")
    @FriendlyName("Interchange sender/receiver ID (UNB2.1/UNB3.1)")
    private String interchangeIdSelf;

    /**
     * Identification qualifier for Mule in exchange (UNB segment). This value is used in combination with the self
     * identification interchange identifier value, and if one is set the other must also be set. If this value is set,
     * it effects both send and receive message handling. On the receive side, the UNB interchange recipient
     * identification code qualifier must match this value. On the send side, this value is used as the UNB interchange
     * sender identification code qualifier unless overridden in either the send configuration or the message parameters.
     */
    @Configurable
    @Optional
    @Placement(group = "Self identification")
    @FriendlyName("Interchange sender/receiver ID qualifier (UNB2.2/UNB3.2)")
    private String interchangeIdQualifierSelf;

    /**
     * Application identification for Mule in group (UNG segment). This value is used in combination with the self
     * identification application identifier qualifier value, and if one is set the other must also be set. If this value
     * is set, it effects both send and receive message handling. On the receive side, the UNG application recipient
     * identification must match this value. On the send side, this value is used as the UNG application
     * sender identification unless overridden in either the send configuration or the message parameters.
     */
    @Configurable
    @Optional
    @Placement(group = "Self identification")
    @FriendlyName("Application sender/receiver ID (UNG2.1/UNG3.1)")
    private String groupIdSelf;

    /**
     * Application identification qualifier for Mule in group (UNG segment). This value is used in combination with the
     * self identification application identifier value, and if one is set the other must also be set. If this value is set,
     * it effects both send and receive message handling. On the receive side, the UNG application recipient
     * identification code qualifier must match this value. On the send side, this value is used as the UNG application
     * sender identification code qualifier unless overridden in either the send configuration or the message parameters.
     */
    @Configurable
    @Optional
    @Placement(group = "Self identification")
    @FriendlyName("Application sender/receiver ID qualifier (UNG2.2/UNG3.2)")
    private String groupIdQualifierSelf;

    /**
     * Identification for partner in exchange (UNB segment). This value is used in combination with the partner
     * identification interchange identifier qualifier value, and if one is set the other must also be set. If this
     * value is set, it effects both send and receive message handling. On the receive side, the UNB interchange sender
     * identification must match this value. On the send side, this value is used as the UNB interchange
     * recipient identification unless overridden in either the send configuration or the message parameters.
     */
    @Configurable
    @Optional
    @Placement(group = "Partner identification")
    @FriendlyName("Interchange sender/receiver ID (UNB2.1/UNB3.1)")
    private String interchangeIdPartner;

    /**
     * Identification qualifier for partner in exchange (UNB segment). This value is used in combination with the partner
     * identification interchange identifier value, and if one is set the other must also be set. If this value is set,
     * it effects both send and receive message handling. On the receive side, the UNB interchange sender
     * identification code qualifier must match this value. On the send side, this value is used as the UNB interchange
     * recipient identification code qualifier unless overridden in either the send configuration or the message parameters.
     */
    @Configurable
    @Optional
    @Placement(group = "Partner identification")
    @FriendlyName("Interchange sender/receiver ID qualifier (UNB2.2/UNB3.2)")
    private String interchangeIdQualifierPartner;

    /**
     * Application identification for partner in group (UNG segment). This value is used in combination with the partner
     * identification application identifier qualifier value, and if one is set the other must also be set. If this value
     * is set, it effects both send and receive message handling. On the receive side, the UNG application sender
     * identification must match this value. On the send side, this value is used as the UNG application
     * recipient identification unless overridden in either the send configuration or the message parameters.
     */
    @Configurable
    @Optional
    @Placement(group = "Partner identification")
    @FriendlyName("Application sender/receiver ID (UNG2.1/UNG3.1)")
    private String groupIdPartner;

    /**
     * Application identification qualifier for partner in group (UNG segment). This value is used in combination with the
     * partner identification application identifier value, and if one is set the other must also be set. If this value is
     * set, it effects both send and receive message handling. On the receive side, the UNG application sender
     * identification code qualifier must match this value. On the send side, this value is used as the UNG application
     * recipient identification code qualifier unless overridden in either the send configuration or the message parameters.
     */
    @Configurable
    @Optional
    @Placement(group = "Partner identification")
    @FriendlyName("Application sender/receiver ID qualifier (UNG2.2/UNG3.2)")
    private String groupIdQualifierPartner;

    /**
     * Fail when string data contains invalid characters flag (only applies when using syntax levels UNOA and UNOB). If
     * true, a received or sent message containing characters not allowed by the syntax level is rejected; if false, the
     * character is either passed through or substituted (depending on whether a substitution character has been set)
     * and the message is not rejected.
     */
    @Configurable
    @Default("false")
    @FriendlyName("Fail document when invalid character in value")
    private boolean invalidCharacterInValueFail;

    /**
     * Substitution character used to replace invalid characters in string values when using syntax level UNOA or UNOB.
     */
    @Configurable
    @Optional
    @FriendlyName("String substitution character")
    private Character stringSubstitutionChar;

    /**
     * Usage of specified delimiters:
     * <ul>
     * <li>USE_DEFAULTS - use defaults as defined by EDIFACT syntax version standards</li>
     * <li>USE_SPECIFIED_FOR_WRITES - use specified delimiters for writes, use standard defaults for reads</li>
     * <li>USE_SPECIFIED_FOR_ALL - use specified delimiters for writes and as defaults for reads</li>
     * </ul>
     */
    @Configurable
    @Default("USE_SPECIFIED_FOR_ALL")
    @Placement(group = "Delimiter options")
    @FriendlyName("Specified delimiters usage")
    private EdifactDelimiterUsage delimiterUsage;

    /**
     * Default data element separator character. The configured value is used by default for all output messages, but
     * can be overridden at the message level.
     */
    @Configurable
    @Default("+")
    @Placement(group = "Delimiter options")
    @FriendlyName("Data element separator character")
    private Character dataSeparator;

    /**
     * Default component element separator character. The configured value is used by default for all output messages,
     * but can be overridden at the message level.
     */
    @Configurable
    @Default(":")
    @Placement(group = "Delimiter options")
    @FriendlyName("Component element separator character")
    private Character componentSeparator;

    /**
     * Default repetition separator character. The configured value is used by default for all output
     * messages, but can be overridden at the message level.
     */
    @Configurable
    @Default("*")
    @Placement(group = "Delimiter options")
    @FriendlyName("Repetition separator character")
    private Character repetitionSeparator;

    /**
     * Default segment terminator character. The configured value is used by default for all output messages, but can be
     * overridden at the message level.
     */
    @Configurable
    @Default("'")
    @Placement(group = "Delimiter options")
    @FriendlyName("Segment terminator character")
    private Character segmentTerminator;

    /**
     * Default release (control character escape) character. The configured value is used by default for all output
     * messages, but can be overridden at the message level.
     */
    @Configurable
    @Default("?")
    @Placement(group = "Delimiter options")
    @FriendlyName("Release character")
    private Character releaseCharacter;

    /**
     * Fail when receive value length outside allowed range flag. If true, a message with this error is rejected; if
     * false, the value is used anyway and the message is not rejected. In either case the error is logged and
     * reported in CONTRL functional acknowledgments.
     */
    @Configurable
    @Placement(group = "Parser options")
    @FriendlyName("Fail document when value length outside allowed range")
    @Default("true")
    private boolean valueLengthErrorFail;

    /**
     * Fail when receive value is repeated too many times. If true, a message with this error is rejected; if false,
     * the value is accepted and the message is not rejected. In either case the error is reported in CONTRL functional
     * acknowledgments.
     */
    @Configurable
    @Default("true")
    @Placement(group = "Parser options")
    @FriendlyName("Fail document if value is repeated too many times")
    private boolean wrongValuesRepeatsFail;

    /**
     * Fail when an unknown segment is present in a message. If true, a message with this error is rejected;
     * if false, the segment is ignored and the message is not rejected. In either case the error is reported in CONTRL
     * functional acknowledgments.
     */
    @Configurable
    @Default("true")
    @Placement(group = "Parser options")
    @FriendlyName("Fail document if unknown segments are used")
    private boolean unknownSegmentFail;

    /**
     * Fail when a segment is out of order in a message. If true, a message with this error is rejected; if
     * false and the segment can be reordered the message is not rejected. In either case the error is reported in
     * CONTRL functional acknowledgments.
     */
    @Configurable
    @Default("true")
    @Placement(group = "Parser options")
    @FriendlyName("Fail document when segments are out of order")
    private boolean segmentOutOfOrderFail;

    /**
     * Fail when a segment marked as Unused is included in a message. If true, a message with this error is
     * rejected; if false, the message is not rejected and the unused segment is ignored. In either case the error
     * is reported in CONTRL functional acknowledgments.
     */
    @Configurable
    @Default("true")
    @Placement(group = "Parser options")
    @FriendlyName("Fail document when unused segments are included")
    private boolean unusedSegmentPresentFail;

    /**
     * Fail when a segment occurs too many times in a message. If true, a message with this error is
     * rejected; if false, the message is not rejected. In either case the error is reported in CONTRL functional
     * acknowledgments.
     */
    @Configurable
    @Default("true")
    @Placement(group = "Parser options")
    @FriendlyName("Fail document when too many repeats of segment")
    private boolean wrongSegmentsRepeatsFail;

    /**
     * Reject an interchange if the Interchange Control Reference has previously been processed. The normal behavior is to
     * record the interchange references previously processed and reject any duplicate interchange references from the same
     * partner (as determined by the interchange sender and recipient identification). If false, this instead allows
     * processing of the received interchange to continue and leaves it to the application flow to track references and
     * handle appropriately.
     */
    @Configurable
    @Default("false")
    @Placement(group = "Parser options")
    @FriendlyName("Require unique Interchange Control References")
    private boolean requireUniqueInterchangeReferences;

    /**
     * Reject a functional group if the Group Reference Number has previously been processed. By default, group numbers
     * only need to be unique with an interchange. If true, this instead requires group numbers to be unique across all
     * interchanges received from the same partner and application (as determined by the interchange sender and recipient
     * identification, combined with the group sender and recipient application identification).
     */
    @Configurable
    @Default("false")
    @Placement(group = "Parser options")
    @FriendlyName("Require unique Group Reference Numbers")
    private boolean requireUniqueGroupNumbers;

    /**
     * Reject a message if the Message Reference Number has previously been processed. By default,
     * message reference numbers only need to be unique with a particular interchange or group. If true, this instead
     * requires message reference numbers to be unique across all interchanges and groups received from the same partner and
     * application (as determined by the interchange sender and recipient identification, combined with the
     * group sender and recipient application identification when groups are used).
     */
    @Configurable
    @Default("false")
    @Placement(group = "Parser options")
    @FriendlyName("Require unique Message Reference Numbers")
    private boolean requireUniqueMessageNumbers;

    /**
     * Character encoding for messages. This character encoding is always used for receive messages. The encoding used
     * for send messages defaults to this, but may be overridden by the send operation configuration parameters.
     */
    @Configurable
    @Placement(group = "Writer options")
    @Default("ASCII_B")
    @FriendlyName("Message character encoding")
    private EdifactCharacterEncoding characterEncoding;

    /**
     * Syntax version for send messages, also used to determine the data structures used for control segments and CONTRL
     * acknowledgment message structures in DataSense metadata for both send and receive. The actual message data for
     * receive messages will reflect the syntax version used in the received interchange.
     */
    @Configurable
    @Placement(group = "Writer options")
    @Default("VERSION4")
    @FriendlyName("Syntax version (UNB1.2)")
    private SyntaxVersion sendSyntaxVersion;

    /**
     * Line ending to add between segments. This allows you to add line endings between segments to improve the
     * readability of the output message text.
     */
    @Configurable
    @Default("NONE")
    @Placement(group = "Writer options")
    @FriendlyName("Line ending between segments")
    private SegmentWhitespace lineEnding;

    /**
     * Always send UNA flag. With the default setting of false a UNA will only be included in an interchange when the
     * syntax separator characters don't match the defaults for the syntax version and character encoding in use, if set
     * true a UNA will always be sent.
     */
    @Configurable
    @Default("false")
    @Placement(group = "Writer options")
    @FriendlyName("Always send UNA")
    private boolean alwaysSendUNA;

    /**
     * Send messages in groups. By default, messages are sent as the direct content of interchanges. If this is set,
     * groups are instead used for messages of the same types.
     */
    @Configurable
    @Default("false")
    @Placement(group = "Writer options")
    @FriendlyName("Use groups")
    private boolean sendGroups;

    /**
     * Send unique Group Reference Numbers. By default, group reference numbers are assigned sequentially within
     * each interchange and are reused in different interchanges. If true, this instead assigns unique group reference
     * numbers across all interchanges sent to the same partner and application (as determined by the interchange sender
     * and recipient identification, combined with the group sender and recipient application identification).
     */
    @Configurable
    @Default("false")
    @Placement(group = "Writer options")
    @FriendlyName("Use unique Group Reference Numbers")
    private boolean sendUniqueGroupNumbers;

    /**
     * Send unique Message Reference Numbers. By default, message reference numbers are assigned sequentially within each
     * interchange and are reused in different interchanges. If true, this instead assigns
     * unique message reference numbers across all interchanges sent to the same partner (as determined by
     * the interchange sender and recipient identification).
     */
    @Configurable
    @Default("false")
    @Placement(group = "Writer options")
    @FriendlyName("Use unique Message Reference Numbers")
    private boolean sendUniqueMessageNumbers;

    /**
     * The initial Interchange Control Reference used for outgoing messages.
     */
    @Configurable
    @Default("1")
    @Placement(group = "Writer options")
    @FriendlyName("Initial Interchange Control Reference")
    private String initialInterchangeNumber;

    /**
     * The initial Group Reference Number used for outgoing messages.
     */
    @Configurable
    @Default("1")
    @Placement(group = "Writer options")
    @FriendlyName("Initial Group Reference")
    private String initialGroupNumber;

    /**
     * The initial Message Reference Number used for outgoing messages.
     */
    @Configurable
    @Default("1")
    @Placement(group = "Writer options")
    @FriendlyName("Initial Message Reference Number")
    private String initialMessageNumber;

    /**
     * Request acknowledgments for sent messages flag. If true, CONTRL acknowledgments will be requested for all sent
     * messages. If false, only recipt confirmations will be requested.
     */
    @Configurable
    @Default("true")
    @Placement(group = "Writer options")
    @FriendlyName("Request acknowledgments (UNB9)")
    private boolean ackRequested;

    /**
     * Test indicator character to be used on send interchanges. By default, interchanges are not sent as tests.
     */
    @Configurable
    @Optional
    @Placement(group = "Writer options")
    @FriendlyName("Default test indicator (UNB11)")
    private Character defaultTestIndicator;

    @Inject
    private MuleRegistry registry;

    public void setRegistry(MuleRegistry registry) {
        this.registry = registry;
    }

    @Override
    protected MuleRegistry getRegistry() {
        return registry;
    }

    @Start
    public void onStart() throws Exception {
        loadSchemas(Collections.singletonList(EdifactSchemaDefs.contrlMsg(sendSyntaxVersion)),
            new EdiSchemaVersion((EdiForm)EdiSchema.convertEdiForm("EDIFACT"), "D00A"));

        if (!useB2BProvider) {
            edifactConfig = createLocalEdifactConfig();
        }

        doStart();
    }

    /**
     * Transforms a native EDI document into a Map
     *
     * {@sample.xml ../../../doc/edi-module.xml.sample edi:read}
     *
     * @param ediContent Native EDI content
     * @param event The Mule Event (with input information)
     * @return the Map
     */
    @Processor
    @Inject
    @MetaDataStaticKey(type = "InMessage")
    public Map<String, Object> read(@Default("#[payload]") InputStream ediContent, MuleEvent event) throws Exception {
        if (ediContent == null) {
            throw new RuntimeException("Couldn't get the file");
        }

        Map<String, Object> result;
        String delims = null;

        if (useB2BProvider) {

            loadB2BProvider();

            QueryObject configQueryObject = queryObjectFactory.create();

            // TODO: the B2B Connector throws an exception and writes an error log if no options are found.
            // But it's expected for options not to be found at this stage if the user hasn't defined the
            // interchange IDs which we use to lookup the options. This logic needs to be reviewed for correctness
            if (getInterchangeIdSelf() != null && getInterchangeIdPartner() != null) {
                configQueryObject.put("interchangeReceiverId", getInterchangeIdSelf());
                configQueryObject.put("interchangeReceiverIdQualifier", getInterchangeIdQualifierSelf());
                configQueryObject.put("applicationReceiverId", getGroupIdSelf());
                configQueryObject.put("applicationReceiverIdQualifier", getGroupIdQualifierSelf());
                configQueryObject.put("interchangeSenderId", getInterchangeIdPartner());
                configQueryObject.put("interchangeSenderIdQualifier", getInterchangeIdQualifierPartner());
                configQueryObject.put("applicationSenderId", getGroupIdPartner());
                configQueryObject.put("applicationSenderIdQualifier", getGroupIdQualifierPartner());

                Config b2bProviderConfig = getB2BProviderConfig(configQueryObject, Direction.INBOUND);

                delims = new String(new char[]{Character.valueOf(b2bProviderConfig.get("dataSeparator").toString().charAt(0)),
                        Character.valueOf(b2bProviderConfig.get("componentSeparator").toString().charAt(0)),
                        Character.valueOf(b2bProviderConfig.get("repetitionSeparator").toString().charAt(0)),
                        Character.valueOf(b2bProviderConfig.get("segmentTerminator").toString().charAt(0)),
                        Character.valueOf(b2bProviderConfig.get("releaseCharacter").toString().charAt(0))});

            }

            byte[] payload = event.getMessage().getPayloadAsBytes();

            EdifactInterchangeParser parser = new EdifactInterchangeParser(new ByteArrayInputStream(payload), delims,
                    new B2BProviderEnvelopeHandler(b2bProvider, this));

            result = parse(parser);
            TransmissionSession transmissionSession = trackInboundTransmission(event, payload, result, TransmissionType.EDIFACT);
            event.setFlowVariable(B2B_TRANSMISSION_SESSION_FLOW_VARIABLE_KEY, transmissionSession);
        } else {
            if (delimiterUsage == EdifactDelimiterUsage.USE_SPECIFIED_FOR_ALL) {
                delims = new String(new char[] { getDataSeparator(), getComponentSeparator(), getRepetitionSeparator(),
                    getSegmentTerminator(), getReleaseCharacter() });
            }
            EdifactInterchangeParser parser = new EdifactInterchangeParser(ediContent, delims,
                new DirectEnvelopeHandler(this));
            Try<Map<String, Object>> parse = parser.parse();
            result = parse.get();
        }
        return result;
    }

    /**
     * Transforms a Map into an EDI document
     *
     * {@sample.xml ../../../doc/edi-module.xml.sample edi:write}
     *
     * @param map Map containing message data
     * @param interSelfIdent UNB sender identification (overrides module configuration setting).
     * @param interSelfQual UNB sender identification code qualifier (overrides module configuration setting).
     * @param grpSelfIdent UNG application sender identification (overrides module configuration setting).
     * @param grpSelfQual UNG application sender identification code qualifier (overrides module configuration setting).
     * @param interPartnerIdent UNB recipient identification (overrides module configuration setting).
     * @param interPartnerQual UNB recipient identification code qualifier (overrides module configuration setting).
     * @param grpPartnerIdent UNG application recipient identification (overrides module configuration setting).
     * @param grpPartnerQual UNG application recipient identification code qualifier (overrides module configuration setting).
     * @param sendGroups use groups for messages (overrides module configuration setting).
     * @param event The Mule Event
     * @return Output handler for writing data
     */
    @Processor
    @Inject
    public OutputHandler write(
        final @MetaDataStaticKey(type = "OutMessage") @Default("#[payload]") Map<String, Object> map,
        @Placement(group = "Sender identification") @FriendlyName("Interchange sender identification (UNB2.1)") @Optional String interSelfIdent,
        @Placement(group = "Sender identification") @FriendlyName("Interchange sender qualifier (UNB2.2)") @Optional String interSelfQual,
        @Placement(group = "Sender identification") @FriendlyName("Group application sender identification (UNG2.1)") @Optional String grpSelfIdent,
        @Placement(group = "Sender identification") @FriendlyName("Group application sender qualifier (UNG2.2)") @Optional String grpSelfQual,
        @Placement(group = "Recipient identification") @FriendlyName("Interchange recipient identification (UNB3.1)") @Optional String interPartnerIdent,
        @Placement(group = "Recipient identification") @FriendlyName("Interchange recipient qualifier (UNB3.2)") @Optional String interPartnerQual,
        @Placement(group = "Recipient identification") @FriendlyName("Group application recipient identification (UNG3.1)") @Optional String grpPartnerIdent,
        @Placement(group = "Recipient identification") @FriendlyName("Group application recipient qualifier (UNG3.2)") @Optional String grpPartnerQual,
        final MuleEvent event) throws Exception {

        final EdifactConfig writeEdifactConfig;

        final String selectedInterchangeIdSelf = nonnull(interSelfIdent, interchangeIdSelf);
        final String selectedInterchangeIdQualifierSelf = nonnull(interSelfQual, interchangeIdQualifierSelf);
        final String selectedGroupIdSelf = nonnull(grpSelfIdent, groupIdSelf);
        final String selectedGroupIdQualifierSelf= nonnull(grpSelfQual, groupIdQualifierSelf);
        final String selectedInterchangeIdPartner = nonnull(interPartnerIdent, interchangeIdPartner);
        final String selectedInterchangeIdQualifierPartner = nonnull(interPartnerQual, interchangeIdQualifierPartner);
        final String selectedGroupIdPartner = nonnull(grpPartnerIdent, groupIdPartner);
        final String selectedGroupIdQualifierPartner = nonnull(grpPartnerQual, groupIdQualifierPartner);

        if (useB2BProvider) {
            loadB2BProvider();
            String messageType = getMessageType(map);
            QueryObject queryObject = queryObjectFactory.create();
            queryObject.put("interchangeSenderId", selectedInterchangeIdSelf);
            queryObject.put("interchangeSenderIdQualifier", selectedInterchangeIdQualifierSelf);
            queryObject.put("applicationSenderId", selectedGroupIdSelf);
            queryObject.put("applicationSenderIdQualifier", selectedGroupIdQualifierSelf);
            queryObject.put("interchangeReceiverId", selectedInterchangeIdPartner);
            queryObject.put("interchangeReceiverIdQualifier", selectedInterchangeIdQualifierPartner);
            queryObject.put("applicationReceiverId", selectedGroupIdPartner);
            queryObject.put("applicationReceiverIdQualifier", selectedGroupIdQualifierPartner);
            queryObject.put("version", getCombinedSchema().ediVersion().version().toUpperCase());
            if (messageType != null) {
                queryObject.put("messageType", messageType);
            }

            Config b2bProviderConfig = getB2BProviderConfig(queryObject, Direction.OUTBOUND);

            if (b2bProviderConfig != null) {
                writeEdifactConfig = mapB2BProviderWriteConfigToEdifactConfig(b2bProviderConfig);
            }
            else {
                throw new Exception("Queried registered B2B Provider for EDIFACT write config but none was found. Check your configuration. Lookup key used was: " + queryObject.toString());
            }

        } else {
            writeEdifactConfig = edifactConfig;
        }

        String delims = (String)map.get(SchemaJavaValues.delimiterCharacters());
        if (delims == null) {
            if (delimiterUsage != EdifactDelimiterUsage.USE_DEFAULTS) {
                delims = new String(new char[]
                    { writeEdifactConfig.getDataSeparator(), writeEdifactConfig.getComponentSeparator(), writeEdifactConfig.getRepetitionSeparator(), writeEdifactConfig.getSegmentTerminator(), writeEdifactConfig.getReleaseCharacter() });
            }
        } else if (delims.length() != 5) {
            throw new IllegalArgumentException(SchemaJavaValues.delimiterCharacters() + " value must be 5 characters: '" + delims + "'");
        }
        if (delims != null) {
            Set<Character> chars = new HashSet<Character>();
            for (int i = 0; i < 5; i++) {
                char delim = delims.charAt(i);
                if (Character.isAlphabetic(delim) || Character.isDigit(delim)) {
                    if (i != 2 || delim != 0) {
                        throw new IllegalArgumentException("Invalid delimiter character: '" + delim + "'");
                    }
                }
                if (!chars.add(delim)) {
                    throw new IllegalArgumentException("Duplicate delimiter character: '" + delim + "'");
                }
            }
        }
        final String useDelims = delims;

        // initialize root map default values for interchange
        Map<String, Object> imap = (Map<String, Object>)map.get(SchemaJavaValues.interchangeKey());
        if (imap == null) {
            imap = new HashMap<String, Object>();
            map.put(SchemaJavaValues.interchangeKey(), imap);
        }
        if (!imap.containsKey(EdifactSchemaDefs.interHeadAckreqKey())) {
            map.put(EdifactSchemaDefs.interHeadAckreqKey(), writeEdifactConfig.isAckRequested() ? "1" : "0");
        }
        setNullDefault(EdifactSchemaDefs.interHeadSenderQualKey(), selectedInterchangeIdQualifierSelf, imap);
        setNullDefault(EdifactSchemaDefs.interHeadSenderIdentKey(), selectedInterchangeIdSelf, imap);
        setNullDefault(EdifactSchemaDefs.interHeadRecipientQualKey(), selectedInterchangeIdQualifierPartner, imap);
        setNullDefault(EdifactSchemaDefs.interHeadRecipientIdentKey(), selectedInterchangeIdPartner, imap);
        if (writeEdifactConfig.getDefaultTestIndicator() != null) {
            setNullDefault(EdifactSchemaDefs.interHeadTestKey(),
                new String(new char[] { writeEdifactConfig.getDefaultTestIndicator().charValue() }), imap);
        }
        return new OutputHandler() {

            final Transmission transmission = useB2BProvider ? transmissionFactory.create() : null;

            @Override
            public void write(MuleEvent onWriteEvent, OutputStream output) throws IOException {
                ByteArrayOutputStream tapStream = new ByteArrayOutputStream();
                int subChar = writeEdifactConfig.getStringSubstitutionChar() == null ? -1 : writeEdifactConfig.getStringSubstitutionChar().charValue();
                EdifactWriterConfig config = new EdifactWriterConfig(writeEdifactConfig.getCharacterEncoding().syntaxIdentifier,
                    writeEdifactConfig.getSendSyntaxVersion(), writeEdifactConfig.isInvalidCharacterInValueFail(), true, subChar, '.',
                    writeEdifactConfig.getCharacterEncoding().characterSet, useDelims, writeEdifactConfig.getLineEnding().whitespace,
                    writeEdifactConfig.isAlwaysSendUNA());

                // set structure schemas for each transaction set, for each version
                Map<String, Object> verMap = (Map<String, Object>)map.get(EdifactSchemaDefs.messagesMap());
                for (String vkey : verMap.keySet()) {
                    Map<String, Object> msgMap = (Map<String, Object>)verMap.get(vkey);
                    for (String tkey : msgMap.keySet()) {
                        List<Map<String, Object>> msgs = (List<Map<String, Object>>)msgMap.get(tkey);
                        if (!msgs.isEmpty()) {
                            Structure structure = getStructureSchema(getSendSyntaxVersion().code(), vkey, tkey);
                            for (Map<String, Object> set : msgs) {
                                set.put(SchemaJavaValues.structureSchema(), structure);
                            }
                        }
                    }
                }

                EdifactSchemaWriter writer = new EdifactSchemaWriter(useB2BProvider ? new TapOutputStream(output, tapStream) : output, new EdifactNumberProvider() {
                    String groupNum;
                    String messageNum;

                    @Override
                    public String contextToken(String senderId, String senderQual, String receiverId,
                        String receiverQual) {
                        return senderId + senderQual + receiverId + receiverQual;
                    }

                    @Override
                    public String nextInterchange(String context) {
                        groupNum = writeEdifactConfig.getInitialGroupNumber();
                        messageNum = writeEdifactConfig.getInitialMessageNumber();
                        String interchangeControlReference = getNextString(context, writeEdifactConfig.getInitialInterchangeNumber(), 14);
                        if (transmission != null) {
                            transmission.put("interchangeControlReference", interchangeControlReference);
                        }
                        return interchangeControlReference;
                    }

                    @Override
                    public String nextGroup(String context, String senderId, String senderQual, String receiverId,
                        String receiverQual) {
                        messageNum = writeEdifactConfig.getInitialMessageNumber();
                        if (writeEdifactConfig.isSendUniqueGroupNumbers()) {
                            return getNextString(context + "*" + senderId + "*" + senderQual + "*" + receiverId + "*" + receiverQual, writeEdifactConfig.getInitialGroupNumber(), 14);
                        }
                        groupNum = incrementString(groupNum, 18);
                        return groupNum;
                    }

                    @Override
                    public String nextMessage(String context, String msgType, String msgVersion, String msgRelease,
                        String agency) {
                        if (writeEdifactConfig.isSendUniqueMessageNumbers()) {
                            return getNextString(context, writeEdifactConfig.getInitialMessageNumber(), 14);
                        }
                        messageNum = incrementString(messageNum, 18);
                        return messageNum;
                    }
                }, config);
                writer.write(map).get();

                try {
                    if (transmission != null) {

                        transmission.put("edi", map);

                        String filename = onWriteEvent.getMessage().getInboundProperty("originalFilename");

                        if (filename != null) {
                            transmission.put("filename", filename);
                        }

                        transmission.put("content", tapStream.toByteArray());

                        transmission.put("interchangeSenderId", selectedInterchangeIdSelf);
                        transmission.put("interchangeSenderIdQualifier", selectedInterchangeIdQualifierSelf);
                        transmission.put("applicationSenderId", selectedGroupIdSelf);
                        transmission.put("applicationSenderIdQualifier", selectedGroupIdQualifierSelf);
                        transmission.put("interchangeReceiverId", selectedInterchangeIdPartner);
                        transmission.put("interchangeReceiverIdQualifier", selectedInterchangeIdQualifierPartner);
                        transmission.put("applicationReceiverId", selectedGroupIdPartner);
                        transmission.put("applicationReceiverIdQualifier", selectedGroupIdQualifierPartner);

                        TransmissionSession transmissionSession = trackTransmission(b2bProvider, onWriteEvent,
                                Direction.OUTBOUND, transmission, TransmissionType.EDIFACT);

                        onWriteEvent.setFlowVariable(B2B_TRANSMISSION_SESSION_FLOW_VARIABLE_KEY, transmissionSession);
                    }

                } catch (Exception e) {
                    logger.error(e.getMessage(), e);
                }
            }
        };
    }

    /**
     * Validate a string property value using the supplied character set.
     *
     * @param text
     * @param maxl
     * @param chflags
     * @param name
     * @return text, throws exception if invalid
     */
    protected String validateStringProperty(String text, int maxl, boolean[] chflags, String name) {
        int length = text.length();
        if (length > maxl) {
            throw new IllegalArgumentException("Value '" + text + "' too long for " + name);
        }
        for (int i = 0; i < length; i++) {
            char chr = text.charAt(i);
            if (chr > chflags.length || !chflags[chr]) {
                throw new IllegalArgumentException("Character '" + chr + "' not allowed in string value " + name);
            }
        }
        return text;
    }

    /**
     * Validate a number property. This limits values to a maximum of 14 digits, matching the limit for interchange and
     * message reference numbers.
     *
     * @param number
     * @param name
     * @return number, throws exception if invalid
     */
    protected long validateNumberProperty(long number, String name) {
        if (number > 99999999999999L) {
            throw new IllegalArgumentException("Value '" + number + "' too large for " + name);
        }
        if (number < 0) {
            throw new IllegalArgumentException("Value for " + name + " must be positive");
        }
        return number;
    }

    @Override
    public List<String> getSchemas() {
        return schemas;
    }

    public void setSchemas(List<String> schemas) {
        this.schemas = schemas;
    }

    public String getInterchangeIdQualifierSelf() {
        return interchangeIdQualifierSelf;
    }

    public void setInterchangeIdQualifierSelf(String interchangeIdQualifierSelf) {
        this.interchangeIdQualifierSelf = validateStringProperty(interchangeIdQualifierSelf, 4, EdifactConstants.levelBCharacterSet, "interchangeIdQualifierSelf");
    }

    public String getInterchangeIdSelf() {
        return interchangeIdSelf;
    }

    public void setInterchangeIdSelf(String interchangeIdSelf) {
        this.interchangeIdSelf = validateStringProperty(interchangeIdSelf, 35, EdifactConstants.levelBCharacterSet, "interchangeIdSelf");
    }

    public String getInitialInterchangeNumber() {
        return initialInterchangeNumber;
    }

    public String getGroupIdSelf() {
        return groupIdSelf;
    }

    public void setGroupIdSelf(String groupIdSelf) {
        this.groupIdSelf = validateStringProperty(groupIdSelf, 35, EdifactConstants.levelBCharacterSet, "groupIdSelf");
    }

    public String getInterchangeIdQualifierPartner() {
        return interchangeIdQualifierPartner;
    }

    public void setInterchangeIdQualifierPartner(String interchangeIdQualifierPartner) {
        this.interchangeIdQualifierPartner = validateStringProperty(interchangeIdQualifierPartner, 4, EdifactConstants.levelBCharacterSet, "interchangeIdQualifierPartner");
    }

    public String getInterchangeIdPartner() {
        return interchangeIdPartner;
    }

    public void setInterchangeIdPartner(String interchangeIdPartner) {
        this.interchangeIdPartner = validateStringProperty(interchangeIdPartner, 35, EdifactConstants.levelBCharacterSet, "interchangeIdPartner");
    }

    public String getGroupIdPartner() {
        return groupIdPartner;
    }

    public void setGroupIdPartner(String groupIdPartner) {
        this.groupIdPartner = validateStringProperty(groupIdPartner, 35, EdifactConstants.levelBCharacterSet, "groupIdPartner");
    }

    public boolean getAckRequested() {
        return ackRequested;
    }

    public void setAckRequested(boolean ackRequested) {
        this.ackRequested = ackRequested;
    }

    public Character getDefaultTestIndicator() {
        return defaultTestIndicator;
    }

    public void setDefaultTestIndicator(Character defaultTestIndicator) {
        this.defaultTestIndicator = defaultTestIndicator;
    }

    public boolean getValueLengthErrorFail() {
        return valueLengthErrorFail;
    }

    public void setValueLengthErrorFail(boolean valueLengthErrorFail) {
        this.valueLengthErrorFail = valueLengthErrorFail;
    }

    public Character getStringSubstitutionChar() {
        return stringSubstitutionChar;
    }

    public void setStringSubstitutionChar(Character stringSubstitutionChar) {
        this.stringSubstitutionChar = validateSeparator(stringSubstitutionChar, "stringSubstitutionChar");
    }

    public boolean getInvalidCharacterInValueFail() {
        return invalidCharacterInValueFail;
    }

    public void setInvalidCharacterInValueFail(boolean invalidCharacterInValueFail) {
        this.invalidCharacterInValueFail = invalidCharacterInValueFail;
    }

    public boolean getWrongValuesRepeatsFail() {
        return wrongValuesRepeatsFail;
    }

    public void setWrongValuesRepeatsFail(boolean wrongValuesRepeatsFail) {
        this.wrongValuesRepeatsFail = wrongValuesRepeatsFail;
    }

    public boolean getUnknownSegmentFail() {
        return unknownSegmentFail;
    }

    public void setUnknownSegmentFail(boolean unknownSegmentFail) {
        this.unknownSegmentFail = unknownSegmentFail;
    }

    public boolean getSegmentOutOfOrderFail() {
        return segmentOutOfOrderFail;
    }

    public void setSegmentOutOfOrderFail(boolean segmentOutOfOrderFail) {
        this.segmentOutOfOrderFail = segmentOutOfOrderFail;
    }

    public boolean getUnusedSegmentPresentFail() {
        return unusedSegmentPresentFail;
    }

    public void setUnusedSegmentPresentFail(boolean unusedSegmentPresentFail) {
        this.unusedSegmentPresentFail = unusedSegmentPresentFail;
    }

    public boolean getWrongSegmentsRepeatsFail() {
        return wrongSegmentsRepeatsFail;
    }

    public void setWrongSegmentsRepeatsFail(boolean wrongSegmentsRepeatsFail) {
        this.wrongSegmentsRepeatsFail = wrongSegmentsRepeatsFail;
    }

    public boolean isRequireUniqueInterchangeReferences() {
        return requireUniqueInterchangeReferences;
    }

    public void setRequireUniqueInterchangeReferences(boolean requireUniqueInterchangeReferences) {
        this.requireUniqueInterchangeReferences = requireUniqueInterchangeReferences;
    }

    public boolean isRequireUniqueGroupNumbers() {
        return requireUniqueGroupNumbers;
    }

    public void setRequireUniqueGroupNumbers(boolean requireUniqueGroupNumbers) {
        this.requireUniqueGroupNumbers = requireUniqueGroupNumbers;
    }

    public boolean isRequireUniqueMessageNumbers() {
        return requireUniqueMessageNumbers;
    }

    public void setRequireUniqueMessageNumbers(boolean requireUniqueMessageNumbers) {
        this.requireUniqueMessageNumbers = requireUniqueMessageNumbers;
    }

    public boolean isSendUniqueMessageNumbers() {
        return sendUniqueMessageNumbers;
    }

    public void setSendUniqueMessageNumbers(boolean sendUniqueMessageNumbers) {
        this.sendUniqueMessageNumbers = sendUniqueMessageNumbers;
    }

    public boolean isSendUniqueGroupNumbers() {
        return sendUniqueGroupNumbers;
    }

    public void setSendUniqueGroupNumbers(boolean sendUniqueGroupNumbers) {
        this.sendUniqueGroupNumbers = sendUniqueGroupNumbers;
    }

    public EdifactDelimiterUsage getDelimiterUsage() {
        return delimiterUsage;
    }

    public void setDelimiterUsage(EdifactDelimiterUsage delimiterUsage) {
        this.delimiterUsage = delimiterUsage;
    }

    public Character getDataSeparator() {
        return dataSeparator;
    }

    public void setDataSeparator(Character dataSeparator) {
        this.dataSeparator = validateSeparator(dataSeparator, "dataSeparator");
    }

    public Character getComponentSeparator() {
        return componentSeparator;
    }

    public void setComponentSeparator(Character componentSeparator) {
        this.componentSeparator = validateSeparator(componentSeparator, "componentSeparator");
    }

    public Character getRepetitionSeparator() {
        return repetitionSeparator;
    }

    public void setRepetitionSeparator(Character repetitionSeparator) {
        this.repetitionSeparator = repetitionSeparator.charValue() == 'U' ?
            repetitionSeparator : validateSeparator(repetitionSeparator, "repetitionSeparator");
    }

    public Character getSegmentTerminator() {
        return segmentTerminator;
    }

    public void setSegmentTerminator(Character segmentTerminator) {
        this.segmentTerminator = validateSeparator(segmentTerminator, "segmentTerminator");
    }

    public Character getReleaseCharacter() {
        return releaseCharacter;
    }

    public void setReleaseCharacter(Character releaseCharacter) {
        this.releaseCharacter = releaseCharacter;
    }

    public SegmentWhitespace getLineEnding() {
        return lineEnding;
    }

    public void setLineEnding(SegmentWhitespace lineEnding) {
        this.lineEnding = lineEnding;
    }

    public boolean isAlwaysSendUNA() {
        return alwaysSendUNA;
    }

    public void setAlwaysSendUNA(boolean alwaysSendUNA) {
        this.alwaysSendUNA = alwaysSendUNA;
    }

    public String getGroupIdQualifierSelf() {
        return groupIdQualifierSelf;
    }

    public void setGroupIdQualifierSelf(String groupIdQualifierSelf) {
        this.groupIdQualifierSelf = groupIdQualifierSelf;
    }

    public String getGroupIdQualifierPartner() {
        return groupIdQualifierPartner;
    }

    public void setGroupIdQualifierPartner(String groupIdQualifierPartner) {
        this.groupIdQualifierPartner = groupIdQualifierPartner;
    }

    public EdifactCharacterEncoding getCharacterEncoding() {
        return characterEncoding;
    }

    public void setCharacterEncoding(EdifactCharacterEncoding characterEncoding) {
        this.characterEncoding = characterEncoding;
    }

    public SyntaxVersion getSendSyntaxVersion() {
        return sendSyntaxVersion;
    }

    public void setSendSyntaxVersion(SyntaxVersion sendSyntaxVersion) {
        this.sendSyntaxVersion = sendSyntaxVersion;
    }

    public boolean isSendGroups() {
        return sendGroups;
    }

    public void setSendGroups(boolean sendGroups) {
        this.sendGroups = sendGroups;
    }

    public String getInitialGroupNumber() {
        return initialGroupNumber;
    }

    public void setInitialGroupNumber(String initialGroupNumber) {
        this.initialGroupNumber = initialGroupNumber;
    }

    public String getInitialMessageNumber() {
        return initialMessageNumber;
    }

    public void setInitialMessageNumber(String initialMessageNumber) {
        this.initialMessageNumber = initialMessageNumber;
    }

    public void setInitialInterchangeNumber(String initialInterchangeNumber) {
        this.initialInterchangeNumber = initialInterchangeNumber;
    }

    /**
     * Get structure schema for message. This first checks for the hardcoded acknowledgment schemas, then trys
     * the cached schemas from specified paths or classpath.
     *
     * @param syntax syntax version code
     * @param version
     * @param setId
     * @return structure
     */
    protected Structure getStructureSchema(String syntax, String version, String setId) {
        if (setId.equals("CONTRL")) {
            return "4".equals(syntax) ? EdifactSchemaDefs.transCONTRLv4() : EdifactSchemaDefs.transCONTRLv3();
        } else {
            return getSchema("EDIFACT", version, setId);
        }
    }

    private EdifactConfig createLocalEdifactConfig() throws Exception {
        EdifactConfig edifactConfig = new EdifactConfig();

        edifactConfig.setAckRequested(ackRequested);
        edifactConfig.setAlwaysSendUNA(alwaysSendUNA);
        edifactConfig.setCharacterEncoding(characterEncoding);
        edifactConfig.setComponentSeparator(componentSeparator);
        edifactConfig.setDataSeparator(dataSeparator);
        edifactConfig.setDefaultTestIndicator(defaultTestIndicator);
        edifactConfig.setInitialGroupNumber(initialGroupNumber);
        edifactConfig.setInitialInterchangeNumber(initialInterchangeNumber);
        edifactConfig.setInitialMessageNumber(initialMessageNumber);
        edifactConfig.setInvalidCharacterInValueFail(invalidCharacterInValueFail);
        edifactConfig.setLineEnding(lineEnding);
        edifactConfig.setReleaseCharacter(releaseCharacter);
        edifactConfig.setRepetitionSeparator(repetitionSeparator);
        edifactConfig.setRequireUniqueGroupNumbers(requireUniqueGroupNumbers);
        edifactConfig.setRequireUniqueInterchangeReferences(requireUniqueInterchangeReferences);
        edifactConfig.setRequireUniqueMessageNumbers(requireUniqueMessageNumbers);
        edifactConfig.setSegmentOutOfOrderFail(segmentOutOfOrderFail);
        edifactConfig.setSegmentTerminator(segmentTerminator);
        edifactConfig.setSendGroups(sendGroups);
        edifactConfig.setSendSyntaxVersion(sendSyntaxVersion);
        edifactConfig.setSendUniqueGroupNumbers(sendUniqueGroupNumbers);
        edifactConfig.setSendUniqueMessageNumbers(sendUniqueMessageNumbers);
        edifactConfig.setStringSubstitutionChar(stringSubstitutionChar);
        edifactConfig.setUnknownSegmentFail(unknownSegmentFail);
        edifactConfig.setUnusedSegmentPresentFail(unusedSegmentPresentFail);
        edifactConfig.setValueLengthErrorFail(valueLengthErrorFail);
        edifactConfig.setWrongSegmentsRepeatsFail(wrongSegmentsRepeatsFail);
        edifactConfig.setWrongValuesRepeatsFail(wrongValuesRepeatsFail);

        return edifactConfig;
    }

     protected Config getB2BProviderConfig(QueryObject configQueryObject, Direction direction) {
        try {
            return b2bProvider.getConfig(ConfigType.EDIFACT, direction, configQueryObject);
        } catch (B2BProviderException e) {
            throw new EdifactInterchangeException(null, e.getMessage(), e);
        }
    }

    private EdifactConfig mapB2BProviderWriteConfigToEdifactConfig(Config ediB2BProviderConfig) throws Exception {
        EdifactConfig edifactConfig = new EdifactConfig();

        if (ediB2BProviderConfig.get("defaultTestIndicator") != null) {
            edifactConfig.setDefaultTestIndicator(new Character(ediB2BProviderConfig.get("defaultTestIndicator").toString().charAt(0)));
        }

        if (ediB2BProviderConfig.get("stringSubstitutionChar") != null) {
            edifactConfig.setStringSubstitutionChar(ediB2BProviderConfig.get("stringSubstitutionChar").toString().charAt(0));
        }

        edifactConfig.setSendSyntaxVersion(Enum.valueOf(SyntaxVersion.class, ediB2BProviderConfig.get("sendSyntaxVersion").toString()));
        edifactConfig.setAckRequested(Boolean.valueOf(ediB2BProviderConfig.get("ackRequested").toString()));
        edifactConfig.setDataSeparator(new Character(ediB2BProviderConfig.get("dataSeparator").toString().charAt(0)));
        edifactConfig.setComponentSeparator(new Character(ediB2BProviderConfig.get("componentSeparator").toString().charAt(0)));
        edifactConfig.setInitialGroupNumber(ediB2BProviderConfig.get("initialGroupNumber").toString());
        edifactConfig.setInitialInterchangeNumber(ediB2BProviderConfig.get("initialInterchangeNumber").toString());
        edifactConfig.setInitialMessageNumber(ediB2BProviderConfig.get("initialMessageNumber").toString());
        edifactConfig.setSendGroups(Boolean.valueOf(ediB2BProviderConfig.get("sendGroups").toString()));
        edifactConfig.setSendUniqueGroupNumbers(Boolean.valueOf(ediB2BProviderConfig.get("sendUniqueGroupNumbers").toString()));
        edifactConfig.setSendUniqueMessageNumbers(Boolean.valueOf(ediB2BProviderConfig.get("sendUniqueMessageNumbers").toString()));
        edifactConfig.setReleaseCharacter(new Character(ediB2BProviderConfig.get("releaseCharacter").toString().charAt(0)));
        edifactConfig.setLineEnding(Enum.valueOf(SegmentWhitespace.class, ediB2BProviderConfig.get("lineEnding").toString()));
        edifactConfig.setSegmentTerminator(new Character(ediB2BProviderConfig.get("segmentTerminator").toString().charAt(0)));
        edifactConfig.setAlwaysSendUNA(Boolean.valueOf(ediB2BProviderConfig.get("alwaysSendUna").toString()));
        edifactConfig.setCharacterEncoding(Enum.valueOf(EdifactCharacterEncoding.class, ediB2BProviderConfig.get("characterEncoding").toString()));
        edifactConfig.setRepetitionSeparator(new Character(ediB2BProviderConfig.get("repetitionSeparator").toString().charAt(0)));
        edifactConfig.setInvalidCharacterInValueFail(Boolean.valueOf(ediB2BProviderConfig.get("invalidCharacterInValueFail").toString()));

        return edifactConfig;
    }

    public Boolean getUseB2BProvider() {
        return useB2BProvider;
    }

    public void setUseB2BProvider(Boolean useB2BProvider) {
        this.useB2BProvider = useB2BProvider;
    }

    private Map<String, Object> parse(EdifactInterchangeParser parser) {
        Try<Map<String, Object>> parse = parser.parse();
        return parse.get();
    }

    // Looking up the config by multiple distinct message types isn't currently supported
    private String getMessageType(Map edifact) {
        Map<String, Map<String, List>> messagesByVersionRelease = (Map) edifact.get(EdifactSchemaDefs.messagesMap());
        String messageType = null;

        for (Map.Entry<String, Map<String, List>> messagesByVersionReleaseEntry : messagesByVersionRelease.entrySet()) {
            for (Map.Entry<String, List> messagesByMessageType : messagesByVersionReleaseEntry.getValue().entrySet()) {
                messageType = messagesByMessageType.getKey();
            }
        }

        return messageType;
    }
}