/**
 * Mule Development Kit
 * Copyright 2010-2011 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * This file was automatically generated by the Mule Development Kit
 */
package org.mule.module.xmlsecurity;

import org.mule.api.annotations.Configurable;
import org.mule.api.annotations.Module;
import org.mule.api.annotations.Processor;
import org.mule.api.annotations.param.Default;
import org.mule.api.annotations.param.Optional;
import org.mule.api.annotations.param.Payload;
import org.mule.module.xmlsecurity.algorithms.CanonicalizationAlgorithm;
import org.mule.module.xmlsecurity.algorithms.DigestMethodAlgorithm;
import org.mule.module.xmlsecurity.algorithms.SignatureMethodAlgorithm;
import org.mule.module.xmlsecurity.keyinfo.KeyPairInfoProvider;
import org.mule.module.xmlsecurity.keyinfo.KeyInfoProvider;
import org.mule.module.xmlsecurity.keyinfo.ScretKeyInfoProvider;
import org.mule.module.xmlsecurity.keyinfo.X509KeyInfoProvider;
import org.mule.module.xmlsecurity.reference.DetachedReferenceProvider;
import org.mule.module.xmlsecurity.reference.EnvelopedReferenceProvider;
import org.mule.module.xmlsecurity.reference.EnvelopingReferenceProvider;
import org.mule.module.xmlsecurity.validator.DefaultSignatureValidator;
import org.mule.module.xmlsecurity.validator.SecretKeyValidator;
import org.mule.module.xmlsecurity.validator.selector.KeyValueKeySelector;
import org.mule.module.xmlsecurity.validator.selector.X509KeySelector;
import org.w3c.dom.Document;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.ByteArrayInputStream;
import java.io.StringWriter;

/**
 * Generic module
 *
 * @author fernando.federico@mulesource.com
 */
@Module(name="xmlsecurity", schemaVersion="1.0")
public class XmlSecurityModule
{
    /**
     * The signature type
     *
     * ACCEPTED VALUES: DETACHED, ENVELOPED, ENVELOPING
     */
    @Configurable
    private SignatureType signatureType;

    /**
     *  Private Key, if you shared a private key use it. Otherwise the document will be signed with DSA512 algorithm, and
     *  sign information will be in the signature tag.
     */
    @Configurable
    @Optional
    private String password;

    /**
     * Keystore path
     */
    @Configurable
    @Optional
    private String keystore;

    /**
     * Keystore key
     */
    @Configurable
    @Optional
    private String keystoreKey;

    /**
     * The Algorithm which is used to encrypt the private key
     * For example: HmacSHA256
     */
    @Configurable
    @Optional
    private String secretKeyAlgorithm;

    /**
     * Reference URI, if you wish to use an external reference URI for Detached signatures.
     */
    @Configurable
    @Optional
     private String referenceURI;

    /**
     * Algorithm used to generate a key Pair.
     *
     * ACCEPTED VALUES: DSA, RSA
     */
    @Configurable
    @Optional
    @Default("DSA")
    private String keyPairAlgorithm;

    /**
     * Digest Method Algorithm
     *
     * For a reference on the Digest Method Algorithms see @see{http://www.w3.org/TR/2009/WD-xmlsec-algorithms-20090226/#digest-method-uris}
     *
     * ACCEPTED VALUES:  RIPEMD160, SHA1, SHA256, SHA512
     */
    @Configurable
    @Optional
    @Default("SHA256")
    private DigestMethodAlgorithm digestMethodAlgorithm;

    /**
     * Canonicalization Method
     *
     * For a canonicalization specification see @see{http://www.w3.org/TR/2009/WD-xmlsec-algorithms-20090226/#canonicalization-uris}
     *
     * Accepted Values: EXCLUSIVE, EXCLUSIVE_WITH_COMMENTS, INCLUSIVE, INCLUSE_WITH_COMMENTS,
     */
    @Configurable
    @Optional
    @Default("EXCLUSIVE")
    private CanonicalizationAlgorithm canonicalizationAlgorithm;


    /**
     * Signature Method Algorithm
     *
     * For Signature Method Algorithms specification see @see{http://www.w3.org/TR/2009/WD-xmlsec-algorithms-20090226/#signature-method-uris}
     *
     * Accepted Values: RSA_SHA1, DSA_SHA1, HMAC_SHA1
     */
    @Configurable
    @Optional
    @Default("RSA_SHA1")
    private SignatureMethodAlgorithm signatureMethodAlgorithm;



    /**
     * Custom processor
     *
     * {@sample.xml ../../../doc/XmlSecurity-connector.xml.sample xmlsecurity:sign}
     *
     * @param payload Content to be processed
     * @return Some string
     */
    @Processor
    public String sign(@Payload Object payload)
    {
        Document doc = documentBasedOnThe((String) payload);


        if ( signatureType.equals(SignatureType.ENVELOPED))
            new Signer(keyInfoProvider(), new EnvelopedReferenceProvider(digestMethodAlgorithm)).sign(doc);
        else if ( signatureType.equals(SignatureType.ENVELOPING))
            new Signer(keyInfoProvider(), new EnvelopingReferenceProvider(digestMethodAlgorithm ) ).sign(doc);
        else
            new Signer(keyInfoProvider(), new DetachedReferenceProvider(digestMethodAlgorithm, referenceURI) ).sign(doc);

        return  createXmlUsing(doc);
    }

    /**
     * Custom processor
     *
     * {@sample.xml ../../../doc/XmlSecurity-connector.xml.sample xmlsecurity:validate}
     *
     * @param payload Content to be processed
     * @return true/false
     */
    @Processor
    public boolean validate(@Payload Object payload)
    {
        try{
            Document doc = documentBasedOnThe((String) payload);

            if ( keystore != null )
               return new DefaultSignatureValidator(new X509KeySelector(keystore, password)).validate(doc);
            if ( password != null )
                return new SecretKeyValidator(secretKeyAlgorithm, password).validate(doc);

            return new DefaultSignatureValidator(new KeyValueKeySelector()).validate(doc);
        }
        catch(Exception e)
        {
            throw new RuntimeException("validate", e);
        }

    }

    private KeyInfoProvider keyInfoProvider() {
        if ( keystore != null )
            return new X509KeyInfoProvider(canonicalizationAlgorithm, keystore, password, keystoreKey, signatureMethodAlgorithm);
        if ( password != null )
            return new ScretKeyInfoProvider(password,secretKeyAlgorithm,canonicalizationAlgorithm);

        return new KeyPairInfoProvider(canonicalizationAlgorithm, keyPairAlgorithm);
    }


    private Document documentBasedOnThe(String xml) {
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware(true);
            DocumentBuilder builder = factory.newDocumentBuilder();
            Document document = builder.parse(new ByteArrayInputStream(xml.getBytes()));

            return document;
        } catch (Exception e) {
            throw new RuntimeException("Could not create signed Document", e);
        }
    }


    private String createXmlUsing(Document doc){
        try
        {
            TransformerFactory transformerFactory = TransformerFactory.newInstance();
            Transformer trans = transformerFactory.newTransformer();
            trans.setOutputProperty(OutputKeys.INDENT, "no");

            StringWriter sw = new StringWriter();
            StreamResult result = new StreamResult(sw);
            DOMSource source = new DOMSource(doc);
            trans.transform(source, result);
            return sw.toString();
        }
        catch(Exception e)
        {
            throw new RuntimeException("Could not build signed xml", e);
        }
    }

    public void setPassword(String password) {
        this.password = password;
    }

    public void setSignatureType(SignatureType signatureType) {
        this.signatureType = signatureType;
    }

    public void setKeystore(String keystore) {
        this.keystore = keystore;
    }

    public void setSecretKeyAlgorithm(String secretKeyAlgorithm) {
        this.secretKeyAlgorithm = secretKeyAlgorithm;
    }

    public void setReferenceURI(String referenceURI) {
        this.referenceURI = referenceURI;
    }

    public void setDigestMethodAlgorithm(DigestMethodAlgorithm digestMethodAlgorithm) {
        this.digestMethodAlgorithm = digestMethodAlgorithm;
    }

    public void setCanonicalizationAlgorithm(CanonicalizationAlgorithm canonicalizationAlgorithm) {
        this.canonicalizationAlgorithm = canonicalizationAlgorithm;
    }

    public void setSignatureMethodAlgorithm(SignatureMethodAlgorithm signatureMethodAlgorithm) {
        this.signatureMethodAlgorithm = signatureMethodAlgorithm;
    }

    public void setKeystoreKey(String keystoreKey) {
        this.keystoreKey = keystoreKey;
    }

    public void setKeyPairAlgorithm(String keyPairAlgorithm) {
        this.keyPairAlgorithm = keyPairAlgorithm;
    }

}
