/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.extension.maven.generator;

import static java.util.Arrays.asList;
import static org.apache.commons.io.filefilter.DirectoryFileFilter.DIRECTORY;
import static org.mule.extension.maven.ExtensionMojoUtils.createDirectoryIfNotExist;

import org.mule.runtime.extension.api.ExtensionWalker;
import org.mule.runtime.extension.api.introspection.ExtensionModel;
import org.mule.runtime.extension.api.introspection.operation.HasOperationModels;
import org.mule.runtime.extension.api.introspection.operation.OperationModel;
import org.mule.runtime.extension.api.introspection.source.HasSourceModels;
import org.mule.runtime.extension.api.introspection.source.SourceModel;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;

import org.apache.commons.io.FileUtils;
import org.apache.maven.plugin.MojoFailureException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Generator for extension's Icons
 * <p>
 * This generator introspects the extension base dir and looks for the 'icons' folder, this folder should comply with
 * a basic structure to provide the required Icons for Studio.'
 * <p>
 * The generator verify the existence of icons for the <b>light</b> and <b>classic</b> themes, if not exist, a generic
 * one will be used.
 * <p>
 * Also, in case that a new icon theme is released, but this packager doesn't support it, any folder inside of the icons
 * folder will be copied into the plugin.
 *
 * @since 1.0
 */
public class StudioIconsGenerator
{

    private static final Logger LOGGER = LoggerFactory.getLogger(StudioIconsGenerator.class);

    private static final String ICONS_FOLDER = "/icons";

    private static final String LARGE = "large";
    private static final String SMALL = "small";

    private static final String LIGHT_THEME = "theme.light";
    private static final String CLASSIC_THEME = "theme.classic";

    private static final String CONNECTOR = "%s-connector-";
    private static final String ENDPOINT = "%s-endpoint-";

    private static final String PNG_EXTENSION = ".png";
    private static final String SUFFIX_CONNECTOR_SMALL_PNG = CONNECTOR + SMALL + PNG_EXTENSION;
    private static final String SUFFIX_CONNECTOR_LARGE_PNG = CONNECTOR + LARGE + PNG_EXTENSION;
    private static final String SUFFIX_ENDPOINT_SMALL_PNG = ENDPOINT + SMALL + PNG_EXTENSION;
    private static final String SUFFIX_ENDPOINT_LARGE_PNG = ENDPOINT + LARGE + PNG_EXTENSION;

    private static final List<String> reservedFolderNames = asList(LIGHT_THEME, CLASSIC_THEME);
    private static final String PATH_SEPARATOR = "/";
    private final ExtensionModel extensionModel;
    private final GenerationContext context;
    private final File projectBasedir;

    //TODO - VERIFY ICONS SIZES - MULE-10023
    public StudioIconsGenerator(GenerationContext context)
    {
        this.projectBasedir = context.getMavenProject().getBasedir();
        this.extensionModel = context.getExtensionModel();
        this.context = context;
    }

    public void generate() throws MojoFailureException
    {
        try
        {
            final File iconsFolder = new File(context.getPluginOutputDirectory(), ICONS_FOLDER);
            createDirectoryIfNotExist(iconsFolder);
            createDirectoryIfNotExist(new File(iconsFolder, LIGHT_THEME));
            createDirectoryIfNotExist(new File(iconsFolder, CLASSIC_THEME));

            final AtomicReference<Boolean> hasOperations = new AtomicReference<>(false);
            final AtomicReference<Boolean> hasSources = new AtomicReference<>(false);

            new ExtensionWalker()
            {
                @Override
                public void onOperation(HasOperationModels owner, OperationModel model)
                {
                    hasOperations.set(true);
                }

                @Override
                public void onSource(HasSourceModels owner, SourceModel model)
                {
                    hasSources.set(true);
                }
            }.walk(extensionModel);

            if (hasOperations.get())
            {
                createOperationIcon(iconsFolder);
            }

            if (hasSources.get())
            {
                createSourceIcon(iconsFolder);
            }

            copyThemesFolders(iconsFolder);
        }
        catch (Exception e)
        {
            throw new MojoFailureException("An error occurred generating extensions Icons", e);
        }
    }

    /**
     * Creates icons for sources
     *
     * @param iconsFolder the output icons folder
     * @throws IOException
     */
    private void createSourceIcon(File iconsFolder) throws IOException
    {
        copyIcon(CLASSIC_THEME + PATH_SEPARATOR + SUFFIX_ENDPOINT_LARGE_PNG, iconsFolder);
        copyIcon(CLASSIC_THEME + PATH_SEPARATOR + SUFFIX_ENDPOINT_SMALL_PNG, iconsFolder);

        copyIcon(LIGHT_THEME + PATH_SEPARATOR + SUFFIX_ENDPOINT_LARGE_PNG, iconsFolder);
        copyIcon(LIGHT_THEME + PATH_SEPARATOR + SUFFIX_ENDPOINT_SMALL_PNG, iconsFolder);
    }

    /**
     * Creates icons for operations
     *
     * @param iconsFolder the output icons folder
     * @throws IOException
     */
    private void createOperationIcon(File iconsFolder) throws IOException
    {
        copyIcon(CLASSIC_THEME + PATH_SEPARATOR + SUFFIX_CONNECTOR_SMALL_PNG, iconsFolder);
        copyIcon(CLASSIC_THEME + PATH_SEPARATOR + SUFFIX_CONNECTOR_LARGE_PNG, iconsFolder);

        copyIcon(LIGHT_THEME + PATH_SEPARATOR + SUFFIX_CONNECTOR_SMALL_PNG, iconsFolder);
        copyIcon(LIGHT_THEME + PATH_SEPARATOR + SUFFIX_CONNECTOR_LARGE_PNG, iconsFolder);
    }

    private void copyIcon(String baseIconName, File iconsFolder) throws IOException
    {
        final String extensionIconName = String.format(baseIconName, context.getExtensionPluginName());
        final File extensionIcon = new File(projectBasedir, "/icons/" + extensionIconName);
        final File destFile = new File(iconsFolder.getPath() + PATH_SEPARATOR + extensionIconName);

        if (extensionIcon.exists())
        {
            Files.copy(extensionIcon.toPath(), destFile.toPath());
        }
        else
        {
            LOGGER.warn(String.format("The icon [%s] was not found in the 'icons' folder. A generic one will be used instead.", extensionIconName));
            final String genericIcon = String.format(baseIconName, "generic");
            final InputStream resource = StudioIconsGenerator.class.getClassLoader().getResourceAsStream(genericIcon);
            Files.copy(resource, destFile.toPath());
        }
    }

    /**
     * Copies the rest of the theme folders found in the '/icons/' folder, this gives the capability to add support
     * of unknown themes
     *
     * @param destIconsFolder folder of the extension's plugin
     * @throws IOException
     */
    private void copyThemesFolders(File destIconsFolder) throws IOException
    {
        final File extensionIconsFolder = new File(projectBasedir, ICONS_FOLDER);
        if (extensionIconsFolder.exists())
        {
            final File[] themeFolders = extensionIconsFolder.listFiles((FileFilter) DIRECTORY);

            for (final File themeFolder : themeFolders)
            {
                if (!isReservedFolderName(themeFolder))
                {
                    FileUtils.copyDirectory(themeFolder, new File(destIconsFolder, themeFolder.getName()));
                }
            }
        }
    }

    private boolean isReservedFolderName(File themeFolder)
    {
        return reservedFolderNames.contains(themeFolder.getName());
    }
}