/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */


package org.mule.extension.maven.generator;

import org.mule.runtime.extension.api.introspection.ExtensionModel;
import org.mule.runtime.extension.api.manifest.ExtensionManifest;

import java.io.File;
import java.util.Optional;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * Generator for 'feature.xml' file
 * <p>
 * This file describes the plugin to be packaged inside the UpdateSite
 *
 * @since 1.0
 */
public class StudioFeatureGenerator
{

    private static final String FEATURE_XML_FILENAME = "feature.xml";
    private final MavenProject project;
    private final ExtensionManifest manifest;
    private final ExtensionModel extensionModel;
    private final GenerationContext context;

    public StudioFeatureGenerator(GenerationContext context)
    {
        this.project = context.getMavenProject();
        this.manifest = context.getExtensionManifest();
        this.extensionModel = context.getExtensionModel();
        this.context = context;
    }

    public void generate() throws MojoFailureException
    {
        try
        {
            final Document document = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();

            final Element feature = document.createElement("feature");
            feature.setAttribute("id", context.getFeatureId());
            feature.setAttribute("label", getLabel(manifest));
            feature.setAttribute("version", context.getBundleVersion());
            feature.setAttribute("provider-name", extensionModel.getVendor());
            document.appendChild(feature);

            final Element licenseElement = document.createElement("license");
            populateLicense(licenseElement);
            feature.appendChild(licenseElement);

            final Element plugin = document.createElement("plugin");
            plugin.setAttribute("id", context.getSymbolicName());
            plugin.setAttribute("download-size", "0");
            plugin.setAttribute("install-size", "0");
            plugin.setAttribute("version", context.getBundleVersion());
            plugin.setAttribute("unpack", "true");
            feature.appendChild(plugin);

            final DOMSource source = new DOMSource(document);
            final StreamResult result = new StreamResult(new File(context.getOutputDirectory(), FEATURE_XML_FILENAME));
            final Transformer transformer = TransformerFactory.newInstance().newTransformer();

            transformer.transform(source, result);
        }
        catch (Exception e)
        {
            throw new MojoFailureException("Error generating Mule Studio 'feature.xml' file", e);
        }
    }

    /**
     * Populates the 'feature.xml' with the project License.
     *
     * @param licenseElement {@link Element} where the license will be placed
     */
    private void populateLicense(Element licenseElement)
    {
        final MavenLicense mavenLicense = new MavenLicenseBuilder(project.getBasedir(), project.getLicenses()).build();

        final Optional<String> url = mavenLicense.getUrl();
        if (url.isPresent())
        {
            licenseElement.setAttribute("url", url.get());
        }

        final Optional<String> optionalLicense = mavenLicense.getLicenseBody();
        if (optionalLicense.isPresent())
        {
            final String licenseBody = optionalLicense.get();
            licenseElement.setTextContent(licenseBody);
            licenseElement.setNodeValue(licenseBody);
        }
    }

    private String getLabel(ExtensionManifest manifest)
    {
        return String.format("%s Extension (Mule %s+)", manifest.getName(), manifest.getMinMuleVersion().toCompleteNumericVersion());
    }
}
