package org.mule.extension.maven.generator;

import static java.io.File.separator;

import org.mule.runtime.extension.api.introspection.property.StudioModelProperty;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.Optional;

import org.apache.maven.plugin.MojoFailureException;

/**
 * Generator for provided Studio editor file
 * <p>
 * The 'editor' file is used when a custom UI is needed to be provided. This file is looked from the 'editor' folder
 * inside the project base dir.
 *
 * @since 1.0
 */
public class StudioEditorsGenerator
{

    private final GenerationContext context;
    private final File projectBasedir;

    public StudioEditorsGenerator(GenerationContext context)
    {
        this.context = context;
        this.projectBasedir = context.getMavenProject().getBasedir();
    }

    public void generate() throws MojoFailureException
    {
        final Optional<StudioModelProperty> modelProperty = context.getExtensionModel().getModelProperty(StudioModelProperty.class);

        if (modelProperty.isPresent() && !modelProperty.get().isDerived())
        {
            final StudioModelProperty studioProperty = modelProperty.get();
            final String editorFileName = studioProperty.getEditorFileName();
            final File editorFile = new File(projectBasedir, "editor" + separator + editorFileName);

            if (!editorFile.exists())
            {
                throw new MojoFailureException(String.format("Error generating editors file. The extension is marked to provide an" +
                                                             "editor file at [%s], but it does not exist.", editorFile));
            }

            final File outputFile = new File(context.getPluginOutputDirectory(), editorFileName);
            try
            {
                Files.copy(editorFile.toPath(), outputFile.toPath());
            }
            catch (IOException e)
            {
                throw new MojoFailureException(String.format("Error generating editors file. An Error occurred copying " +
                                                             "the file from [%s] to [%s]", editorFile, outputFile), e);
            }
        }
    }

}
