package org.mule.extension.maven.generator;

import static org.mule.runtime.extension.api.Category.COMMUNITY;

import java.io.File;

import org.apache.maven.plugin.MojoFailureException;
import org.codehaus.plexus.archiver.Archiver;
import org.codehaus.plexus.archiver.jar.JarArchiver;

/**
 * Generator for extension Update Site plugin JAR.
 * <p>
 * This JAR will contain:
 * <ul>
 * <li>Extension plugin ZIP</li>
 * <li>If exist, the sources and javadoc JARs</li>
 * <li>The content of the extension plugin output folder. "{target-folder}/classes/{extension-name}"</li>
 * </ul>
 *
 * @since 1.0
 */
public class PluginJarGenerator
{

    private final GenerationContext context;

    public PluginJarGenerator(GenerationContext context)
    {
        this.context = context;
    }

    public File generate() throws MojoFailureException
    {
        try
        {
            JarArchiver archiver = new JarArchiver();
            final File classesDirectory = context.getClassesDirectory();
            final File outputDirectory =  context.getOutputDirectory();
            final String extensionZipName = context.getArtifactName() + ".zip";
            final String sourcesJarName = context.getArtifactName() + "-sources.jar";
            final String javadocJarName = context.getArtifactName() + "-javadoc.jar";

            archiver.addFile(new File(outputDirectory, extensionZipName), extensionZipName);
            if (context.getExtensionCategory().equals(COMMUNITY))
            {
                addIfExist(archiver, sourcesJarName, outputDirectory);
            }
            addIfExist(archiver, javadocJarName, outputDirectory);

            archiver.setManifest(new File(classesDirectory, context.getExtensionPluginName() + "/META-INF/MANIFEST.MF"));
            archiver.addDirectory(new File(classesDirectory, context.getExtensionPluginName()));
            final File pluginJar = new File(outputDirectory, "update-site/plugins/" + context.getPluginJarName());
            archiver.setDestFile(pluginJar);

            archiver.createArchive();
            return pluginJar;
        }
        catch (Exception e)
        {
            throw new MojoFailureException("An error occurred generating the extension plugin JAR", e);
        }
    }

    private void addIfExist(Archiver archiver, String sourcesJarName, File outputDirectory)
    {
        final File sourcesJar = new File(outputDirectory, sourcesJarName);
        if (sourcesJar.exists())
        {
            archiver.addFile(sourcesJar, sourcesJarName);
        }
    }
}
