/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.extension.maven.generator;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.maven.model.License;
import org.apache.maven.project.MavenProject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class given a project base directory and the {@link License}s from the {@link MavenProject} will do the best
 * effort to obtain the {@link License} body, being from the project directory or downloading it from the given
 * {@link License#getUrl()}
 *
 * @since 1.0
 */
class MavenLicenseBuilder
{

    private static final Logger LOGGER = LoggerFactory.getLogger(MavenLicenseBuilder.class);

    private static final List<String> POSSIBLE_LICENSE_FILE_NAMES = Arrays.asList("LICENSE.md", "license.txt", "License.md");
    private final List<File> possibleLicenseFiles = new ArrayList<>();
    private final List<License> pomFoundLicenses;

    MavenLicenseBuilder(File basedir, List<License> licenses)
    {
        this.pomFoundLicenses = licenses;
        this.possibleLicenseFiles.addAll(POSSIBLE_LICENSE_FILE_NAMES.stream()
                                                 .map(licenseFileName -> new File(basedir, licenseFileName))
                                                 .collect(Collectors.toList()));
    }

    /**
     * @return a {@link MavenLicense} with the information of the license.
     */
    MavenLicense build()
    {
        Optional<String> licenseBody = getLicenseFromProject();
        if (!licenseBody.isPresent())
        {
            licenseBody = getLicenseFromPom();
        }

        return new MavenLicense(licenseBody.orElse(null), getLicenseURLFromPom().orElse(null));
    }

    /**
     * Given the {@link MavenLicenseBuilder#possibleLicenseFiles} checks if exist at least one and returns the body of
     * this one
     *
     * @return an {@link Optional} with the body value if the license exist, {@link Optional#empty()} otherwise
     */
    private Optional<String> getLicenseFromProject()
    {
        final Optional<File> optionalLicenseFile = possibleLicenseFiles.stream().filter(File::exists).findFirst();
        Optional<String> optionalLicenseBody = Optional.empty();

        if (optionalLicenseFile.isPresent())
        {
            File licenseFile = optionalLicenseFile.get();
            try
            {
                optionalLicenseBody = Optional.of(readContentFromFile(licenseFile));
            }
            catch (IOException e)
            {
                LOGGER.warn(String.format("License could not be read from %s: ", licenseFile) + e.getLocalizedMessage());
            }
        }
        else
        {
            LOGGER.warn(String.format("No license could be found in the project with the following names: [%s]", String.join(",", POSSIBLE_LICENSE_FILE_NAMES)));
        }
        return optionalLicenseBody;
    }

    /**
     * Verify if the project POM defines at least a {@link License}, if exist tries to download the license body using
     * the {@link License#getUrl()}
     *
     * @return {@link Optional<String>} with the license body
     */
    private Optional<String> getLicenseFromPom()
    {
        Optional<String> licenseBody = Optional.empty();

        if (pomFoundLicenses.isEmpty())
        {
            LOGGER.warn("No license will be used in the Studio plugin: No license section was found at pom.xml.");
        }
        else
        {
            License mavenLicenseObject = pomFoundLicenses.get(0);
            final Optional<String> optionalUrl = getLicenseUrl(mavenLicenseObject);

            if (optionalUrl.isPresent())
            {
                licenseBody = downloadContentFromUrl(optionalUrl.get());
            }
            else
            {
                LOGGER.warn("No license will be used in the Studio plugin: No license URL was specified.");
            }
        }

        return licenseBody;
    }

    /**
     * Given a URL, tries to download the content and save it into a String.
     *
     * @param url of the file to download
     * @return an {@link Optional<String>} with the content of the URL
     */
    private Optional<String> downloadContentFromUrl(String url)
    {
        Reader reader = null;
        try
        {
            URL licenseURL = new URL(url);
            reader = new BufferedReader(new InputStreamReader(licenseURL.openStream()));
            return Optional.of(IOUtils.toString(reader));
        }
        catch (IOException e)
        {
            LOGGER.warn("No license will be used in the Studio plugin: Error reading from the remote URL:  " + e.getMessage());
            return Optional.empty();
        }
        finally
        {
            IOUtils.closeQuietly(reader);
        }
    }

    private Optional<String> getLicenseURLFromPom()
    {
        if (!pomFoundLicenses.isEmpty())
        {
            final License firstLicense = pomFoundLicenses.get(0);
            if (pomFoundLicenses.size() > 1)
            {
                LOGGER.warn(String.format("More than one license has been found for this project, using [%s] license", firstLicense.getName()));
            }
            return getLicenseUrl(firstLicense);
        }

        return Optional.empty();
    }

    private Optional<String> getLicenseUrl(License mavenLicenseObject)
    {
        if (mavenLicenseObject == null || StringUtils.isBlank(mavenLicenseObject.getUrl()))
        {
            return Optional.empty();
        }
        return Optional.of(mavenLicenseObject.getUrl());
    }

    private String readContentFromFile(File filePath) throws IOException
    {
        try (FileReader licenseReader = new FileReader(filePath))
        {
            return IOUtils.toString(licenseReader);
        }
    }
}