package org.mule.extension.maven.generator;

import java.io.File;
import java.net.URLDecoder;

import org.apache.maven.plugin.MojoFailureException;
import org.eclipse.sisu.equinox.launching.internal.P2ApplicationLauncher;

/**
 * Generator that populates the final Update Site generating the 'artifacts.xml' and 'content.xml', files that describes
 * the artifacts that are bundled inside of it.
 *
 * @since 1.0
 */
public class FeaturesAndBundlesGenerator
{

    private static final String UTF_8 = "UTF-8";
    private static final String CATEGORY_XML = "category.xml";
    private final GenerationContext context;
    private final P2ApplicationLauncher launcher;
    private final P2ApplicationLauncher launcher2;
    private final String UPDATE_SITE_ERROR_MESSAGE = "Error creating update site: Return code ";

    public FeaturesAndBundlesGenerator(GenerationContext context, P2ApplicationLauncher launcher, P2ApplicationLauncher launcher2)
    {
        this.context = context;
        this.launcher = launcher;
        this.launcher2 = launcher2;
    }

    public void generate() throws MojoFailureException
    {
        try
        {
            File categoryFile = new File(context.getOutputDirectory(), CATEGORY_XML).getAbsoluteFile();

            if (!categoryFile.exists())
            {
                throw new MojoFailureException(String.format("Update Site: '%s' does not exist in [%s]", CATEGORY_XML, categoryFile.getParentFile()));
            }
            String categoryXmlURI = URLDecoder.decode(categoryFile.toURI().toString(), UTF_8);

            final File updateSitePath = new File(context.getOutputDirectory(), "update-site");
            String updateSiteURI = URLDecoder.decode(updateSitePath.toURI().toString(), UTF_8);

            int returnCode = featuresAndBundles(launcher, updateSitePath, updateSiteURI);
            if (returnCode != 0)
            {
                throw new MojoFailureException(UPDATE_SITE_ERROR_MESSAGE + returnCode);
            }

            returnCode = categoryPublisher(launcher2, updateSiteURI, categoryXmlURI);
            if (returnCode != 0)
            {
                throw new MojoFailureException(UPDATE_SITE_ERROR_MESSAGE + returnCode);
            }
        }
        catch (Exception e)
        {
            throw new MojoFailureException("Error occurred populating the Update Site", e);
        }
    }

    /**
     * @param launcher       in charge of create the 'artifacts.xml' and 'content.xml' file
     * @param updateSitePath to the Update Site to populate
     * @param updateSiteURI  to the Update Site to populate
     * @return an result status code
     */
    private int featuresAndBundles(P2ApplicationLauncher launcher, File updateSitePath, String updateSiteURI)
    {
        launcher.addArguments("-artifactRepository", updateSiteURI);
        launcher.addArguments("-metadataRepository", updateSiteURI);
        launcher.addArguments("-source", updateSitePath.getAbsolutePath());
        launcher.addArguments("-publishArtifacts", "-append");
        launcher.setApplicationName("org.eclipse.equinox.p2.publisher.FeaturesAndBundlesPublisher");

        return launcher.execute(20);
    }

    /**
     * @param launcher       in to publish the existent 'category.xml' file
     * @param updateSiteURI  to the Update Site to populate
     * @param categoryXmlURI to the generated 'category.xml' file
     * @return an result status code
     */
    private int categoryPublisher(P2ApplicationLauncher launcher, String updateSiteURI, String categoryXmlURI)
    {
        launcher.addArguments("-categoryDefinition", categoryXmlURI);
        launcher.addArguments("-categoryQualifier");
        launcher.addArguments("-metadataRepository", updateSiteURI);

        launcher.setApplicationName("org.eclipse.equinox.p2.publisher.CategoryPublisher");
        return launcher.execute(20);
    }
}
