/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.extension.maven.generator;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

/**
 * Utility class to given a {@link File} pointing to a JAR, signs it using the configured certificate in the
 * constructor.
 *
 * @since 1.0
 */
public class ComponentSigner
{

    private static final int OPTIONS_ARRAY_CAPACITY = 11;
    private final String keystorePath;
    private final String storepass;
    private final String keypass;
    private final String alias;
    private final String tsaUrl;

    /**
     * Creates a new {@link ComponentSigner} instance. This instance can be reusable to sign different artifacts
     *
     * @param keystorePath path of the security certificates repository
     * @param storepass    password to access to the keystore repository
     * @param keypass      password to access to the particular certificate inside the repository
     * @param alias        identifier of the certificate to use
     * @param tsaUrl       url of the Time Stamping Authority (TSA)
     */
    public ComponentSigner(String keystorePath, String storepass, String keypass, String alias, String tsaUrl)
    {
        if (keystorePath == null)
        {
            throw new IllegalArgumentException("Keystore Path can not be null");
        }

        if (!new File(keystorePath).exists())
        {
            throw new IllegalArgumentException("The given Keystore [%s] doesn't");
        }

        this.keystorePath = keystorePath;
        this.storepass = storepass;
        this.keypass = keypass;
        this.alias = alias;
        this.tsaUrl = tsaUrl;
    }

    /**
     * Signs the given JAR {@link File} using the configured certificate for this instance.
     *
     * @param jarFile {@link File} pointing to the JAR file to be signed
     */
    public void sign(File jarFile)
    {
        List<String> jarSignerParameters = buildParametersList(jarFile.getPath());
        try
        {
            sun.security.tools.jarsigner.Main.main(jarSignerParameters.toArray(new String[OPTIONS_ARRAY_CAPACITY]));
        }
        catch (Exception e)
        {
            throw new RuntimeException("Could not sign the generated jar file.", e);
        }
    }

    /**
     * Given the {@param path} the the JAR to be signed and the configured properties
     * {@link ComponentSigner#keystorePath}, {@link ComponentSigner#keypass}, {@link ComponentSigner#storepass},
     * {@link ComponentSigner#tsaUrl} and {@link ComponentSigner#alias}, creates the list of parameters to be passed
     * to the JarSigner
     *
     * @param path to the JAR to sign
     * @return the list of parameters to be passed to the JarSigner
     */
    private List<String> buildParametersList(String path)
    {
        ArrayList<String> options = new ArrayList<>();
        options.add("-keystore");
        options.add(keystorePath);

        if (storepass != null)
        {
            options.add("-storepass");
            options.add(storepass);
        }

        if (keypass != null)
        {
            options.add("-keypass");
            options.add(keypass);
        }

        if (tsaUrl != null)
        {
            options.add("-tsa");
            options.add(tsaUrl);
        }

        options.add("-verbose");
        options.add(path);
        options.add(alias);

        return options;
    }
}
