/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.extension.maven;

import static org.mule.extension.maven.ExtensionMojoUtils.createDirectoryIfNotExist;

import org.mule.extension.maven.generator.ComponentSigner;
import org.mule.extension.maven.generator.FeatureJarGenerator;
import org.mule.extension.maven.generator.FeaturesAndBundlesGenerator;
import org.mule.extension.maven.generator.GenerationContext;
import org.mule.extension.maven.generator.PluginJarGenerator;
import org.mule.extension.maven.generator.StudioEditorsGenerator;
import org.mule.extension.maven.generator.StudioFeatureGenerator;
import org.mule.extension.maven.generator.StudioIconsGenerator;
import org.mule.extension.maven.generator.StudioManifestGenerator;
import org.mule.extension.maven.generator.StudioPluginXmlGenerator;
import org.mule.extension.maven.generator.StudioPluginZipGenerator;
import org.mule.extension.maven.generator.StudioSiteXmlGenerator;
import org.mule.plugin.maven.AbstractMuleMojo;
import org.mule.runtime.extension.api.introspection.ExtensionModel;
import org.mule.runtime.extension.api.manifest.ExtensionManifest;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.util.jar.JarFile;
import java.util.zip.ZipEntry;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProjectHelper;
import org.eclipse.sisu.equinox.launching.internal.P2ApplicationLauncher;

/**
 * Maven Plugin to create Studio Update Sites for Mule Extensions.
 *
 * @since 1.0
 */
@Mojo(name = "studio-package", defaultPhase = LifecyclePhase.PACKAGE, requiresDependencyResolution = ResolutionScope.RUNTIME)
public class ExtensionStudioPackageMojo extends AbstractMuleMojo
{

    private static final String EXTENSION_MODEL_JSON = "extension-model.json";
    @Component
    private MavenProjectHelper projectHelper;

    @Parameter(defaultValue = "${project.build.outputDirectory}", required = true)
    private File classesDirectory;

    @Parameter(property = "skipStudioPackage", defaultValue = "${skip.studio.package}")
    private boolean skipStudioPackage;

    @Parameter(property = "signExtension", defaultValue = "${signExtension}")
    private boolean signExtension;

    @Parameter(property = "keystorePath", defaultValue = "${keystorePath}")
    private String keystorePath;

    @Parameter(property = "alias", defaultValue = "${alias}")
    private String alias;

    @Parameter(property = "storepass", defaultValue = "${storepass}")
    private String storepass;

    @Parameter(property = "keypass", defaultValue = "${keypass}")
    private String keypass;

    @Parameter(property = "tsaUrl", defaultValue = "${tsaUrl}")
    private String tsaUrl;

    @Component
    private P2ApplicationLauncher launcher;

    @Component
    private P2ApplicationLauncher launcher2;

    private GenerationContext context;
    private ExtensionMojoUtils extensionMojoUtils;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException
    {
        if (!skipStudioPackage)
        {
            try
            {
                extensionMojoUtils = new ExtensionMojoUtils(outputDirectory, finalName);
                final ExtensionManifest manifest = extensionMojoUtils.getExtensionManifest();
                final ExtensionModel extensionModel = extensionMojoUtils.getExtensionModel();
                context = new GenerationContext(project, manifest, extensionModel, outputDirectory);

                generateRequiredArtifacts();
                packageArtifacts();
                packageUpdateSite();
            }
            catch (Exception e)
            {
                throw new MojoFailureException("An error occurred creating the Studio UpdateSite", e);
            }
        }
    }

    private void generateRequiredArtifacts() throws MojoFailureException, IOException, ParserConfigurationException, TransformerException
    {
        final File extensionFolder = new File(classesDirectory, context.getExtensionPluginName());
        createDirectoryIfNotExist(extensionFolder);
        copyExtensionModel();

        new StudioIconsGenerator(context).generate();
        new StudioEditorsGenerator(context).generate();
        new StudioManifestGenerator(context).generate();
        new StudioPluginXmlGenerator(context).generate();
        new StudioSiteXmlGenerator(context).generate();
        new StudioFeatureGenerator(context).generate();
    }

    private void packageArtifacts() throws MojoFailureException
    {
        final File featureJar = new FeatureJarGenerator(context).generate();
        final File pluginJar = new PluginJarGenerator(context).generate();

        if (signExtension)
        {
            ComponentSigner componentSigner = new ComponentSigner(keystorePath, storepass, keypass, alias, tsaUrl);
            componentSigner.sign(featureJar);
            componentSigner.sign(pluginJar);
        }
    }

    private void packageUpdateSite() throws MojoFailureException
    {
        new FeaturesAndBundlesGenerator(context, launcher, launcher2).generate();
        new StudioPluginZipGenerator(context, projectHelper).generate();
    }

    private void copyExtensionModel() throws MojoFailureException
    {
        try (JarFile jar = new JarFile(extensionMojoUtils.getExtensionJar()))
        {
            ZipEntry extensionModelEntry = jar.getEntry("META-INF/" + EXTENSION_MODEL_JSON);
            if (extensionModelEntry == null)
            {
                throw new MojoFailureException(String.format("The extension does not contain the '%s' file", EXTENSION_MODEL_JSON));
            }
            final InputStream inputStream = jar.getInputStream(extensionModelEntry);
            final File file = new File(context.getPluginOutputDirectory(), EXTENSION_MODEL_JSON);
            Files.copy(inputStream, file.toPath());
        }
        catch (Exception e)
        {
            throw new MojoFailureException("Could not copy extension model file to the Studio Plugin", e);
        }
    }
}
