/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.extension.mule.testing.processing.strategies.test.internal;

import static java.lang.Thread.currentThread;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static org.mule.runtime.api.connection.ConnectionValidationResult.success;
import static org.mule.runtime.api.scheduler.SchedulerConfig.config;

import org.mule.extension.mule.testing.processing.strategies.test.api.ExecutionThreadTracker;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.connection.PoolingConnectionProvider;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.lifecycle.Lifecycle;
import org.mule.runtime.api.lifecycle.LifecycleException;
import org.mule.runtime.api.scheduler.SchedulerService;

import javax.inject.Inject;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;

public class MuleTestingProcessingStrategiesTestConnectionProvider implements PoolingConnectionProvider<CompletionCallbacksDeferrer>, Lifecycle {

  private final AtomicBoolean isStopped = new AtomicBoolean(false);

  private ScheduledExecutorService executorService;
  private BlockingQueue<CompleterTask> callbacksQueue;
  private Future<?> allCallbacksCompleted;

  @Inject
  private SchedulerService schedulerService;

  @Inject
  private ExecutionThreadTracker executionThreadTracker;

  @Override
  public CompletionCallbacksDeferrer connect() throws ConnectionException {
    return new CompletionCallbacksDeferrer(callbacksQueue);
  }

  @Override
  public void disconnect(CompletionCallbacksDeferrer connection) {
  }

  @Override
  public ConnectionValidationResult validate(CompletionCallbacksDeferrer connection) {
    return success();
  }

  @Override
  public void initialise() throws InitialisationException {
    this.callbacksQueue = new ArrayBlockingQueue<>(10);
    this.executorService = schedulerService.customScheduler(config().withName("Deferred callbacks completion thread").withMaxConcurrentTasks(1));
  }

  @Override
  public void dispose() {
    this.executorService.shutdown();
  }

  @Override
  public void start() throws MuleException {
    allCallbacksCompleted = executorService.submit(this::completeCallbacks);
    isStopped.set(false);
  }

  private void completeCallbacks() {
    try {
      while (!isStopped.get()) {
        CompleterTask completerTask = callbacksQueue.poll(500L, MILLISECONDS);
        if (completerTask != null) {
          completerTask.complete(executionThreadTracker);
        }
      }
    } catch (InterruptedException e) {
      currentThread().interrupt();
    }
  }

  @Override
  public void stop() throws MuleException {
    try {
      isStopped.set(true);
      allCallbacksCompleted.get();
    } catch (ExecutionException e) {
      throw new LifecycleException(e, this);
    } catch (InterruptedException e) {
      currentThread().interrupt();
    }
  }
}
