/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.support;

import static java.lang.String.format;
import static javax.jms.DeliveryMode.NON_PERSISTENT;
import static javax.jms.DeliveryMode.PERSISTENT;
import static org.mule.jms.commons.api.connection.JmsSpecification.JMS_1_1;

import org.mule.jms.commons.api.connection.JmsSpecification;
import org.mule.jms.commons.api.connection.LookupJndiDestination;
import org.mule.jms.commons.api.destination.ConsumerType;
import org.mule.jms.commons.api.destination.TopicConsumer;

import java.util.Optional;
import java.util.function.Function;

import javax.jms.CompletionListener;
import javax.jms.DeliveryMode;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageProducer;
import javax.jms.Queue;
import javax.jms.QueueSender;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicPublisher;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <code>Jms11Support</code> is a template class to provide an abstraction to to
 * the JMS 1.1 API specification.
 *
 * @since 1.0
 */
public class Jms11Support extends Jms20Support {

  private Logger LOGGER = LoggerFactory.getLogger(Jms11Support.class);

  public Jms11Support() {
    super();
  }

  public Jms11Support(LookupJndiDestination lookupJndiDestination, Function<String, Optional<Destination>> jndiObjectSupplier) {
    super(lookupJndiDestination, jndiObjectSupplier);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JmsSpecification getSpecification() {
    return JMS_1_1;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MessageConsumer createConsumer(Session session, Destination destination, String messageSelector, ConsumerType type)
      throws JMSException {

    if (!type.topic()) {
      return session.createConsumer(destination, messageSelector);
    }

    TopicConsumer topicConsumer = (TopicConsumer) type;
    if (topicConsumer.isDurable()) {
      return session.createDurableSubscriber((Topic) destination, topicConsumer.getSubscriptionName(), messageSelector,
                                             topicConsumer.isNoLocal());
    }

    return session.createConsumer(destination, messageSelector, topicConsumer.isNoLocal());
  }

  /**
   * Sends the given {@link Message} to the indicated {@link Destination} using the parametrized configurations.
   * Because of async publishing didn't exist on JMS 1.1 or prior. This method will be blocking and sync.
   *
   * @param producer the {@link MessageProducer} to use for sending the message
   * @param message the {@link Message} to send
   * @param persistent {@code true} if {@link DeliveryMode#PERSISTENT} should be used
   * @param priority the {@link Message#getJMSPriority} to be set on send
   * @param ttl the message's lifetime (in milliseconds)
   * @param topic used by {@link JmsSpecification#JMS_1_0_2b} to decide whether to use the {@link MessageProducer}
   *              as {@link TopicPublisher} or a {@link QueueSender}
   * @throws JMSException if an error occurs during messsage sending
   */
  @Override
  public void send(MessageProducer producer, Message message, boolean persistent, int priority, long ttl, boolean topic,
                   CompletionListener completionListener)
      throws JMSException {
    if (LOGGER.isDebugEnabled()) {
      Destination dest = producer.getDestination();
      LOGGER.debug(format("Sending message to [%s], persistent:[%s], with priority:[%s] and ttl:[%s]",
                          dest instanceof Queue ? ((Queue) dest).getQueueName() : ((Topic) dest).getTopicName(),
                          persistent, priority, ttl));
    }
    try {
      producer.send(message, (persistent ? PERSISTENT : NON_PERSISTENT), priority, ttl);
      completionListener.onCompletion(message);
    } catch (Exception e) {
      completionListener.onException(message, e);
    }
  }
}
