/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.source.push;

import static java.lang.String.format;
import static org.mule.jms.commons.internal.common.JmsCommons.closeQuietly;
import static org.mule.jms.commons.internal.common.JmsCommons.getDestinationType;
import static org.mule.jms.commons.internal.config.InternalAckMode.AUTO;
import static org.mule.jms.commons.internal.config.InternalAckMode.DUPS_OK;
import static org.slf4j.LoggerFactory.getLogger;
import org.mule.jms.commons.api.destination.ConsumerType;
import org.mule.jms.commons.api.exception.JmsExtensionException;
import org.mule.jms.commons.api.lock.JmsListenerLockFactory;
import org.mule.jms.commons.internal.config.InternalAckMode;
import org.mule.jms.commons.internal.config.JmsConfig;
import org.mule.jms.commons.internal.connection.JmsConnection;
import org.mule.jms.commons.internal.connection.session.JmsSession;
import org.mule.jms.commons.internal.consume.JmsMessageConsumer;
import org.mule.jms.commons.internal.source.JmsListenerLock;
import org.mule.jms.commons.internal.source.JmsResourceReleaser;
import org.mule.jms.commons.internal.source.MessageConsumerDelegate;
import org.mule.jms.commons.internal.support.JmsSupport;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.message.Error;
import org.mule.runtime.extension.api.runtime.source.SourceCallbackContext;

import java.util.ArrayList;
import java.util.List;

import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.MessageListener;

import org.slf4j.Logger;

/**
 * {@link MessageConsumerDelegate} implementation which uses {@link MessageListener}
 *
 * @since 1.0
 */
public class JmsMessageListenerDelegate implements MessageConsumerDelegate {

  private static final Logger LOGGER = getLogger(JmsMessageListenerDelegate.class);

  private JmsMessageListenerFactory messageListenerFactory;
  private JmsConnection connection;
  private final JmsSupport jmsSupport;
  private final ConsumerType consumerType;
  private final String destination;
  private final JmsConfig config;
  private final InternalAckMode resolvedAckMode;
  private final String selector;
  private final List<MessageListenerContext> createdListeners = new ArrayList<>();
  private final JmsResourceReleaser resourceCleaner;
  private final JmsListenerLockFactory lockFactory;

  static final String JMS_LOCK_VAR = "JMS_LOCK";

  public JmsMessageListenerDelegate(JmsMessageListenerFactory messageListenerFactory, JmsConnection connection,
                                    JmsSupport jmsSupport, ConsumerType consumerType, String destination, JmsConfig config,
                                    InternalAckMode resolvedAckMode, String selector, JmsListenerLockFactory lockFactory,
                                    JmsResourceReleaser resourceCleaner) {
    this.messageListenerFactory = messageListenerFactory;
    this.connection = connection;
    this.jmsSupport = jmsSupport;
    this.consumerType = consumerType;
    this.destination = destination;
    this.config = config;
    this.resolvedAckMode = resolvedAckMode;
    this.selector = selector;
    this.lockFactory = lockFactory;
    this.resourceCleaner = resourceCleaner;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void createConsumers(int numberOfConsumers) throws ConnectionException {
    try {
      lockFactory.setEnabled(true);
      for (int i = 0; i < numberOfConsumers; i++) {
        JmsSession session;
        session = connection.createSession(resolvedAckMode, consumerType.topic());
        final Destination jmsDestination = jmsSupport.createDestination(session.get(), destination, consumerType.topic(), config);
        final JmsMessageConsumer consumer = connection.createConsumer(session, jmsDestination, selector, consumerType);

        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug(format("Creating Message Listener on Session [%s] for destination [%s]",
                              session.get(), destination));
        }


        JmsListenerLock jmsLock = lockFactory.createLock(resolvedAckMode);
        JmsMessageListener listener = messageListenerFactory.createMessageListener(session, jmsLock);
        createdListeners.add(new MessageListenerContext(session, jmsLock, consumer, listener));
        consumer.listen(listener);
      }
    } catch (JMSException | JmsExtensionException e) {
      String msg = format("An error occurred while creating the consumers for destination [%s:%s]: %s",
                          getDestinationType(consumerType), destination, e.getMessage());
      LOGGER.error(msg, e);
      stop();

      throw new ConnectionException(msg, e, null, connection);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onSuccess(SourceCallbackContext callbackContext) {
    callbackContext.<JmsListenerLock>getVariable(JMS_LOCK_VAR)
        .ifPresent(JmsListenerLock::unlock);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void stop() {
    try {
      createdListeners.forEach(info -> {
        closeConsumerQuietly(info);
      });
    } finally {
      createdListeners.clear();
    }
  }

  @Override
  public void disableConsumers() {
    // Clean message listeners
    createdListeners.forEach(info -> disableConsumerListener(info));
  }

  protected void disableConsumerListener(MessageListenerContext info) {
    resourceCleaner.releaseConsumerMessageListener(info);
  }

  protected void closeConsumerQuietly(MessageListenerContext info) {
    try {
      closeConsumer(info);
    } catch (Exception e) {
      LOGGER.debug("Failed to close consumer %s", info.getConsumer());
    } finally {
      closeQuietly(info.getSession());
    }
  }


  private void closeConsumer(MessageListenerContext info) {
    try {
      disableConsumerListener(info);
    } finally {
      info.getLock().unlockWithFailure();
      closeQuietly(info.getConsumer());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onError(SourceCallbackContext callbackContext, Error error) {
    callbackContext.<JmsListenerLock>getVariable(JMS_LOCK_VAR)
        .ifPresent(jmsLock -> {
          if (resolvedAckMode.equals(AUTO) || resolvedAckMode.equals(DUPS_OK)) {
            jmsLock.unlockWithFailure(error);
          } else {
            jmsLock.unlock();
          }
        });
  }

}
