/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.operation;

import static java.lang.String.format;
import static org.mule.jms.commons.internal.common.JmsCommons.EXAMPLE_CONTENT_TYPE;
import static org.mule.jms.commons.internal.common.JmsCommons.EXAMPLE_ENCODING;
import static org.mule.jms.commons.internal.common.JmsCommons.createJmsSession;
import static org.mule.jms.commons.internal.common.JmsCommons.evaluateMessageAck;
import static org.mule.jms.commons.internal.common.JmsCommons.getDestinationType;
import static org.mule.jms.commons.internal.common.JmsCommons.isPartOfCurrentTx;
import static org.mule.jms.commons.internal.common.JmsCommons.releaseResources;
import static org.mule.jms.commons.internal.common.JmsCommons.resolveMessageContentType;
import static org.mule.jms.commons.internal.common.JmsCommons.resolveMessageEncoding;
import static org.mule.jms.commons.internal.common.JmsCommons.resolveOverride;
import static org.mule.jms.commons.internal.common.JmsCommons.toInternalAckMode;
import static org.mule.jms.commons.internal.config.InternalAckMode.AUTO;
import static org.mule.jms.commons.internal.config.InternalAckMode.DUPS_OK;
import static org.mule.jms.commons.internal.config.InternalAckMode.IMMEDIATE;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.jms.commons.api.config.JmsConsumerConfig;
import org.mule.jms.commons.api.destination.ConsumerType;
import org.mule.jms.commons.api.exception.JmsConsumeErrorTypeProvider;
import org.mule.jms.commons.api.exception.JmsConsumeException;
import org.mule.jms.commons.api.exception.JmsExtensionException;
import org.mule.jms.commons.api.exception.JmsSecurityException;
import org.mule.jms.commons.api.message.JmsAttributes;
import org.mule.jms.commons.internal.config.InternalAckMode;
import org.mule.jms.commons.internal.config.JmsAckMode;
import org.mule.jms.commons.internal.config.JmsConfig;
import org.mule.jms.commons.internal.connection.JmsTransactionalConnection;
import org.mule.jms.commons.internal.connection.session.JmsSession;
import org.mule.jms.commons.internal.connection.session.JmsSessionManager;
import org.mule.jms.commons.internal.consume.JmsMessageConsumer;
import org.mule.jms.commons.internal.message.JmsResultFactory;
import org.mule.jms.commons.internal.metadata.JmsOutputResolver;
import org.mule.jms.commons.internal.support.JmsSupport;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.metadata.OutputResolver;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.ConfigOverride;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.display.Example;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.mule.runtime.extension.api.tx.OperationTransactionalAction;

import java.util.concurrent.TimeUnit;

import javax.jms.CompletionListener;
import javax.jms.Destination;
import javax.jms.IllegalStateException;
import javax.jms.JMSException;
import javax.jms.JMSSecurityException;
import javax.jms.Message;

import org.slf4j.Logger;

/**
 * Operation that allows the user to consume a single {@link Message} from a given {@link Destination}
 *
 * @since 1.0
 */
public class JmsConsume implements Disposable {

  private static final Logger LOGGER = getLogger(JmsConsume.class);

  private final JmsResultFactory resultFactory = JmsResultFactory.getInstance();
  private Scheduler scheduler;

  public JmsConsume(JmsSessionManager sessionManager, SchedulerService schedulerService) {
    this.sessionManager = sessionManager;
    this.scheduler = schedulerService.ioScheduler();
  }

  private JmsSessionManager sessionManager;

  @OutputResolver(output = JmsOutputResolver.class)
  @Throws(JmsConsumeErrorTypeProvider.class)
  public void consume(@Config JmsConfig config,
                      @Connection JmsTransactionalConnection connection,
                      @Summary("The name of the Destination from where the Message should be consumed") String destination,
                      @ConfigOverride @Summary("The Type of the Consumer that should be used for the provided destination") ConsumerType consumerType,
                      @Optional @Summary("The Session ACK mode to use when consuming a message") JmsAckMode ackMode,
                      @ConfigOverride @Summary("The JMS selector to be used for filtering incoming messages") String selector,
                      @Optional @Summary("The content type of the message body") @Example(EXAMPLE_CONTENT_TYPE) String contentType,
                      @Optional @Summary("The encoding of the message body") @Example(EXAMPLE_ENCODING) String encoding,
                      @Optional(
                          defaultValue = "10000") @Summary("Maximum time to wait for a message to arrive before timeout") Long maximumWait,
                      @Optional(
                          defaultValue = "MILLISECONDS") @Example("MILLISECONDS") @Summary("Time unit to be used in the maximumWaitTime configuration") TimeUnit maximumWaitUnit,
                      OperationTransactionalAction transactionalAction,
                      CompletionCallback<Object, JmsAttributes> completionCallback)
      throws JmsExtensionException {

    InternalAckMode resolvedAckMode = resolveAck(config.getConsumerConfig(), ackMode);
    JmsSession session = null;
    boolean partOfCurrentTx = false;
    JmsMessageConsumer consumer = null;
    try {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Begin [consume] on the " + getDestinationType(consumerType) + ": ["
            + destination + "]");
      }

      JmsSupport jmsSupport = connection.getJmsSupport();
      session = createJmsSession(connection, resolvedAckMode, consumerType.topic(), sessionManager, transactionalAction);
      partOfCurrentTx = isPartOfCurrentTx(session, connection, sessionManager);
      Destination jmsDestination = jmsSupport.createDestination(session.get(), destination, consumerType.topic(), config);

      consumer = connection.createConsumer(session, jmsDestination, selector, consumerType);

      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Consuming Message from the " + getDestinationType(consumerType) + ": ["
            + destination + "]");
      }

      JmsMessageConsumer finalConsumer = consumer;
      JmsSession finalSession = session;
      boolean finalPartOfCurrentTx = partOfCurrentTx;

      consumer.consume(maximumWaitUnit.toMillis(maximumWait), scheduler, new CompletionListener() {

        @Override
        public void onCompletion(Message received) {
          try {
            // If no explicit content type was provided to the operation, fallback to the
            // one communicated in the message properties. Finally if no property was set,
            // use the default one provided by the config

            if (received == null) {
              completionCallback.success(resultFactory.createEmptyResult());
            } else {
              String resolvedContentType =
                  resolveOverride(resolveMessageContentType(received, config.getContentType()), contentType);
              String resolvedEncoding = resolveOverride(resolveMessageEncoding(received, config.getEncoding()), encoding);

              Result<Object, JmsAttributes> result = resultFactory.createResult(received, jmsSupport.getSpecification(),
                                                                                resolvedContentType, resolvedEncoding,
                                                                                finalSession.getAckId());

              evaluateMessageAck(resolvedAckMode, finalSession, received, sessionManager, null);

              completionCallback.success(result);
            }

          } catch (Exception e) {
            String msg = format("An error occurred while consuming a message from the %s [%s]: %s",
                                getDestinationType(consumerType), destination, e.getMessage());
            completionCallback.error(new JmsConsumeException(msg, e));
          } finally {
            scheduler.submit(() -> releaseResources(finalSession, finalPartOfCurrentTx, finalConsumer));
          }
        }

        @Override
        public void onException(Message message, Exception e) {
          try {
            finalConsumer.listen(null);
          } catch (JMSException exception) {
            LOGGER
                .warn("An unknown error occurred trying to shutdown a listener. Listener's Session and Consumer will be closed.",
                      exception);
          } finally {
            releaseResources(finalSession, finalPartOfCurrentTx, finalConsumer);
          }
          if (e instanceof JMSSecurityException) {
            String msg = format("A security error occurred while consuming a message from the %s: [%s]: %s",
                                getDestinationType(consumerType), destination, e.getMessage());
            completionCallback.error(new JmsSecurityException(msg, e));
          } else if (e instanceof JmsConsumeException) {
            completionCallback.error(e);
          } else {
            String msg = format("An error occurred while consuming a message from the %s [%s]: %s",
                                getDestinationType(consumerType), destination, e.getMessage());
            completionCallback.error(new JmsConsumeException(msg, e));
          }
        }
      });

    } catch (JMSSecurityException e) {
      String msg = format("A security error occurred while consuming a message from the %s: [%s]: %s",
                          getDestinationType(consumerType), destination, e.getMessage());
      releaseResources(session, partOfCurrentTx, consumer);
      completionCallback.error(new JmsSecurityException(msg, e));
    } catch (IllegalStateException e) {
      boolean isConnectionError = session == null;
      String msg = format((isConnectionError ? "A connection error" : "An error")
          + " occurred while consuming a message from the %s: [%s]: %s",
                          getDestinationType(consumerType), destination, e.getMessage());
      completionCallback.error(isConnectionError ? new ConnectionException(e) : new JmsConsumeException(msg, e));
      releaseResources(session, partOfCurrentTx, consumer);
    } catch (Exception e) {
      String msg = format("An error occurred while consuming a message from the %s [%s]: %s",
                          getDestinationType(consumerType), destination, e.getMessage());
      releaseResources(session, partOfCurrentTx, consumer);
      completionCallback.error(new JmsConsumeException(msg, e));
    }
  }

  private InternalAckMode resolveAck(JmsConsumerConfig config, JmsAckMode ackMode) {
    InternalAckMode fallbackAck = toInternalAckMode(config.getAckMode());
    if (AUTO.equals(fallbackAck) || DUPS_OK.equals(fallbackAck)) {
      fallbackAck = IMMEDIATE;
    }
    return resolveOverride(fallbackAck, toInternalAckMode(ackMode));
  }

  @Override
  public void dispose() {
    if (scheduler != null) {
      scheduler.stop();
    }
  }
}
