/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.message;

import static java.lang.String.format;
import static java.lang.String.join;
import static java.util.Optional.ofNullable;
import static org.mule.jms.commons.internal.message.JMSXDefinedPropertiesNames.JMSXAppID;
import static org.mule.jms.commons.internal.message.JMSXDefinedPropertiesNames.JMSXConsumerTXID;
import static org.mule.jms.commons.internal.message.JMSXDefinedPropertiesNames.JMSXDeliveryCount;
import static org.mule.jms.commons.internal.message.JMSXDefinedPropertiesNames.JMSXGroupID;
import static org.mule.jms.commons.internal.message.JMSXDefinedPropertiesNames.JMSXGroupSeq;
import static org.mule.jms.commons.internal.message.JMSXDefinedPropertiesNames.JMSXProducerTXID;
import static org.mule.jms.commons.internal.message.JMSXDefinedPropertiesNames.JMSXRcvTimestamp;
import static org.mule.jms.commons.internal.message.JMSXDefinedPropertiesNames.JMSXUserID;
import static org.mule.jms.commons.internal.message.JMSXDefinedPropertiesNames.JMSX_NAMES;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import org.mule.jms.commons.api.message.JmsxProperties;
import org.mule.runtime.api.util.Preconditions;

import java.util.HashMap;
import java.util.Map;

import javax.jms.Message;

/**
 * Builder that provides a simple way of creating a {@link JmsxProperties} instance based on
 * the predefined properties {@link JMSXDefinedPropertiesNames names}.
 * <p>
 * This is useful for converting the properties {@link Map} found in the original {@link Message}
 * to their representation as {@link JmsxProperties}.
 * A default value is provided for the properties that are not set.
 *
 * @since 1.0
 */
public final class JmsxPropertiesBuilder {

  private final Map<String, Object> properties = new HashMap<>();

  private JmsxPropertiesBuilder() {}

  public static JmsxPropertiesBuilder create() {
    return new JmsxPropertiesBuilder();
  }

  public JmsxPropertiesBuilder add(String key, Object value) {
    Preconditions.checkArgument(JMSXDefinedPropertiesNames.JMSX_NAMES.contains(key),
                                format("Invalid key [%s], supported keys for JMSXProperties are [%s]", key,
                                       String.join(", ", JMSXDefinedPropertiesNames.JMSX_NAMES)));

    properties.put(key, value);
    return this;
  }

  public JmsxProperties build() {
    return new JmsxProperties(ofNullable(properties.get(JMSXDefinedPropertiesNames.JMSXUserID)).map(String::valueOf).orElse(null),
                              ofNullable(properties.get(JMSXDefinedPropertiesNames.JMSXAppID)).map(String::valueOf).orElse(null),
                              Integer
                                  .valueOf(properties.getOrDefault(JMSXDefinedPropertiesNames.JMSXDeliveryCount, "1").toString()),
                              ofNullable(properties.get(JMSXDefinedPropertiesNames.JMSXGroupID)).map(String::valueOf)
                                  .orElse(null),
                              Integer.valueOf(properties.getOrDefault(JMSXDefinedPropertiesNames.JMSXGroupSeq, "1").toString()),
                              ofNullable(properties.get(JMSXDefinedPropertiesNames.JMSXProducerTXID)).map(String::valueOf)
                                  .orElse(null),
                              ofNullable(properties.get(JMSXDefinedPropertiesNames.JMSXConsumerTXID)).map(String::valueOf)
                                  .orElse(null),
                              ofNullable(properties.get(JMSXDefinedPropertiesNames.JMSXRcvTimestamp))
                                  .map(n -> Long.valueOf(n.toString())).orElse(null));
  }
}
