/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.api;

import static org.mule.metadata.api.model.MetadataFormat.JAVA;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.metadata.resolving.AttributesStaticTypeResolver;

/**
 * Is used to override the metadata generated by
 *
 * @since 1.0
 */
public class AttributesOutputResolver extends AttributesStaticTypeResolver {

  @Override
  public final MetadataType getStaticMetadata() {
    ObjectTypeBuilder attributesBuilder = BaseTypeBuilder.create(JAVA).objectType();

    attributesBuilder.id(attributesName());

    addJmsMessageProperties(attributesBuilder);
    addJmsHeaders(attributesBuilder);
    attributesBuilder.addField().required(false).key("ackId").value().stringType();

    return attributesBuilder.build();
  }

  public String attributesName() {
    return "JMS Attributes";
  }

  public void createJmsProperties(ObjectTypeBuilder propertiesBuilder) {
    propertiesBuilder.openWith().anyType();
  }

  private void addJmsHeaders(ObjectTypeBuilder attributesBuilder) {
    ObjectTypeBuilder headersBuilder = attributesBuilder
        .addField()
        .key("headers")
        .required()
        .value()
        .objectType();

    integerField(headersBuilder, "deliveryMode", true);
    integerField(headersBuilder, "expiration", true);
    integerField(headersBuilder, "priority", true);
    stringField(headersBuilder, "messageId", true);
    integerField(headersBuilder, "timestamp", true);
    stringField(headersBuilder, "correlationId", true);
    stringField(headersBuilder, "type", false);
    integerField(headersBuilder, "deliveryTime", true);

    headersBuilder.addField().key("redelivered").value().booleanType();

    headersBuilder.addField().key("destination").value(createDestinationType());
    headersBuilder.addField().key("replyTo").value(createDestinationType());
  }

  private MetadataType createDestinationType() {
    ObjectTypeBuilder objectTypeBuilder = BaseTypeBuilder.create(JAVA).objectType();
    objectTypeBuilder.id("JMS Destination");

    objectTypeBuilder
        .addField()
        .key("destination")
        .value().stringType();

    objectTypeBuilder
        .addField()
        .key("destinationType")
        .value().stringType()
        .enumOf("QUEUE", "TOPIC");

    return objectTypeBuilder.build();
  }

  private void addJmsMessageProperties(ObjectTypeBuilder attributesBuilder) {
    ObjectFieldTypeBuilder objectFieldTypeBuilder = attributesBuilder.addField();
    objectFieldTypeBuilder.key("properties");

    ObjectTypeBuilder propertiesBuilder = objectFieldTypeBuilder
        .value()
        .objectType()
        .id("JMS Message Properties");

    addMapField(propertiesBuilder, "all");
    addMapField(propertiesBuilder, "userProperties");
    createJmsProperties(propertiesBuilder.addField().key("jmsProperties").required().value().objectType());
    addJmsxProperties(propertiesBuilder);
  }

  private void addJmsxProperties(ObjectTypeBuilder propertiesBuilder) {
    ObjectTypeBuilder jmsxPropertiesBuilder = propertiesBuilder.addField()
        .key("jmsxProperties")
        .value()
        .objectType();
    jmsxPropertiesBuilder.id("JMSX Properties");

    stringField(jmsxPropertiesBuilder, "jmsxUserID", false);
    stringField(jmsxPropertiesBuilder, "jmsxAppID", false);
    integerField(jmsxPropertiesBuilder, "jmsxDeliveryCount", false);
    stringField(jmsxPropertiesBuilder, "jmsxGroupID", false);
    integerField(jmsxPropertiesBuilder, "jmsxGroupSeq", false);
    stringField(jmsxPropertiesBuilder, "jmsxProducerTXID", false);
    stringField(jmsxPropertiesBuilder, "jmsxConsumerTXID", false);
    integerField(jmsxPropertiesBuilder, "jmsxRcvTimestamp", false);
  }

  public void stringField(ObjectTypeBuilder jmsxPropertiesBuilder, String name, boolean required) {
    jmsxPropertiesBuilder
        .addField()
        .required(required)
        .key(name)
        .value()
        .stringType();
  }

  public void integerField(ObjectTypeBuilder jmsxPropertiesBuilder, String name, boolean required) {
    jmsxPropertiesBuilder
        .addField()
        .required(required)
        .key(name)
        .value()
        .numberType()
        .integer();
  }

  public void addMapField(ObjectTypeBuilder propertiesBuilder, String fieldName) {
    propertiesBuilder.addField()
        .key(fieldName)
        .required()
        .value()
        .objectType()
        .openWith().anyType();
  }
}
