/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.connection;

import static org.mule.jms.commons.internal.config.InternalAckMode.TRANSACTED;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.jms.commons.internal.common.JmsCommons;
import org.mule.jms.commons.internal.config.InternalAckMode;
import org.mule.jms.commons.internal.connection.exception.CompositeJmsExceptionListener;
import org.mule.jms.commons.internal.connection.session.DefaultJmsSession;
import org.mule.jms.commons.internal.connection.session.JmsSession;
import org.mule.jms.commons.internal.connection.session.JmsSessionManager;
import org.mule.jms.commons.internal.support.JmsSupport;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.tx.MuleXaObject;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.core.api.extension.ExtensionManager;
import org.mule.runtime.extension.api.connectivity.XATransactionalConnection;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Optional;
import java.util.function.Supplier;

import javax.jms.Connection;
import javax.jms.JMSException;
import javax.jms.Session;
import javax.transaction.xa.XAResource;

import org.slf4j.Logger;

/**
 * JMS connection created when using a {@link javax.jms.ConnectionFactory} with XA enabled.
 * <p/>
 * This class provides all the functionality required to work with Bitronix Transaction Manager.
 *
 * @since 1.4.0
 */
public class XaJmsTransactionalConnection extends JmsTransactionalConnection implements XATransactionalConnection {

  private static final Logger LOGGER = getLogger(XaJmsTransactionalConnection.class);

  private static Supplier<Method> getXAResourceMethod;

  static {
    try {
      Class<?> aClass = ExtensionManager.class.getClassLoader().loadClass("bitronix.tm.resource.jms.DualSessionWrapper");
      getXAResourceMethod = new LazyValue<>(aClass.getMethod("getXAResource"));
    } catch (Exception e) {
      getXAResourceMethod = () -> {
        throw new MuleRuntimeException(e);
      };
    }
  }

  public XaJmsTransactionalConnection(JmsSupport jmsSupport, Connection connection, JmsSessionManager jmsSessionManager,
                                      CompositeJmsExceptionListener exceptionListener) {
    super(jmsSupport, connection, jmsSessionManager, exceptionListener);
  }

  /**
   * This method gets called when there's an XA transaction in context. It can be invoked even before the
   * XAJmsTransactionalConnection get used to retrieve the {@link JmsSession}.
   */
  @Override
  public XAResource getXAResource() {
    Optional<JmsXaContext> optionalJmsXaContext = getJmsSessionManager().getJmsXaContext(this);
    if (optionalJmsXaContext.isPresent()) {
      return optionalJmsXaContext.get().getXaResource();
    } else {
      try {
        // TODO MULE-16877 - JMS Client creates a session per XA Polling iteration
        Session session = getJmsSupport().createSession(get(), false, true, TRANSACTED.getAckModeValue());
        // method call to trigger session creation before using reflection so the XAResource gets generated.
        session.createMessage();
        XAResource xaResource = getXAResource((MuleXaObject) session);
        getJmsSessionManager().bindToTransaction(this, new DefaultJmsSession(session), new JmsXaContext(xaResource));
        return xaResource;
      } catch (Exception e) {
        throw new MuleRuntimeException(e);
      }
    }
  }

  private XAResource getXAResource(MuleXaObject session) throws IllegalAccessException, InvocationTargetException {
    return (XAResource) getXAResourceMethod.get().invoke(session.getTargetObject());
  }

  /**
   * Provides access to the {@link JmsSession} to be used. If there's a transactional context with a session already attached then
   * that session will be returned. Otherwise a new session will be created.
   *
   * @param ackMode
   * @param isTopic
   * @return
   * @throws JMSException
   */
  public JmsSession getSession(InternalAckMode ackMode, boolean isTopic) throws JMSException {
    Optional<JmsSession> transactedSession = getJmsSessionManager().getXaTransactedSession(this);
    if (transactedSession.isPresent()) {
      return transactedSession.get();
    }
    return createSession(ackMode, isTopic);
  }

  /**
   * Cleanups all the resources associated with the session.
   */
  @Override
  public void close() {
    getJmsSessionManager().getTransactedSession(this).ifPresent(JmsCommons::closeQuietly);
    getJmsSessionManager().getJmsXaContext(this).ifPresent(jmx -> {
      try {
        jmx.end();
      } catch (Throwable t) {
        LOGGER.warn("Exception found while stopping Jms XA Context", t);
      }
    });
    getJmsSessionManager().unbindSession(this);
  }

}
