/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.consume;

import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static java.lang.String.format;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.jms.commons.api.exception.JmsConsumeException;
import org.mule.jms.commons.api.exception.JmsTimeoutException;
import org.mule.runtime.api.scheduler.Scheduler;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

import javax.jms.CompletionListener;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageListener;

import org.apache.commons.lang3.time.StopWatch;
import org.slf4j.Logger;

/**
 * Wrapper implementation of a JMS {@link MessageConsumer}
 *
 * @since 1.0
 */
public final class JmsMessageConsumer implements AutoCloseable {

  private static final Double DELTA = 0.01;
  private static final Logger LOGGER = getLogger(JmsMessageConsumer.class);
  private final MessageConsumer consumer;

  public JmsMessageConsumer(MessageConsumer consumer) {
    checkArgument(consumer != null, "A non null MessageConsumer is required to use as delegate");
    this.consumer = consumer;
  }

  /**
   * Sets the {@link MessageConsumer} message listener.
   *
   * @param listener the listener to which the messages are to be delivered.
   * @throws JMSException - if the JMS provider fails to receive the next message due to some internal error.
   */
  public void listen(MessageListener listener) throws JMSException {
    consumer.setMessageListener(listener);
  }

  /**
   * Consumes a message in a sync wait.
   *
   * @param maximumWaitTime Time to wait, in milliseconds, until a timeout is raised.
   * @return the next {@link Message} received by the message consumer, null if the timeout is 0 (message polling).
   * @throws JMSException        - if the JMS provider fails to receive the next message due to some internal error.
   * @throws JmsTimeoutException - if no message is received within maximumWaitTime.
   */
  public Message consume(Long maximumWaitTime) throws JMSException, JmsTimeoutException {
    synchronized (consumer) {

      if (maximumWaitTime == -1) {
        return receive();
      }

      if (maximumWaitTime == 0) {
        return receiveNoWait();
      }

      return receiveWithTimeout(maximumWaitTime);
    }
  }

  public void consume(Long maximumWaitTime, CompletionListener completionListener) {
    // TODO - The Non Blocking mechanism is behaving in a flaky way, rolling back to blocking until totally fix it
    synchronized (consumer) {

      try {
        if (maximumWaitTime == 0) {
          completionListener.onCompletion(consumer.receiveNoWait());
        } else if (maximumWaitTime == -1) {
          completionListener.onCompletion(consumer.receive());
        } else {
          completionListener.onCompletion(receiveWithTimeout(maximumWaitTime));
        }
      } catch (Exception e) {
        completionListener.onException(null, e);
      }
    }
  }

  @Override
  public void close() throws JMSException {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Closing consumer " + consumer);
    }
    consumer.close();
  }

  private Long calculateThreshold(Long timeout) {
    return timeout * DELTA < 1000 ? (long) (timeout * DELTA) : 1000;
  }

  private Message receiveWithTimeout(Long maximumWaitTime) throws JMSException, JmsTimeoutException {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug(format("Waiting for a message, timeout will be in [%s] millis", maximumWaitTime));
    }

    StopWatch timeoutValidator = new StopWatch();
    timeoutValidator.start();
    Message message = consumer.receive(maximumWaitTime);
    timeoutValidator.stop();
    // In Windows it could timeout a little earlier than it should be
    Long threshold = calculateThreshold(maximumWaitTime);

    if (message == null && timeoutValidator.getTime() >= maximumWaitTime - threshold) {
      throw new JmsTimeoutException(format("Failed to retrieve a Message. Operation timed out after %s milliseconds",
                                           maximumWaitTime));
    }
    return message;
  }

  private Message receiveNoWait() throws JMSException {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Trying to consume an immediately available message");
    }

    return consumer.receiveNoWait();
  }

  private Message receive() throws JMSException {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("No Timeout set, waiting for a message until one arrives");
    }

    return consumer.receive();
  }

  public MessageConsumer get() {
    return consumer;
  }

  /**
   * Consumes a message in a async wait. The result of the consumption will be communicated to the given
   * {@link CompletionListener}.
   * 
   * @param maximumWaitTime    Time to wait, in milliseconds, until a timeout is raised.
   * @param scheduler          Scheduler to create async tasks.
   * @param completionListener Result callback that will be called once the message is received or a error is raised.
   */
  public void consume(Long maximumWaitTime, Scheduler scheduler, CompletionListener completionListener) {
    CompletableFuture<Message> completableFuture = new CompletableFuture<>();
    try {
      consumer.setMessageListener(completableFuture::complete);
      scheduler.execute(() -> {
        try {
          Message message;
          if (maximumWaitTime > 0) {
            message = completableFuture.get(maximumWaitTime, MILLISECONDS);
          } else if (maximumWaitTime == 0) {
            message = completableFuture.getNow(null);
          } else {
            message = completableFuture.get();
          }
          completionListener.onCompletion(message);
        } catch (ExecutionException e) {
          completionListener.onException(null, new JmsConsumeException("Unable to listen for message.", e));
        } catch (TimeoutException e) {
          completionListener
              .onException(null,
                           new JmsTimeoutException(format("Failed to retrieve a Message. Operation timed out after %s milliseconds",
                                                          maximumWaitTime)));
        } catch (InterruptedException e) {
          completionListener.onException(null, new JMSException("Unable to listen for message.", e.getMessage()));
        }
      });
    } catch (Exception e) {
      completionListener.onException(null, e);
    }
  }

}
