/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.common;

import org.slf4j.Logger;

import static org.slf4j.LoggerFactory.getLogger;

public final class IBMClientCommons {

  private static final Logger LOGGER = getLogger(IBMClientCommons.class);

  /**
   * We are checking for IBM-MQ connection broken exception and the reason code for the same is 2009 because the customer
   * encountered a condition where it was not reconnecting after getting 2009.
   * <p>
   * Haven't found a specific source to verify the format of the message contained in MQException or JmqiException. But after
   * checking a lot of logs online the following format seems to be prevalent.
   * <p>
   * MQException - "com.ibm.mq.MQException: JMSCMQ0001: IBM MQ call failed with compcode '2' ('MQCC_FAILED') reason '2009'
   * ('MQRC_CONNECTION_BROKEN')."
   * <p>
   * JmqiException - "com.ibm.mq.jmqi.JmqiException: CC=2;RC=2009;AMQ9208: Error on receive from host 'hostname'."
   *
   * @param throwable
   * @return
   */
  public static boolean isConnectionBrokenException(Throwable throwable) {
    if (throwable == null)
      return false;

    for (; throwable != null; throwable = throwable.getCause()) {
      if (throwable.getClass().getName().startsWith("com.ibm") && throwable.getMessage() != null
          && (throwable.getMessage().toUpperCase().contains("REASON '2009'")
              || throwable.getMessage().toUpperCase().contains("RC=2009"))) {
        LOGGER.error("Caught IBM connection broken exception");
        return true;
      }
    }
    return false;
  }
}
