/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.source.push;

import static java.lang.String.format;
import static org.mule.jms.commons.internal.common.JmsCommons.closeQuietly;
import static org.mule.jms.commons.internal.common.JmsCommons.getDestinationType;
import static org.mule.jms.commons.internal.config.InternalAckMode.AUTO;
import static org.mule.jms.commons.internal.config.InternalAckMode.DUPS_OK;
import static org.slf4j.LoggerFactory.getLogger;

import java.util.ArrayList;
import java.util.List;

import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.MessageListener;

import org.mule.jms.commons.api.destination.ConsumerType;
import org.mule.jms.commons.api.exception.JmsExtensionException;
import org.mule.jms.commons.api.lock.JmsListenerLockFactory;
import org.mule.jms.commons.internal.config.InternalAckMode;
import org.mule.jms.commons.internal.config.JmsConfig;
import org.mule.jms.commons.internal.connection.JmsConnection;
import org.mule.jms.commons.internal.connection.session.JmsSession;
import org.mule.jms.commons.internal.consume.JmsMessageConsumer;
import org.mule.jms.commons.internal.source.JmsListenerLock;
import org.mule.jms.commons.internal.source.JmsResourceReleaser;
import org.mule.jms.commons.internal.source.MessageConsumerDelegate;
import org.mule.jms.commons.internal.support.JmsSupport;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.message.Error;
import org.mule.runtime.extension.api.runtime.source.SourceCallbackContext;
import org.slf4j.Logger;

/**
 * {@link MessageConsumerDelegate} implementation which uses {@link MessageListener}
 *
 * @since 1.0
 */
public class JmsMessageListenerDelegate implements MessageConsumerDelegate {

  private static final Logger LOGGER = getLogger(JmsMessageListenerDelegate.class);

  private JmsMessageListenerFactory messageListenerFactory;
  private JmsConnection connection;
  private final JmsSupport jmsSupport;
  private final ConsumerType consumerType;
  private final String destination;
  private final JmsConfig config;
  private final InternalAckMode resolvedAckMode;
  private final String selector;
  private final List<MessageListenerContext> createdListeners = new ArrayList<>();
  private final JmsResourceReleaser resourceCleaner;
  private final JmsListenerLockFactory lockFactory;

  static final String JMS_LOCK_VAR = "JMS_LOCK";

  public JmsMessageListenerDelegate(JmsMessageListenerFactory messageListenerFactory, JmsConnection connection,
                                    JmsSupport jmsSupport, ConsumerType consumerType, String destination, JmsConfig config,
                                    InternalAckMode resolvedAckMode, String selector, JmsListenerLockFactory lockFactory,
                                    JmsResourceReleaser resourceCleaner) {
    this.messageListenerFactory = messageListenerFactory;
    this.connection = connection;
    this.jmsSupport = jmsSupport;
    this.consumerType = consumerType;
    this.destination = destination;
    this.config = config;
    this.resolvedAckMode = resolvedAckMode;
    this.selector = selector;
    this.lockFactory = lockFactory;
    this.resourceCleaner = resourceCleaner;
  }

  private MessageListenerContext createMessageConsumer() throws Exception {
    JmsSession session = connection.createSession(resolvedAckMode, consumerType.topic());
    JmsMessageConsumer consumer;

    try {
      final Destination jmsDestination = jmsSupport.createDestination(session.get(), destination, consumerType.topic(), config);
      consumer = connection.createConsumer(session, jmsDestination, selector, consumerType);
    } catch (JMSException | JmsExtensionException e) {
      LOGGER.error("Error creating JMS consumer  , [%s]", session.get(), e);
      session.close();
      throw e;
    }

    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug(format("Creating Message Listener on Session [%s] for destination [%s]", session.get(), destination));
    }

    JmsListenerLock jmsLock = lockFactory.createLock(resolvedAckMode);
    JmsMessageListener listener = messageListenerFactory.createMessageListener(session, jmsLock);
    consumer.listen(listener);
    return new MessageListenerContext(session, jmsLock, consumer, listener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void createConsumers(int numberOfConsumers) throws ConnectionException {
    try {
      lockFactory.setEnabled(true);
      synchronized (createdListeners) {
        for (int i = 0; i < numberOfConsumers; i++) {
          createdListeners.add(createMessageConsumer());
        }
      }
    } catch (Exception e) {
      String msg = format("An error occurred while creating the consumers for destination [%s:%s]: %s",
                          getDestinationType(consumerType), destination, e.getMessage());
      LOGGER.error(msg, e);
      stop();

      throw new ConnectionException(msg, e, null, connection);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onSuccess(SourceCallbackContext callbackContext) {
    callbackContext.<JmsListenerLock>getVariable(JMS_LOCK_VAR)
        .ifPresent(JmsListenerLock::unlock);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void stop() {
    synchronized (createdListeners) {
      try {
        createdListeners.forEach(this::closeConsumerQuietly);
      } finally {
        createdListeners.clear();
      }
    }
  }

  @Override
  public void disableConsumers() {
    // Clean message listeners
    synchronized (createdListeners) {
      createdListeners.forEach(this::disableConsumerListener);
    }
  }

  protected void disableConsumerListener(MessageListenerContext info) {
    resourceCleaner.releaseConsumerMessageListener(info);
  }

  protected void closeConsumerQuietly(MessageListenerContext info) {
    try {
      closeConsumer(info);
    } catch (Exception e) {
      LOGGER.debug("Failed to close consumer {}", info.getConsumer());
    } finally {
      closeQuietly(info.getSession());
    }
  }


  private void closeConsumer(MessageListenerContext info) {
    try {
      disableConsumerListener(info);
    } finally {
      info.getLock().unlock();
      closeQuietly(info.getConsumer());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onError(SourceCallbackContext callbackContext, Error error) {
    callbackContext.<JmsListenerLock>getVariable(JMS_LOCK_VAR)
        .ifPresent(jmsLock -> {
          if (resolvedAckMode.equals(AUTO) || resolvedAckMode.equals(DUPS_OK)) {
            jmsLock.unlockWithFailure(error);
          } else {
            jmsLock.unlock();
          }
        });
  }
}
