/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.lock;

import static java.lang.String.format;
import static org.mule.jms.commons.internal.config.InternalAckMode.IMMEDIATE;
import static org.mule.jms.commons.internal.config.InternalAckMode.TRANSACTED;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.jms.commons.api.lock.JmsListenerLockFactory;
import org.mule.jms.commons.internal.config.InternalAckMode;
import org.mule.jms.commons.internal.source.DefaultJmsListenerLock;
import org.mule.jms.commons.internal.source.JmsListenerLock;
import org.mule.jms.commons.internal.source.NullJmsListenerLock;

import java.util.concurrent.atomic.AtomicBoolean;

import org.slf4j.Logger;


/**
 * Default implementation of {@link JmsListenerLockFactory}
 *
 * @since 1.2.0
 */
public class DefaultJmsListenerLockFactory implements JmsListenerLockFactory {

  private static final Logger LOGGER = getLogger(DefaultJmsListenerLockFactory.class);

  private AtomicBoolean enabled = new AtomicBoolean(true);

  @Override
  public JmsListenerLock createLock(InternalAckMode resolvedAckMode) {
    checkState(enabled.get(), "Cannot generate a new lock. Listener is stopped");

    if (resolvedAckMode.equals(IMMEDIATE) || resolvedAckMode.equals(TRANSACTED)) {
      if (LOGGER.isTraceEnabled()) {
        LOGGER.trace(format("Session lock skipped for ACK mode [%s].", resolvedAckMode.name()));
      }
      return new NullJmsListenerLock();
    }

    return new DefaultJmsListenerLock();
  }

  @Override
  public void setEnabled(boolean enabled) {
    this.enabled.set(enabled);
  }
}
