/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.api.message;

import org.mule.jms.commons.internal.message.JMSXDefinedPropertiesNames;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.Summary;

import java.io.Serializable;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import javax.jms.Message;

import static java.util.Collections.unmodifiableMap;


/**
 * JMS reserves the 'JMSX' property name prefix for JMS defined properties. Here we define the set of 'well known' properties of
 * JMS.
 *
 * JMSX properties 'set by provider on send' are available to both the producer and the consumers of the message. If they are not
 * present in a message, they are treated like any other absent property.
 *
 *
 * @since 1.0
 */
public class JmsxProperties implements Serializable {

  /**
   * The identity of the user sending the message
   */
  @Parameter
  @Optional
  @Summary("The identity of the user sending the message")
  private String jmsxUserID;

  /**
   * The identity of the application sending the message
   */
  @Parameter
  @Optional
  @Summary("The identity of the application sending the message")
  private String jmsxAppID;

  /**
   * The number of message delivery attempts
   */
  @Parameter
  @Optional
  @Summary("The number of message delivery attempts")
  private Integer jmsxDeliveryCount;

  /**
   * The identity of the message group this message is part of
   */
  @Parameter
  @Optional
  @Summary("The identity of the message group this message is part of")
  private String jmsxGroupID;

  /**
   * The sequence number of this message within the group
   */
  @Parameter
  @Optional
  @Summary("The sequence number of this message within the group")
  private Integer jmsxGroupSeq;

  /**
   * The transaction identifier of the transaction within which this message was produced
   */
  @Parameter
  @Optional
  @Summary("The transaction identifier of the transaction within which this message was produced")
  private String jmsxProducerTXID;

  /**
   * The transaction identifier of the transaction within which this message was consumed
   */
  @Parameter
  @Optional
  @Summary("The transaction identifier of the transaction within which this message was consumed")
  private String jmsxConsumerTXID;

  /**
   * The time JMS delivered the message to the consumer
   */
  @Parameter
  @Optional
  @Summary("The time JMS delivered the message to the consumer")
  private Long jmsxRcvTimestamp;

  public JmsxProperties() {}

  public JmsxProperties(String JMSXUserID, String JMSXAppID, Integer JMSXDeliveryCount, String JMSXGroupID, Integer JMSXGroupSeq,
                        String JMSXProducerTXID, String JMSXConsumerTXID, Long JMSXRcvTimestamp) {
    this.jmsxUserID = JMSXUserID;
    this.jmsxAppID = JMSXAppID;
    this.jmsxDeliveryCount = JMSXDeliveryCount;
    this.jmsxGroupID = JMSXGroupID;
    this.jmsxGroupSeq = JMSXGroupSeq;
    this.jmsxProducerTXID = JMSXProducerTXID;
    this.jmsxConsumerTXID = JMSXConsumerTXID;
    this.jmsxRcvTimestamp = JMSXRcvTimestamp;
  }

  /**
   * @return all the JMSX properties in a plain {@link Map} representation using the corresponding keys as defined in
   *         {@link JMSXDefinedPropertiesNames}
   */
  public Map<String, Object> asMap() {

    Map<String, Object> tmpMap = new HashMap<String, Object>();

    addIfNotNullValue(tmpMap, JMSXDefinedPropertiesNames.JMSXUserID, this.jmsxUserID);
    addIfNotNullValue(tmpMap, JMSXDefinedPropertiesNames.JMSXAppID, this.jmsxAppID);
    addIfNotNullValue(tmpMap, JMSXDefinedPropertiesNames.JMSXDeliveryCount, this.jmsxDeliveryCount);
    addIfNotNullValue(tmpMap, JMSXDefinedPropertiesNames.JMSXGroupID, this.jmsxGroupID);
    addIfNotNullValue(tmpMap, JMSXDefinedPropertiesNames.JMSXGroupSeq, this.jmsxGroupSeq);
    addIfNotNullValue(tmpMap, JMSXDefinedPropertiesNames.JMSXProducerTXID, this.jmsxProducerTXID);
    addIfNotNullValue(tmpMap, JMSXDefinedPropertiesNames.JMSXConsumerTXID, this.jmsxConsumerTXID);
    addIfNotNullValue(tmpMap, JMSXDefinedPropertiesNames.JMSXRcvTimestamp, this.jmsxRcvTimestamp);
    return unmodifiableMap(new LinkedHashMap<String, Object>(tmpMap));

  }

  /**
   * @return the JMSXUserID {@link Message} property
   */
  public String getJmsxUserID() {
    return jmsxUserID;
  }

  /**
   * @return the JMSXAppID {@link Message} property
   */
  public String getJmsxAppID() {
    return jmsxAppID;
  }

  /**
   * @return the JMSXDeliveryCount {@link Message} property
   */
  public Integer getJmsxDeliveryCount() {
    return jmsxDeliveryCount;
  }

  /**
   * @return the JMSXGroupID {@link Message} property
   */
  public String getJmsxGroupID() {
    return jmsxGroupID;
  }

  /**
   * @return the JMSXGroupSeq {@link Message} property
   */
  public Integer getJmsxGroupSeq() {
    return jmsxGroupSeq;
  }

  /**
   * @return the JMSXProducerTXID {@link Message} property
   */
  public String getJmsxProducerTXID() {
    return jmsxProducerTXID;
  }

  /**
   * @return the JMSXConsumerTXID {@link Message} property
   */
  public String getJmsxConsumerTXID() {
    return jmsxConsumerTXID;
  }

  /**
   * @return the JMSXRcvTimestamp
   */
  public Long getJmsxRcvTimestamp() {
    return jmsxRcvTimestamp;
  }

  public void setJmsxUserID(String jmsxUserID) {
    this.jmsxUserID = jmsxUserID;
  }

  public void setJmsxAppID(String jmsxAppID) {
    this.jmsxAppID = jmsxAppID;
  }

  public void setJmsxDeliveryCount(Integer jmsxDeliveryCount) {
    this.jmsxDeliveryCount = jmsxDeliveryCount;
  }

  public void setJmsxGroupID(String jmsxGroupID) {
    this.jmsxGroupID = jmsxGroupID;
  }

  public void setJmsxGroupSeq(Integer jmsxGroupSeq) {
    this.jmsxGroupSeq = jmsxGroupSeq;
  }

  public void setJmsxProducerTXID(String jmsxProducerTXID) {
    this.jmsxProducerTXID = jmsxProducerTXID;
  }

  public void setJmsxConsumerTXID(String jmsxConsumerTXID) {
    this.jmsxConsumerTXID = jmsxConsumerTXID;
  }

  public void setJmsxRcvTimestamp(Long jmsxRcvTimestamp) {
    this.jmsxRcvTimestamp = jmsxRcvTimestamp;
  }

  private void addIfNotNullValue(Map<String, Object> tmpMap, String key, Object value) {
    if (value != null) {
      tmpMap.put(key, value);
    }
  }

}
