/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.util;

import com.google.gson.Gson;
import com.mchange.v2.c3p0.PooledDataSource;
import org.enhydra.jdbc.standard.StandardXADataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.management.MBeanAttributeInfo;
import javax.management.MBeanInfo;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.sql.DataSource;
import java.lang.management.ManagementFactory;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static java.lang.System.getProperty;

/**
 * Utility class for logging information related to the pooling profile being used.
 *
 * @since 1.3.0
 */

public class DbPoolingProfileLoggerUtils {

  private static final Logger LOGGER = LoggerFactory.getLogger(DbPoolingProfileLoggerUtils.class);
  private static final String POOLING_DATA_LOGGER_MBEAN_PATTERN = "mule.db.connector.pooling.logger.pattern";
  private static final String INITIATION_POLLING_PHRASE = "Pooling data being gathered at {} method";

  private DbPoolingProfileLoggerUtils() {
    throw new IllegalStateException("Utility class - Do not instantiate.");
  }

  /**
   * Logs the pooling data from the given DataSource that implements the {@link com.mchange.v2.c3p0.PooledDataSource} interface.
   *
   * @param dataSource DataSource object from where pooling data is going to be gathered.
   * @param caller     the name of the method that logs the pooling data.
   * @exception SQLException when trying to call methods from the {@link com.mchange.v2.c3p0.PooledDataSource} interface.
   *
   */
  public static void getC3P0sPoolingData(DataSource dataSource, String caller) {
    if (LOGGER.isTraceEnabled() && dataSource instanceof PooledDataSource) {
      try {
        LOGGER.trace(INITIATION_POLLING_PHRASE, caller);

        PooledDataSource pooledDataSource = (PooledDataSource) dataSource;
        Map<String, Object> poolingData = new HashMap<>();

        poolingData.put("NumConnections", pooledDataSource.getNumConnectionsDefaultUser());
        poolingData.put("NumBusyConnections: ", pooledDataSource.getNumBusyConnectionsDefaultUser());
        poolingData.put("NumIdleConnections: ", pooledDataSource.getNumIdleConnectionsDefaultUser());
        poolingData.put("NumUnclosedOrphanedConnections", pooledDataSource.getNumUnclosedOrphanedConnectionsDefaultUser());
        poolingData.put("StatementCacheNumStatements", pooledDataSource.getStatementCacheNumStatementsDefaultUser());
        poolingData.put("StatementCacheNumCheckedOut", pooledDataSource.getStatementCacheNumCheckedOutDefaultUser());
        poolingData.put("StatementCacheNumConnectionsWithCached",
                        pooledDataSource.getStatementCacheNumConnectionsWithCachedStatementsDefaultUser());
        poolingData.put("StartTimeMillis", pooledDataSource.getStartTimeMillisDefaultUser());
        poolingData.put("UpTimeMillis", pooledDataSource.getUpTimeMillisDefaultUser());
        poolingData.put("NumFailedCheckins", pooledDataSource.getNumFailedCheckinsDefaultUser());
        poolingData.put("NumFailedIdleTests", pooledDataSource.getNumFailedIdleTestsDefaultUser());
        poolingData.put("EffectivePropertyCycle", pooledDataSource.getEffectivePropertyCycleDefaultUser());
        poolingData.put("NumThreadsAwaitingCheckout", pooledDataSource.getNumThreadsAwaitingCheckoutDefaultUser());

        LOGGER.trace(poolingData.toString());
      } catch (SQLException sqlException) {
        LOGGER.error("An error occurred while gathering pooling data: {}", sqlException.getMessage());
      }
    }
  }

  /**
   * Logs the pooling data from the given DataSource that implements the {@link javax.sql.XADataSource} interface.
   *
   * @param dataSource DataSource object from where pooling data is going to be gathered.
   * @param caller     the name of the method that logs the pooling data.
   * @exception IllegalAccessException    when the requested field can not be accessed.
   * @exception NoSuchMethodException     when the requested method does not exists.
   * @exception InvocationTargetException when the request field does not exists.
   * @exception NoSuchFieldException      when the requested field does not exists.
   *
   */
  // TODO - DBCLI-20: review all possible combinations of configs where this should be used
  public static void getXAPoolData(DataSource dataSource, String caller) {
    if (LOGGER.isTraceEnabled()) {
      boolean fieldChanged = false;
      boolean isAccessible = false;
      Field xaDataSourceField = null;

      try {
        LOGGER.trace(INITIATION_POLLING_PHRASE, caller);

        Method getWrappedDataSourceMethod = dataSource.getClass().getMethod("getWrappedDataSource");
        Object poolingDataSource = getWrappedDataSourceMethod.invoke(dataSource);
        Method toStringMethod = poolingDataSource.getClass().getMethod("toString");
        xaDataSourceField = poolingDataSource.getClass().getDeclaredField("xaDataSource");

        isAccessible = xaDataSourceField.isAccessible();
        xaDataSourceField.setAccessible(true);
        fieldChanged = true;

        StandardXADataSource xa = (StandardXADataSource) xaDataSourceField.get(null);
        xaDataSourceField.setAccessible(isAccessible);

        Map<String, Object> poolingData = new HashMap<>();
        poolingData.put("Bitronix.PoolingDataSource", toStringMethod.invoke(poolingDataSource));
        poolingData.put("AllConnections", xa.getAllConnections());
        poolingData.put("ConnectionCount", xa.getConnectionCount());
        poolingData.put("DeadLockMaxWait", xa.getDeadLockMaxWait());
        poolingData.put("DeadLockRetryWait", xa.getDeadLockRetryWait());
        poolingData.put("MaxCon", xa.getMaxCon());
        poolingData.put("MinCon", xa.getMinCon());
        poolingData.put("LoginTimeout", xa.getLoginTimeout());
        poolingData.put("PreparedStmtCacheSize", xa.getPreparedStmtCacheSize());

        LOGGER.trace(poolingData.toString());
      } catch (IllegalAccessException | NoSuchMethodException | InvocationTargetException | NoSuchFieldException exception) {
        if (fieldChanged) {
          xaDataSourceField.setAccessible(isAccessible);
        }

        LOGGER.error("An error occurred while gathering pooling data: {}", exception.getMessage());
      }
    }
  }

  /**
   * Logs the pooling data from the MBean, registered against the {@link javax.management.MBeanServer}, that follows the pattern
   * determined by POOLING_DATA_LOGGER_MBEAN_PATTERN system property.
   *
   * @param caller the name of the method that logs the pooling data.
   *
   */
  // TODO - DBCLI-20: review all possible combinations of configs where this should be used
  public static void getDataSourcePoolData(String caller) {
    if (LOGGER.isTraceEnabled()) {
      String searchPattern = getMBeanPattern();

      if (searchPattern != null) {
        LOGGER.trace(INITIATION_POLLING_PHRASE, caller);

        MBeanServerConnection mBeanServerConnection = ManagementFactory.getPlatformMBeanServer();
        List<Map<String, String>> poolingData = getConnectionPoolStats(searchPattern, mBeanServerConnection);

        LOGGER.trace(new Gson().toJson(poolingData));
      } else {
        LOGGER.trace("Couldn't gather datasource pooling data, no MBean Search Pattern provided. Use system property "
            + POOLING_DATA_LOGGER_MBEAN_PATTERN);
      }
    }
  }

  private static List<Map<String, String>> getConnectionPoolStats(String searchPattern, MBeanServerConnection mBeanServer) {
    LOGGER.trace("MBean Search pattern {} ", searchPattern);
    List<Map<String, String>> stats = new ArrayList<>();

    try {
      Set<ObjectName> objects = mBeanServer.queryNames(new ObjectName(searchPattern), null);

      if (!objects.isEmpty()) {
        for (ObjectName obj : objects) {
          Map<String, String> infoMap = new HashMap<>();
          MBeanInfo info = mBeanServer.getMBeanInfo(obj);
          MBeanAttributeInfo[] attr = info.getAttributes();

          for (MBeanAttributeInfo mBeanAttributeInfo : attr) {
            String name = mBeanAttributeInfo.getName();
            if (mBeanAttributeInfo.isReadable()) {
              readAttributesValue(mBeanServer, obj, infoMap, name);
            }
          }

          stats.add(infoMap);
        }

        return stats;
      } else {
        LOGGER.trace("MBean Search pattern returned zero results, no pooling data gathered.");
      }
    } catch (Exception e) {
      LOGGER.error("Error while gathering pooling data: {}", e.getMessage());
    }

    return stats;
  }

  private static void readAttributesValue(MBeanServerConnection mBeanServer, ObjectName obj, Map<String, String> infoMap,
                                          String name) {
    try {
      Object value = mBeanServer.getAttribute(obj, name);
      infoMap.put(name, String.valueOf(value));
    } catch (Exception e) {
      LOGGER.error("Error in reading config {}, error {}", name, e);
    }
  }

  private static String getMBeanPattern() {
    return getProperty(POOLING_DATA_LOGGER_MBEAN_PATTERN);
  }

}
