/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.domain.type;

import org.mule.db.commons.api.DbAggregate;
import org.mule.db.commons.internal.domain.connection.DbConnection;

import java.sql.CallableStatement;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Types;
import java.util.Arrays;
import java.util.List;

/**
 * Represents a data type for a template, which real type is unknown until it is instantiated
 */
public class UnknownDbType extends AbstractDbType {

  public static final String UNKNOWN_TYPE_NAME = "UNKNOWN";

  private static final UnknownDbType instance = new UnknownDbType();

  private UnknownDbType() {
    super(Types.OTHER, UNKNOWN_TYPE_NAME);
  }

  @Override
  public void setParameterValue(PreparedStatement statement, int index, Object value, DbConnection connection)
      throws SQLException {
    value = createArraysAndStructs(value, connection);
    statement.setObject(index, value);
  }

  /**
   * Replaces instances of DbAggregate with the right SQL value.
   *
   * @return A newly created SQL array or struct if the value was a {@link DbAggregate}, the original value if not.
   */
  static Object createArraysAndStructs(Object value, DbConnection connection) throws SQLException {
    if (value instanceof DbAggregate) {
      DbAggregate dbArray = (DbAggregate) value;
      Object[] arr = createArraysAndStructsInner(dbArray.getElements(), connection);
      switch (dbArray.getKind()) {
        case ARRAY:
          return connection.createArray(dbArray.getTypeName(), arr);
        case STRUCT:
          return connection.createStruct(dbArray.getTypeName(), arr);
        default:
          throw new RuntimeException("Unknown kind of DbAggregate: " + dbArray.getKind());
      }
    } else if (value instanceof List) {
      // noinspection unchecked
      return createArraysAndStructsInner((List<Object>) value, connection);
    } else if (value instanceof Object[]) {
      return createArraysAndStructsInner(Arrays.asList((Object[]) value), connection);
    }
    return value;
  }

  /**
   * Replaces instances of DbAggregate with the right SQL value in a list.
   *
   * @return an array with the content of the original list but will all {@link DbAggregate} values replaced.
   */
  static Object[] createArraysAndStructsInner(List<?> elements, DbConnection connection) throws SQLException {
    Object[] returnedElements = new Object[elements.size()];
    for (int i = 0; i < elements.size(); i++) {
      returnedElements[i] = createArraysAndStructs(elements.get(i), connection);
    }
    return returnedElements;
  }

  @Override
  public Object getParameterValue(CallableStatement statement, int index) throws SQLException {
    return statement.getObject(index);
  }

  public static DbType getInstance() {
    return instance;
  }
}
