/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.domain.executor;

import org.mule.db.commons.api.StatementResult;
import org.mule.db.commons.internal.domain.autogeneratedkey.AutoGenerateKeysStrategy;
import org.mule.db.commons.internal.domain.autogeneratedkey.NoAutoGenerateKeysStrategy;
import org.mule.db.commons.internal.domain.connection.DbConnection;
import org.mule.db.commons.internal.domain.query.Query;
import org.mule.db.commons.internal.domain.statement.StatementFactory;
import org.mule.db.commons.internal.result.resultset.ListResultSetHandler;
import org.mule.db.commons.internal.result.row.InsensitiveMapRowHandler;

import java.math.BigInteger;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;
import java.util.Map;

/**
 * Executes queries that return an update count or an update count and auto generated keys
 */
public class UpdateExecutor extends AbstractSingleQueryExecutor {

  public UpdateExecutor(StatementFactory statementFactory) {
    super(statementFactory);
  }

  @Override
  protected Object doExecuteQuery(DbConnection connection, Statement statement, Query query) throws SQLException {
    return doExecuteQuery(connection, statement, query, new NoAutoGenerateKeysStrategy());
  }

  @Override
  protected Object doExecuteQuery(DbConnection dbConnection, Statement statement, Query query,
                                  AutoGenerateKeysStrategy autoGenerateKeysStrategy)
      throws SQLException {
    try {
      int updateCount = autoGenerateKeysStrategy.executeUpdate(statement, query.getQueryTemplate());
      StatementResult result;

      if (autoGenerateKeysStrategy.returnsAutoGenerateKeys()) {
        ResultSet generatedKeys = statement.getGeneratedKeys();
        ListResultSetHandler listResultSetHandler = new ListResultSetHandler(new InsensitiveMapRowHandler(dbConnection));

        List generatedKeyList = listResultSetHandler.processResultSet(dbConnection, generatedKeys);
        Map<String, BigInteger> keyMap = generatedKeyList.isEmpty() ? null : (Map<String, BigInteger>) generatedKeyList.get(0);
        result = new StatementResult(updateCount, keyMap);
      } else {
        result = new StatementResult(updateCount, null);
      }

      return result;
    } finally {
      statement.close();
    }
  }

}
